/*
 *	cache.cc
 *
 *	Main program for the FOS cache server.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 SUSCEVAZ
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <c++/ntrel.h>
#include <c++/mon.h>

unsigned long _stksize   = 10*1024;					// pile
unsigned char _SMAPRIO   = 8;						// priorit leve
unsigned char _SMAREV    = 1;						// rvision
unsigned char _SMAVER    = 0;						// version
unsigned long _SMAUNIT   = 0x00000080;				// utilise binaire GNU (no cache)
unsigned short _SMADIS[] = {20, 600, 20, 400};		// petite fentre

static Card32 max_cache_size = 512*4;
static Card32 max_chunk_size = 32*4;
static Card32 floppy_factor  = 4;
static Bool   do_lazy_floppy = FALSE;
static Bool   do_lazy_scsi   = FALSE;

// For debugging, set following variables to FALSE :

static Bool do_volatile_floppy = TRUE;
static Bool ok_scsi = TRUE;

#include "cache.h"
#include "cachedata.h"

static NtrBar* g_cache_bar = 0;


/*
 *	Send a command to the cache engine. This simply posts a message
 *	to the BAR and waits for the cache engine to process it.
 */

static void
cache_send (Card16 op)
{
	CacheRec rec = { 0 };
	void*    ptr = & rec;
	rec.op = op;
	NtrBar* bar = new NtrBar ("#:FosCacheBlk", 1);
	if ( (bar == 0) || (bar->GetError () != 0) ) return;
	bar->Offer ();
	bar->EndOffer (ptr,ptr);
	
	//	No delete here : we don't want to kill the bar !
}


/*
 *	Force the cache to synchronise its contents; this will flush
 *	any buffers still in use.
 */

static void
cache_sync_all (CacheData* cache, void* ptr)
{
	cache->Sync ();
}


/*
 *	Invalidate the cache's contents. This will brutally forget about
 *	any data in the cache -- even dirty data will not be written back.
 */

static void
cache_inv_all (CacheData* cache, void* ptr)
{
	cache->Sync ();
	cache->Invalidate ();
}


/*
 *	BIOS: Read blocks.
 */

Card16
bios_dread (Card32 id, Card32 pos, Card32 len, void* ptr)
{
	Card16 error = 0;
	
	asm volatile ( "movel %1,d2\n\t"
				   "movel %2,d4\n\t"
				   "movel %3,d5\n\t"
				   "movel %4,a4\n\t"
				   ".long 0x4E440500\n\t"
				   "movew d7,%0"
				 : "=g" (error)
				 : "g" (id), "g" (pos), "g" (len), "g" (ptr)
				 : "d2", "d4", "d5", "d7", "a4" );
	
	return error;
}

/*
 *	BIOS: Write blocks.
 */

Card16
bios_dwrite (Card32 id, Card32 pos, Card32 len, const void* ptr)
{
	Card16 error = 0;
	
	asm volatile ( "movel %1,d2\n\t"
				   "movel %2,d4\n\t"
				   "movel %3,d5\n\t"
				   "movel %4,a4\n\t"
				   ".long 0x4E440600\n\t"
				   "movew d7,%0"
				 : "=g" (error)
				 : "g" (id), "g" (pos), "g" (len), "g" (ptr)
				 : "d2", "d4", "d5", "d7", "a4" );
	
	return error;
}


/*
 *	Create the BAR used by the cache engine to handle the
 *	requests.
 */

static Bool
create_cache_bar ()
{
	NtrBar* bar = new NtrBar ("#:FosCacheBlk");
	if (bar->GetError ()) return FALSE;
	g_cache_bar = bar;
	return TRUE;
}

static void
kill_cache_bar ()
{
	delete g_cache_bar;
	g_cache_bar = 0;
}


/*
 *	Wait for requests on the cache engine BAR and dispatch
 *	them.
 */

void
handle_cache_bar ()
{
	Card32 open = 0;
	
	for (;;) {
		
		Bool      stop = FALSE;
		CacheRec* arg  = 0;
		
		Ntr::SetTim (5);
		CacheData::PurgeAgedVolatile ();
		
		if (g_cache_bar->Accept (arg)) continue;
		
		if (arg->op == caOPClose) {
			stop = (Bool) getenv ("#CACHE_END");
		}
		
//		printf ("[%d ", arg->op);
		
		switch (arg->op) {
			
			case caOPOpen:
				if ( ( ( (arg->unit_name[0] == '$')
				      && (arg->unit_name[1] == 'S')
				      && (arg->unit_name[2] == 'C')
				      && (arg->unit_name[3] == 'S')
				      && (arg->unit_name[4] == 'I')
				      && (ok_scsi) )

				    || ( (arg->unit_name[0] == '$')
				      && (arg->unit_name[1] == 'D')
				      && (arg->unit_name[2] == 'I')
				      && (arg->unit_name[3] == 'S')
				      && (arg->unit_name[4] == 'K') )
				    
				    || ( (arg->unit_name[0] == '$')
				      && (arg->unit_name[1] == 'F')
				      && (arg->unit_name[2] == 'L')
				      && (arg->unit_name[3] == 'O') ) )
				  && (arg->bios_no != 0)
				  && (stop == FALSE) ) {
					arg->bar_id    = (Card32) g_cache_bar->GetBar ();
					arg->typ_error = 0;
					arg->cache     = new CacheData ();
					
					Bool is_floppy = (arg->unit_name[1] == 'F');
					int  factor    = (is_floppy) ? floppy_factor : 1;
					
					arg->cache->SetName (arg->unit_name);
					arg->cache->SetBiosNum (arg->bios_no);
					arg->cache->SetMaxChunkSize (max_chunk_size);
					arg->cache->SetMaxTotalSize (max_cache_size / factor);
					
					if (is_floppy) {
						arg->cache->SetVolatile (do_volatile_floppy);
						arg->cache->SetLazyWrite (do_lazy_floppy);
					} else {
						arg->cache->SetVolatile (do_lazy_scsi);
						arg->cache->SetLazyWrite (do_lazy_scsi);
					}
					
//					AfText ("Opened "); AfText (arg->unit_name); AfText (" => "); AfX8 (arg->cache); AfCR ();
					
					open++;
				} else {
					arg = 0;
				}
				break;
			
			case caOPClose:
//				AfText ("Closing "); AfX8 (arg->cache); AfCR ();
				
				delete arg->cache;
				arg->cache     = 0;
				arg->typ_error = 0;
				open--;
				break;
			
			case caOPRead:
//				AfText ("[R"); AfX8 (arg->arg_1); AfText ("]");
				arg->typ_error = arg->cache->ReadRange (arg->arg_1, arg->arg_2, arg->p_data);
				break;
			
			case caOPWrite:
//				AfText ("[W"); AfX8 (arg->arg_1); AfText ("]");
				arg->typ_error = arg->cache->WriteRange (arg->arg_1, arg->arg_2, arg->p_data);
				break;
			
			case caOPSyncAll:
				CacheData::ForAll (cache_sync_all, 0);
				arg->typ_error = 0;
				break;
			
			case caOPInvAll:
				CacheData::ForAll (cache_inv_all, 0);
				arg->typ_error = 0;
				break;
			
			case caOPSync:
				arg->cache->Sync ();
				arg->typ_error = 0;
				break;
			
			case caOPInvalidate:
				arg->cache->Sync ();		//	guarantee that the cache is clean !
				arg->cache->Invalidate ();
				arg->typ_error = 0;
				break;
			
			case caOPAbout:
				arg->cache = CacheData::FindNth (arg->arg_1);
				
				if ( (arg->cache)
				  && (arg->arg_2 >= sizeof (CaAboutRec)) ) {
					CaAboutRec* rec = (CaAboutRec*)(arg->p_data);
					
					strcpy (rec->name, arg->cache->GetName ());
					
					rec->rec_size       = sizeof (CaAboutRec);
					rec->max_total_size = arg->cache->GetMaxTotalSize ();
					rec->max_chunk_size = arg->cache->GetMaxChunkSize ();
					rec->current_size   = arg->cache->GetCurrentSize ();
					rec->is_volatile    = arg->cache->IsVolatile ();
					rec->is_dirty       = arg->cache->IsDirty ();
					rec->use_lazy_write = arg->cache->UseLazyWrite ();
					
					arg->arg_1++;
					arg->arg_2     = sizeof (CaAboutRec);
					arg->typ_error = 0;
				} else {
					arg->typ_error = 1;
				}
				
				break;
			
			case caOPConfig:
				
				//	By default, return an error...
				
				arg->typ_error = 1;
				
				if (arg->cache) {
					CacheData*  cache = CacheData::FindNth (arg->arg_1);
					CaAboutRec* rec   = (CaAboutRec*)(arg->p_data);
					
					if (cache == arg->cache) {
						
						arg->cache->Sync ();
						arg->cache->Invalidate ();
						
						arg->cache->SetMaxTotalSize (rec->max_total_size);
						arg->cache->SetMaxChunkSize (rec->max_chunk_size);
						arg->cache->SetVolatile (rec->is_volatile);
						arg->cache->SetLazyWrite (rec->use_lazy_write);
						
						arg->typ_error = 0;
					}
				}
				
				break;
			
			default:
				arg->typ_error = 0;
				break;
		}
		
//		printf (" er=%04x]\n", arg->typ_error);
		g_cache_bar->EndAccept (arg);
		
		if ( (stop) && (open == 0) ) {
			break;
		}
	}
}


int
main (int argc, const char* argv[])
{
	Bool ok_arg;
	
	argc--;
	argv++;
	
	while (argc) {
		
		if (strncmp (argv[0], "-max=", 5) == 0) {
			max_cache_size = atoi (argv[0]+5) * 4;
			if (max_cache_size == 0) return 1;
			ok_arg = TRUE;
		} else if (strncmp (argv[0], "-chunk=", 7) == 0) {
			max_chunk_size = atoi (argv[0]+7) * 4;
			if (max_chunk_size == 0) return 1;
			ok_arg = TRUE;
		} else if (strncmp (argv[0], "-floppy=", 8) == 0) {
			floppy_factor  = atoi (argv[0]+8);
			if (floppy_factor < 1) return 1;
			if (floppy_factor > 16) return 1;
			ok_arg = TRUE;
		} else if (strcmp (argv[0], "-lazy-floppy") == 0) {
			do_lazy_floppy = TRUE;
			ok_arg = TRUE;
		} else if (strcmp (argv[0], "-lazy-scsi") == 0) {
			do_lazy_scsi = TRUE;
			ok_arg = TRUE;
		} else if (strcmp (argv[0], "stop") == 0) {
			putenv ("CACHE_END=YES");
			return 0;
		} else if (strcmp (argv[0], "sync") == 0) {
			cache_send (caOPSyncAll);
			return 0;
		} else if (strcmp (argv[0], "purge") == 0) {
			cache_send (caOPInvAll);
			return 0;
		}
		
		argc--;
		argv++;
	}
	
	if (create_cache_bar ()) {
		handle_cache_bar ();
		putenv ("CACHE_END");
		kill_cache_bar ();
	}
	
	return 0;
}


