/*
 *	cachedata.cc
 *
 *	Caching methods for the FOS units.
 *
 *	(C) Copyright 1996, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */


#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <c++/portable.h>
#include <c++/mon.h>

#include "cachedata.h"

#define	VOLAGE	40		//	approx. 4 seconds inactivity

CacheData* CacheData::c_head = 0;
CacheData* CacheData::c_tail = 0;


CacheData::CacheData ()
{
	this->bios_num = 0;
	this->max_chunk_size = 32*4;
	this->max_total_size = 1024*4;
	this->chunk_num = 0;
	this->total_size = 0;
	this->head = 0;
	this->tail = 0;
	
	this->ok_write      = FALSE;
	this->do_lazy_write = FALSE;
	this->dirty_cache   = FALSE;
	this->dirty_count   = 0;
	
	this->volatile_data = FALSE;
	this->volatile_age  = VOLAGE;

	this->name[0] = 0;
	this->next = this->c_head;
	this->prev = 0;
	
	this->c_head = this;
	(this->next ? this->next->prev : this->c_tail) = this;
}

CacheData::~CacheData ()
{
	this->Sync ();
	this->Invalidate ();
	
	(this->next ? this->next->prev : this->c_tail) = this->prev;
	(this->prev ? this->prev->next : this->c_head) = this->next;
}

void
CacheData::SetName (const char* name)
{
	strcpy (this->name, name);
}


/*
 *	Synchronise the contents of the disk with those of the
 *	cache. This writes the dirty data back to the disk, by
 *	starting at the lowest block number. Writing data in a
 *	linear order dramatically increases the speed of the
 *	operation !
 */

void
CacheData::Sync ()
{
	while (this->dirty_cache) {
		Chunk* iter  = this->head;
		Chunk* best  = 0;
		Card32 first = 0xFFFFFFFF;
		
		while (iter) {
			if (iter->is_dirty) {
				if (iter->pos_begin < first) {
					best  = iter;
					first = best->pos_begin;
				}
			}
			iter = iter->next;
		}
		
		if (best == 0) break;
		
		this->WriteChunk (best);
	}
}


/*
 *	Invalidate the contents of the cache. This should be done
 *	after the cache has been synchronised.
 */

void
CacheData::Invalidate ()
{
	while (this->head) {
		this->RemoveChunk (this->head);
	}
}


/*
 *	Find the first range of blocks which fit in the described
 *	set of blocks.
 */

Bool
CacheData::FindRange (Card32 pos, Card32 len, Card32& hit_pos, Card32& hit_len, void*& hit_ptr)
{
	Chunk* iter = this->head;
	
	Card32 begin = pos;
	Card32 end   = pos+len;
	Card32 dist  = 0xFFFFFFFF;
	
	while (iter) {
		if ( (begin < iter->pos_end)
		  && (iter->pos_begin < end) ) {
			
			Bool fit = FALSE;
			
			if (begin < iter->pos_begin) {
				Card32 d = iter->pos_begin - begin;
				if (d < dist) {
					dist = d;
					fit  = TRUE;
				}
			} else {
				dist = 0;
				fit  = TRUE;
			}
			
			if (fit) {
				hit_pos = iter->pos_begin;
				hit_len = iter->pos_end - iter->pos_begin;
				hit_ptr = iter->data;
			}
			
			if (dist == 0) {
				return TRUE;
			}
		}
		
		iter = iter->next;
	}
	
	if (dist != 0xFFFFFFFF) {
		return TRUE;
	}
	
	return FALSE;
}


/*
 *	Write a dirty chunk from the cache to the disk. This marks
 *	the block as "clean".
 */

void
CacheData::WriteChunk (Chunk* iter)
{
	Card32 pos = iter->pos_begin;
	Card32 len = iter->pos_end - iter->pos_begin;
	
	if (iter->is_dirty) {
		Card16 error = bios_dwrite (this->bios_num, pos, len, iter->data);
		if (error) {
			AfText ("Cache: ");
			AfText (this->name);
			AfText (" error on lazy write is ");
			AfX4 (error);
			AfText (", data loss !\r");
		}
		
		iter->is_dirty = FALSE;
		this->dirty_count--;
		
		if (this->dirty_count == 0) {
			this->dirty_cache = FALSE;
		}
	}
}


/*
 *	Free a chunk from the cache. If it is dirty, first write it
 *	back to the disk.
 */

void
CacheData::RemoveChunk (Chunk* iter)
{
	if (iter) {
		
		(iter->next ? iter->next->prev : this->tail) = iter->prev;
		(iter->prev ? iter->prev->next : this->head) = iter->next;

		this->chunk_num--;
		this->total_size -= iter->pos_end - iter->pos_begin;
		
		if (iter->is_dirty) this->WriteChunk (iter);
		
		delete iter;
	}
}


/*
 *	Add a chunk to the cache. The chunk must not yet exist in the
 *	cache, or else the behaviour will be unpredictable.
 */

CacheData::Chunk*
CacheData::AddNewChunk (Card32 pos, Card32 len, const void* data, Bool dirty)
{
	{
		Chunk* i = this->head;
		Card32 e = pos + len;
		while (i) {
			if ( (i->pos_begin < e)
			  && (i->pos_end > pos) ) {
				AfText ("CACHE: conflit de bloc @ ");
				AfX8 (pos);
				AfText ("L");
				AfX8 (len);
				AfCR ();
			}
			i = i->next;
		}
	}
	
	Chunk* iter = (Chunk*) new char[sizeof (Chunk)+len*256];
	
	iter->pos_begin = pos;
	iter->pos_end   = pos+len;
	iter->is_dirty  = dirty;
	
	if (dirty) {
		this->dirty_count++;
	}
	
	memcpy (iter->data, data, len*256);
	
	iter->next = this->head;
	iter->prev = 0;
	
	(iter->next ? iter->next->prev : this->tail) = iter;
	(iter->prev ? iter->prev->next : this->head) = iter;
	
	this->chunk_num++;
	this->total_size += len;
	
	return iter;
}


/*
 *	Try to merge the current chunk with other ones (before
 *	and after it). If the chunk is merged, it will be replaced
 *	by another one.
 */

void
CacheData::MergeChunk (Chunk* chunk)
{
again:
	Chunk* iter = this->head;
	
	while (iter) {
		if (iter->pos_end == chunk->pos_begin) {
			if (this->MergeTwoChunks (iter, chunk)) {
				chunk = iter;
				goto again;
			}
			return;
		}
		if (chunk->pos_end == iter->pos_begin) {
			if (this->MergeTwoChunks (chunk, iter)) {
				goto again;
			}
			return;
		}
		iter = iter->next;
	}
}


/*
 *	Merge two chunks and return the result.
 */

Bool
CacheData::MergeTwoChunks (Chunk*& a, Chunk* b)
{
	Card32 a_len = a->pos_end - a->pos_begin;
	Card32 b_len = b->pos_end - b->pos_begin;
	Card32 len   = a_len + b_len;
	
	Chunk* iter = (Chunk*) new char[sizeof (Chunk)+len*256];
	if (iter == 0) return FALSE;
	
	memcpy (iter->data, a->data, a_len*256);
	memcpy (iter->data + a_len*256, b->data, b_len*256);
	
	//	Remove the second chunk from the list. It will be freed.
	
	(b->next ? b->next->prev : this->tail) = b->prev;
	(b->prev ? b->prev->next : this->head) = b->next;
	
	//	Replace the first chunk by the grown one.
	
	iter->prev      = a->prev;
	iter->next      = a->next;
	iter->is_dirty  = a->is_dirty | b->is_dirty;
	iter->pos_begin = a->pos_begin;
	iter->pos_end   = b->pos_end;
	
	if (a->is_dirty && b->is_dirty) {
		this->dirty_count--;
	}
	
	(iter->next ? iter->next->prev : this->tail) = iter;
	(iter->prev ? iter->prev->next : this->head) = iter;
	
	this->chunk_num--;
	
	delete a;
	delete b;
	
	a = iter;
	return TRUE;
}


/*
 *	Change the status of a chunk and place it at the head of
 *	the list (this means it has been accessed recently).
 */

void
CacheData::UpdateChunk (Card32 pos, Bool make_dirty)
{
	Chunk* iter = this->head;
	
	while (iter) {
		if (iter->pos_begin == pos) {
			(iter->next ? iter->next->prev : this->tail) = iter->prev;
			(iter->prev ? iter->prev->next : this->head) = iter->next;
			iter->next = this->head;
			iter->prev = 0;
			(iter->next ? iter->next->prev : this->tail) = iter;
			(iter->prev ? iter->prev->next : this->head) = iter;
			
			if ( (iter->is_dirty == FALSE) 
			  && (make_dirty) ) {
				this->dirty_count++;
				iter->is_dirty = TRUE;
			}
			
			return;
		}
		
		iter = iter->next;
	}
}


Card16
CacheData::ReadBlocks (Card32 pos, Card32 len, void* data, Chunk*& new_chunk)
{
	new_chunk = 0;
	Card16 error = bios_dread (this->bios_num, pos, len, data);
	if (error) return error;
	
	//	Don't store any large data into the cache. This would spill
	//	lots of memory.
	
	if (len <= this->max_chunk_size) {
		
		while (this->total_size+len > this->max_total_size) {
			if (this->chunk_num < 4) break;
			this->RemoveChunk (this->tail);
		}
		
		new_chunk = this->AddNewChunk (pos, len, data);
	}
	
	return 0;
}


Card16
CacheData::WriteBlocks (Card32 pos, Card32 len, const void* data, Chunk*& new_chunk)
{
	new_chunk = 0;
	
	//	Don't store any large data into the cache. This would spill
	//	lots of memory.
	
	if (len <= this->max_chunk_size) {
		
		while (this->total_size+len > this->max_total_size) {
			if (this->chunk_num < 4) break;
			this->RemoveChunk (this->tail);
		}
		
		new_chunk = this->AddNewChunk (pos, len, data, TRUE);
		
	} else {
		Card16 error = bios_dwrite (this->bios_num, pos, len, data);
		if (error) return error;
	}
	
	return 0;
}


Card16
CacheData::ReadRange (Card32 pos, Card32 len, void* data)
{
	Card16 error   = 0;
	Card32 hit_pos = 0;
	Card32 hit_len = 0;
	void*  hit_ptr = 0;
	
	this->volatile_age  = VOLAGE;
	
	while (len) {
		
		Bool   hit  = this->FindRange (pos, len, hit_pos, hit_len, hit_ptr);
		Card32 lead = (hit) ? ( (pos < hit_pos) ? hit_pos - pos : 0 ) : len;
		Chunk* new_chunk = 0;
		
		//	Need to read any leading blocks (because they are not in the
		//	cache yet) ?
		
		if (lead) {
			error = this->ReadBlocks (pos, lead, data, new_chunk);
			pos  += lead;
			len  -= lead;
			data += lead*256;
			if (error) return error;
		}
		
		if (hit) {
			
			this->UpdateChunk (hit_pos);
			
			//	Read blocks from within the cache...
			
			Card32 offset = pos - hit_pos;
			Card32 end    = ((pos+len) > (hit_pos+hit_len)) ? (hit_pos+hit_len) : (pos+len);
			Card32 copy   = end - pos;
			
			hit_ptr += offset*256;
			memcpy (data, hit_ptr, copy*256);
			
			data += copy*256;
			pos  += copy;
			len  -= copy;
		}
		
		//	If the new chunk is contiguous to other chunks, merge them... This
		//	will both speed up the operations and simplify the searches.
	
		if (new_chunk) this->MergeChunk (new_chunk);
	}
	
	return 0;
}

Card16
CacheData::WriteRange (Card32 pos, Card32 len, const void* data)
{
	Card16 error     = 0;
	Card16 ret_error = 0;
	
	this->volatile_age  = VOLAGE;
	
	if ( (this->do_lazy_write)
	  && (this->ok_write) ) {
		
		//	If we are using lazy write, we do not write the data directly
		//	to the disk, but just copy it to the cache. It will be written
		//	on the final synchronisation or when the cache gets full.
		
		//	We don't do any lazy stuff now unless there has been at least
		//	one successful write to the disk (this avoids caching data for
		//	a write-protected media).
		
		this->dirty_cache = TRUE;
		
	} else {
		error = bios_dwrite (this->bios_num, pos, len, data);
		if (error) return error;
		this->ok_write = TRUE;
	}
	
	Card32 hit_pos = 0;
	Card32 hit_len = 0;
	void*  hit_ptr = 0;
	
	while (len) {
		
		Bool   hit  = this->FindRange (pos, len, hit_pos, hit_len, hit_ptr);
		Card32 lead = (hit) ? ( (pos < hit_pos) ? hit_pos - pos : 0 ) : len;
		Chunk* new_chunk = 0;
		
		if (lead) {
			
			if (this->do_lazy_write) {
				error = this->WriteBlocks (pos, lead, data, new_chunk);
				if (error) ret_error = error;
			}
			
			pos  += lead;
			len  -= lead;
			data += lead*256;
		}
		
		if (hit) {
			
			this->UpdateChunk (hit_pos, this->do_lazy_write);
			
			//	Write blocks to update the cache...
			
			Card32 offset = pos - hit_pos;
			Card32 end    = ((pos+len) > (hit_pos+hit_len)) ? (hit_pos+hit_len) : (pos+len);
			Card32 copy   = end - pos;
			
			hit_ptr += offset*256;
			memcpy (hit_ptr, data, copy*256);
			
			data += copy*256;
			pos  += copy;
			len  -= copy;
		}
		
		//	If the new chunk is contiguous to other chunks, merge them... This
		//	will both speed up the operations and simplify the searches.
	
		if (new_chunk) this->MergeChunk (new_chunk);
	}
	
	return ret_error;
}

void
CacheData::ForAll (void (*func)(CacheData*, void*), void* arg)
{
	CacheData* ptr = CacheData::c_head;
	
	while (ptr) {
		(func) (ptr, arg);
		ptr = ptr->next;
	}
}

void
CacheData::PurgeAgedVolatile ()
{
	CacheData* ptr = CacheData::c_head;
	
	while (ptr) {
		if ( (ptr->volatile_data)
		  && (ptr->volatile_age) ) {
			if (--ptr->volatile_age == 0) {
				ptr->Sync ();
				ptr->Invalidate ();
			}
		}
		ptr = ptr->next;
	}
}

CacheData*
CacheData::FindNth (Card32 i)
{
	CacheData* ptr = CacheData::c_head;
	
	while (ptr && i--) {
		ptr = ptr->next;
	}
	
	return ptr;
}

