/*
 *	cachedata.h
 *
 *	Caching methods for the FOS units.
 *
 *	(C) Copyright 1996, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */


#include <c++/portable.h>

extern Card16 bios_dread (Card32 id, Card32 pos, Card32 len, void* ptr);
extern Card16 bios_dwrite (Card32 id, Card32 pos, Card32 len, const void* ptr);


/*
 *	The CacheData class provides the interface needed to handle
 *	caching on a FOS unit.
 */

class CacheData
{
	static CacheData* c_head;
	static CacheData* c_tail;
	
	struct Chunk
	{
		Chunk*		next;
		Chunk*		prev;
		Bool		is_dirty;
		Card32		pos_begin;
		Card32		pos_end;
		char		data[0];
	};
	
protected:
	CacheData*		next;
	CacheData*		prev;
	
	char			name[16];
	
	Card16			bios_num;
	Card32			max_chunk_size;
	Card32			max_total_size;
	Card32			chunk_num;
	Card32			total_size;
	
	Bool			do_lazy_write;
	Bool			ok_write;
	Bool			dirty_cache;
	Bool			volatile_data;
	int				volatile_age;
	int				dirty_count;
	
	Chunk*			head;
	Chunk*			tail;
	
protected:
	Bool FindRange (Card32 pos, Card32 len, Card32& hit_pos, Card32& hit_len, void*& hit_ptr);
	
	Card16 ReadBlocks (Card32 pos, Card32 len, void* data, Chunk*& new_chunk);
	Card16 WriteBlocks (Card32 pos, Card32 len, const void* data, Chunk*& new_chunk);
	
	void RemoveChunk (Chunk*);
	Chunk* AddNewChunk (Card32 pos, Card32 len, const void* data, Bool dirty = FALSE);
	void UpdateChunk (Card32 pos, Bool make_dirty = FALSE);
	void WriteChunk (Chunk*);
	void MergeChunk (Chunk*);
	Bool MergeTwoChunks (Chunk*& a, Chunk* b);
	
public:
	CacheData ();
	~ CacheData ();
	
	void Invalidate ();
	void Sync ();
	
	void SetName (const char* name);
	void SetBiosNum (Card16 bios_num)		{ this->bios_num = bios_num; }
	void SetMaxChunkSize (Card32 size)		{ this->max_chunk_size = size; }
	void SetMaxTotalSize (Card32 size)		{ this->max_total_size = size; }
	void SetVolatile (Bool v)				{ this->volatile_data = v; }
	void SetLazyWrite (Bool l)				{ this->do_lazy_write = l; }
	
	Card16 ReadRange (Card32 pos, Card32 len, void* data);
	Card16 WriteRange (Card32 pos, Card32 len, const void* data);
	
	Card32 GetMaxChunkSize () const			{ return this->max_chunk_size; }
	Card32 GetMaxTotalSize () const			{ return this->max_total_size; }
	Card32 GetCurrentSize () const			{ return this->total_size; }
	Bool   IsVolatile () const				{ return this->volatile_data; }
	Bool   IsDirty () const					{ return this->dirty_cache; }
	Bool   UseLazyWrite () const			{ return this->do_lazy_write; }
	const char* GetName () const			{ return this->name; }
	
	static void ForAll (void (*)(CacheData*, void*), void*);
	static void PurgeAgedVolatile ();
	static CacheData* FindNth (Card32 i);
};


