/*
 *	menu.cc
 *
 *	Menu related structures, used by the LIB mouse menu extender, which is
 *	being provided since system 10.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 Suscvaz
 */

extern "C" long L_getcar ();

#include "menu.h"
#include "util.h"

#define	RtypMMenus	0x00000020
#define	RtypRaster	0x00000080

#define	MARGIN		4

Menu::Menu ()
{
	this->res_chan = 0;
	this->res_key  = 0;
	this->res_mode = 0;
	this->dis_chan = 0;
	this->dis_desc = 0;
	this->baseline = 0;

	this->org_x    = 0;
	this->org_y    = 0;
	this->org_dx   = 0;
	this->org_dy   = 0;
	
	this->wdo_dx   = 0;
	this->wdo_dy   = 0;
}

Menu::~Menu ()
{
}

void
Menu::LibInteract (LibMenuRecord* record)
{
	this->param    = record->param;
	this->res_chan = record->res_chan;
	this->res_key  = record->res_key;
	this->res_mode = (record->flags & 0x20) ? 0x29 : 0x09;
	this->dis_chan = record->dis_chan;
	this->dis_desc = record->dis_desc;
	this->baseline = record->baseline;
	this->rank     = 0;
	this->wdo_dx   = ((Card16*)(this->dis_desc+0x0E))[0];
	this->wdo_dy   = ((Card16*)(this->dis_desc+0x0C))[0];
	this->org_x    = ((Card16*)(this->dis_desc+0x28))[0];
	this->org_y    = ((Card16*)(this->dis_desc+0x26))[0];
	this->org_dx   = ((Card16*)(this->dis_desc+0x2C))[0];
	this->org_dy   = ((Card16*)(this->dis_desc+0x2A))[0];
	
	MenuBody* body = this->BuildBody (record->mmenu);
	
	this->MenuPos (body, record->x, record->y);
	this->MenuOpen (body, 0);
	this->MenuOpen (body, 1);
	
	L_getcar ();
	
	this->MenuClose (body, 1);
	this->MenuClose (body, 0);
	
	make_window (this, this->org_x, this->org_y, this->org_dx, this->org_dy);
	
	delete body;
}



void
Menu::MenuPos (MenuBody* menu, Card16 x, Card16 y)
{
	MenuCell* cell = menu->cell;
	Card32    num  = 0;
	Bool      left = FALSE;
	
	Int16 dx = menu->cell_width;
	Int16 dy = menu->cell_ascender + menu->cell_descender;
	Int16 ox = x - dx / 2 - MARGIN; if (ox < 0) ox = 0;
	Int16 oy = y - dy / 2 - MARGIN; if (oy < 0) oy = 0;
	
	menu->rank = this->rank++;
	menu->x    = ox;
	menu->y    = oy;
	menu->dx   = dx + 2*MARGIN;
	menu->dy   = (dy+3)*menu->cell_count + 2*MARGIN - 1;
	
	if ((menu->x+menu->dx) > this->wdo_dx) menu->x = this->wdo_dx - menu->dx;
	if ((menu->y+menu->dy) > this->wdo_dy) menu->y = this->wdo_dy - menu->dy;
	
	//	Find out which of the submenus is the widest one. This will be useful
	//	to decide whether to put the submenus on the left or on the right of
	//	the current menu.
	
	Card16 max_submenu_dx = 0;
	
	while (cell) {
		if ( (cell->submenu)
		  && (cell->submenu->cell_width > max_submenu_dx) ) {
			max_submenu_dx = cell->submenu->cell_width;
		}
		cell = cell->next;
	}
	
	left = ((max_submenu_dx+menu->x+menu->dx) > this->wdo_dx);
	cell = menu->cell;
	
	while (cell) {
		if (param[menu->rank].inactive & (1 << num++)) {
			cell->is_dimmed = TRUE;
		}
		
		if (cell->submenu) {
			if (left) {
				this->MenuPos (cell->submenu, x - cell->submenu->cell_width - MARGIN, y);
			} else {
				this->MenuPos (cell->submenu, x + dx, y);
			}
		}
		
		y   += dy;
		cell = cell->next;
	}
}


Bool
Menu::MenuOpen (MenuBody* menu, Card32 rank)
{
	//	Handle the case when the current menu doesn't match the
	//	menu which should be opened.
	
	if (rank != menu->rank) {
		
		MenuCell* cell = menu->cell;
		
		while (cell) {
			if (cell->submenu) {
				if (this->MenuOpen (cell->submenu, rank)) {
					return TRUE;
				}
			}
			cell = cell->next;
		}
		
		return FALSE;
	}
	
	MenuCell* cell = menu->cell;
	Card16    vert = MARGIN;
	
	make_window (this, menu->x, menu->y, menu->dx, menu->dy);
	this->LibString ("\255", 1); /* SAVWDO */
	draw_menu_frame (this, menu->dx, menu->dy);
	
	//	Open and draw the menu...
	
	while (cell) {
		
		if (cell->is_line) {
			draw_sep_line (this, MARGIN,
						   vert + menu->cell_ascender + menu->cell_descender + 2,
						   menu->cell_width);
		}
		
		use_color_clear (this, cell->is_hilite ? 0x1102 : 0x1101);
		use_color_set (this, cell->is_dimmed ? 0x0008 : 0x000B);
		
		MenuText* text = cell->text;
		set_cursor (this, MARGIN, vert + menu->cell_ascender + 1);
		
		while (text) {
			if (text->type == MENU_TEXT_STRING) {
				this->LibString (text->data, text->len);
			}
			
			text = text->next;
		}
		
		if (cell->submenu) {
			if (menu->x > cell->submenu->x) {
				set_cursor (this, MARGIN, vert + menu->cell_ascender + 1);
				this->LibString ("<", 1);
			} else {
				set_cursor (this, MARGIN+menu->dx, vert + menu->cell_ascender + 1);
				this->LibString ("\204>", 2);
			}
		}
		
		vert = vert + menu->cell_ascender + menu->cell_descender + 3;
		cell = cell->next;
	}
	
	use_color_clear (this, 0x00000001);
	use_color_set (this, 0x00000002);
	
	return TRUE;
}



Bool
Menu::MenuClose (MenuBody* menu, Card32 rank)
{
	//	Handle the case when the current menu doesn't match the
	//	menu which should be closed.
	
	if (rank != menu->rank) {
		
		MenuCell* cell = menu->cell;
		
		while (cell) {
			if (cell->submenu) {
				if (this->MenuClose (cell->submenu, rank)) {
					return TRUE;
				}
			}
			cell = cell->next;
		}
		
		return FALSE;
	}
	
	this->LibString ("\256", 1); /* GETWDO */
	
	return TRUE;
}


/*
 *	From a simple text description (compatible with the LIB format),
 *	build a hierarchical mouse menu.
 */

MenuBody*
Menu::BuildBody (const Card8* menu)
{
	const Card8* start_of_menu = menu;
	
	if (menu == 0) return 0;
	if (*menu == 0) return 0;
	
	MenuBody* body = new MenuBody;
	
	while (*menu) {
		MenuCell* cell = this->BuildCell (menu, start_of_menu);
		
		if (cell) {
			(body->cell_last ? body->cell_last->next : body->cell) = cell;
			body->cell_last = cell;
			body->cell_count++;
			if (body->cell_width     < cell->width)     body->cell_width     = cell->width;
			if (body->cell_ascender  < cell->ascender)  body->cell_ascender  = cell->ascender;
			if (body->cell_descender < cell->descender) body->cell_descender = cell->descender;
		}
	}
	
	return body;
}


/*
 *	Build a cell from the description. Return 0 if the current
 *	description is not a valid cell.
 */

MenuCell*
Menu::BuildCell (const Card8*& menu, const Card8* start_of_menu)
{
	if (*menu == 0x00) return 0;
	if (*menu == 0xFF) return 0;
	
	MenuCell* cell = new MenuCell;
	
	while (*menu) {
		
		if (*menu == 0xFF) {				//	separation line
			
			//	.8    -1
			
			cell->is_line = TRUE;
			menu         += 1;
			continue;
		}
		
		if (*menu == 0xFE) {				//	relative submenu
			
			//	.8.16 -2,MENU-START
			
			Card16 offset = (menu[1] << 8) + (menu[2]);
			Card32 ptr    = (Card32)(start_of_menu) + offset;
			cell->submenu = this->BuildBody ((const Card8*)(ptr));
			menu         += 3;
			continue;
		}
		
		if (*menu == 0xFD) {				//	resource icon
			
			//	.8.32 -3,icon_res_id
			
			Card32 res_id = (menu[1] << 24) + (menu[2] << 16) + (menu[3] << 8) + (menu[4]);
			menu         += 5;
			this->CellAddIcon (cell, this->ResGet (res_id, RtypRaster));
			continue;
		}
		
		if (*menu == 0xFC) {				//	absolute submenu
			
			//	.8.32 -4,MENU
			
			Card32 ptr    = (menu[1] << 24) + (menu[2] << 16) + (menu[3] << 8) + (menu[4]);
			cell->submenu = this->BuildBody ((const Card8*)(ptr));
			menu         += 5;
			continue;
		}
		
		if (*menu == 0xFB) {				//	resource submenu
			
			//	.8.32 -5,menu_res_id
			
			Card32 res_id = (menu[1] << 24) + (menu[2] << 16) + (menu[3] << 8) + (menu[4]);
			cell->submenu = this->BuildBody (this->ResGet (res_id, RtypMMenus));
			menu         += 5;
			continue;
		}
		
		this->CellAddText (cell, menu);
	}
	
	menu++;
	
	if (*menu == 0xFF) {
		cell->is_line = TRUE;
		menu++;
	}
	
	return cell;
}


/*
 *	Add an icon into the cell. This will be taken directly as
 *	a small [dx;dy] matrix.
 */

void
Menu::CellAddIcon (MenuCell* cell, const Card8* icon)
{
	if (icon[0] == 0) return;
	if (icon[1] == 0) return;
	
	cell->position  += icon[0];
	Card16 ascender  = icon[1] - this->baseline;
	Card16 descender = this->baseline;
	
	MenuText* text = new MenuText ();
	
	text->data = icon;
	text->type = MENU_TEXT_ICON;
	
	(cell->text_last ? cell->text_last->next : cell->text) = text;
	
	cell->text_last = text;
	
	if (cell->width < cell->position) cell->width     = cell->position;
	if (cell->ascender  < ascender)   cell->ascender  = ascender;
	if (cell->descender < descender)  cell->descender = descender;
}


/*
 *	Add some text into the cell. We will have to analyse the
 *	text and find the terminator.
 */

void
Menu::CellAddText (MenuCell* cell, const Card8*& string)
{
	if (*string == 0) return;
	
	Card16 ascender  = 0;
	Card16 descender = 0;
	
	MenuText*    text = new MenuText ();
	const Card8* font = 0;
	
	this->GetFontTable (font, ascender, descender);
	
	text->type = MENU_TEXT_STRING;
	text->data = string;
	
	while (*string) {
		
		if (*string == 0xA0) {				//	use new gencar
			
			this->LibString (string, 4);
			this->GetFontTable (font, ascender, descender);
			
			string    += 4;
			text->len += 4;
			continue;
		}
		
		if (*string == 0xB0) {				//	change horizontal position
			
			cell->position = (string[1] << 8) + (string[2]);
			if (cell->position > cell->width) cell->width = cell->position;
			
			string    += 3;
			text->len += 3;
			continue;
		}
		
		if (*string == 0xB8) {				//	set absolute color
			string    += 7;
			text->len += 7;
			continue;
		}
		
		if (*string == 0xBA) {				//	set relative color
			string    += 5;
			text->len += 5;
			continue;
		}
		
		if (*string == 0xFF) break;
		if (*string == 0xFE) break;
		if (*string == 0xFD) break;
		if (*string == 0xFC) break;
		if (*string == 0xFB) break;
		
		if (*string & 0x80) {
			string    += 1;
			text->len += 1;
			continue;
		}
		
		cell->position += font[*string];
		string         += 1;
		text->len      += 1;
		
		if (cell->position > cell->width) cell->width = cell->position;
	}
	
	(cell->text_last ? cell->text_last->next : cell->text) = text;
	
	cell->text_last = text;
	
	if (cell->ascender  < ascender)  cell->ascender  = ascender;
	if (cell->descender < descender) cell->descender = descender;
}


/*
 *	Send several bytes to the display driver.
 */

void
Menu::LibString (const Card8* string, Card32 len)
{
	asm volatile ( "movew %0,d6\n\t"
				   "movel %1,a4\n\t"
				   "movel %2,d4\n\t"
				   ".long 0x4E450002"
				 :
				 : "g" (this->dis_chan), "g" (string), "g" (len)
				 : "d4", "d6", "d7", "a4" );
}


/*
 *	Find out about the font width table and the ascender
 *	and descender. Update if needed.
 */

void
Menu::GetFontTable (const Card8*& font, Card16& ascender, Card16& descender)
{
	Card32 addr = ((Card32*)(this->dis_desc+0x2E))[0];
	Card32 off  = ((Card8*)(addr+10))[0];
	
	font = (Card8*)(addr + 256 - off);
	
	Card16 h1 = ((Card16*)(this->dis_desc+0x14))[0];
	Card16 h2 = ((Card16*)(this->dis_desc+0x18))[0];
	
	if (h1 > ascender)  ascender  = h1;
	if (h2 > descender) descender = h2;
}


/*
 *	Return the specified resource...
 */

const Card8*
Menu::ResGet (Card32 id, Card32 type)
{
	Card32 ptr = 0;
	Card16 err = 0;
	
	asm volatile ( "movel %2,a2\n\t"
				   "movel %3,d2\n\t"
				   "movel %4,d0\n\t"
				   "movel %5,d1\n\t"
				   ".long 0xA7A2\n\t"
				   "movew d7,%0\n\t"
				   "movel a4,%1"
				  : "=g" (err), "=g" (ptr)
				  : "g" (this->res_chan),
				    "g" (this->res_mode),
				    "g" (type), "g" (id)
				  : "d0", "d1", "d2", "d7", "a4" );
	
	return err ? 0 : (const Card8*)(ptr);
}



MenuBody::MenuBody ()
{
	this->x  = 0;
	this->y  = 0;
	this->dx = 0;
	this->dy = 0;
	
	this->rank           = 0;
	this->cell_width     = 0;
	this->cell_ascender  = 0;
	this->cell_descender = 0;
	this->cell_count     = 0;
	this->cell           = 0;
	this->cell_last      = 0;
}

MenuBody::~MenuBody ()
{
	if (this->cell) delete cell;
}


MenuCell::MenuCell ()
{
	this->next      = 0;
	this->submenu   = 0;
	this->width     = 0;
	this->ascender  = 0;
	this->descender = 0;
	this->position  = 0;
	this->is_hilite = FALSE;
	this->is_line   = FALSE;
	this->is_dimmed = FALSE;
	this->text      = 0;
	this->text_last = 0;
}

MenuCell::~MenuCell ()
{
	if (this->next) delete this->next;
	if (this->submenu) delete this->submenu;
	if (this->text) delete this->text;
}


MenuText::MenuText ()
{
	this->next = 0;
	this->type = MENU_TEXT_NONE;
	this->len  = 0;
	this->data = 0;
}

MenuText::~MenuText ()
{
	if (this->next) delete this->next;
}


