/*
 *	colorman.cxx
 *
 *	Little color manager for def_sms
 *
 *	Code taken from SMA_SMAKY lookman.cxx
 *	for most part
 *
 *	(C)	1995 Erik BRUCHEZ
 */

#include "colorman.h"
#include "lib/gra.h"
#include "lib/tcolor.h"

extern "C"  void *memcpy (void* dest, const void* source, unsigned long n);

/****************************************************************/

/*
 *	Numbers of the system CRET relative colors.
 */

enum RelativeColor
{
	COLOR_PPWHITE	= 10,		//	white
	COLOR_PPLGRAY	= 7,		//	light gray
	COLOR_PPMGRAY	= 8,		//	middle gray
	COLOR_PPDGRAY	= 9,		//	dark gray
	COLOR_PPBLACK	= 11,		//	black
	
	COLOR_SSBGMENU	= 0x1201,	//	menu background
	COLOR_SSBGSK	= 0x1202,	//	softkeys background
	COLOR_SSINSK	= 0x1203,	//	softkeys interior
	COLOR_SSMEM		= 0x1204,	//	memory jauge
	COLOR_SSCPU		= 0x1205,	//	cpu jauge
	COLOR_SSRED		= 0x1206,	//	adornment 1 (red)
	COLOR_SSGREEN	= 0x1207,	//	adornment 2 (green)
	COLOR_SSBLUE	= 0x1208,	//	adornment 3 (blue)
	COLOR_SSCLTXT	= 0x1209,	//	clock text
	COLOR_SSCAPSTXT	= 0x120a,	//	"caps" text
	COLOR_SSSKTXT	= 0x120b,	//	softkeys text
	COLOR_SSMSGTXT	= 0x120c,	//	messages text
	COLOR_SSSKBARS	= 0x120d,	//	softkeys bars
	COLOR_SSNET		= 0x120e,	//	network arrows
	COLOR_SSBGMEM	= 0x120f,	//	memory jauge background
	COLOR_SSBGCPU	= 0x1210,	//	cpu jauge background
	
	COLOR_DBCLEAR	= 0x1401	//	dbox background
};

/****************************************************************/

/*
 *	This array maps the colors to their system relative
 *	color number;
 */

static const RelativeColor map_relcol[SS_COL_NUM] = {
	COLOR_PPWHITE,		//	white
	COLOR_PPLGRAY,		//	light gray
	COLOR_PPMGRAY,		//	middle gray
	COLOR_PPDGRAY,		//	dark gray
	COLOR_PPBLACK,		//	black
	COLOR_DBCLEAR		//	background
};

/****************************************************************/

ColorMan::ColorMan ()
{
	this->gra = 0;
	this->cdata = 0;
	this->depth = 1;
	this->use_color = FALSE;
	this->clut256 = 0;
	
	this->mimage.data = 0;
	this->cimage.data = 0;
	
	this->force_mono = FALSE;
	this->clutimg = 0;
	
	this->icnumber = 0;
	
	this->err = 0;
	
	//	Set gra / tcolor
	
	this->gra = new Gra ((DisDesc *)(this->GetPDis ()));
	
	if (this->gra == 0) {
		this->err = 1;
		return;
	}
	this->tcolor = new TColor;
	if (this->tcolor == 0) {
		this->err = 1;
		return;
	}
	this->tcolor->Update (this->gra->GetTChannel ());
	
	//	Initialize conversion table
	
	for (int i = 0; i < 16; i++) {
		this->ctable[i] = i;
	}
	
	//	Allocate 16/256 colors clut
	
	this->clut256 = (TCClut *)(new char[sizeof (TCClut) + 256 * sizeof (TCEntry)]);
	
	if (this->clut256 == 0) {
		this->err = 1;
		return;
	}
	
	for (i = 0; i < SS_COL_NUM; i++) {
		this->colors[i] = i;
	}
}


static void
do_asm_convc (void* p1, void* p2, Card16 p3, Card16 p4, void* p5)
{
	asm volatile (	"	movel	%0,a2			\n\t"
					"	movel	%1,a3			\n\t"
					"	movew	%2,d2			\n\t"
					"	movew	%3,d3			\n\t"
					"	movel	%4,a4			\n\t"
					"	clrl	d4				\n\t"
					"	subqw	#1,d2			\n\t"
					"1:	movel	a2,a1			\n\t"
					"	movel	a3,a5			\n\t"
					"	movew	d3,d5			\n\t"
					"	subqw	#1,d5			\n\t"
					"2:	moveb	a1@+,d4			\n\t"
					"	moveb	a4@(0,d4:l),a5@+\n\t"
					"	dbra	d5,2b			\n\t"
					"	addw	d3,a2			\n\t"
					"	addw	d3,a3			\n\t"
					"	dbra	d2,1b			\n\t"
		:
		: "g" (p1), "g" (p2), "g" (p3), "g" (p4), "g" (p5)
		: "d2", "d3", "d4", "d5", "a1", "a2", "a3", "a4", "a5" );
}

void
ColorMan::ConvC (const ImageDescr *src, void *dest, const Card8 *tbl)
{
	if (this->depth == 8) {
		Card16 table[256];
		Card16* t = table;
		
		for (int i = 0; i < 16; i++) {
			for (int j = 0; j < 16; j++) {
				*t++ = (((Card16)(tbl[i])) << 8) + tbl[j];
			}
		}
		
		int dy = src->box.d.dy;
		int dx = src->box.d.dx / 2;
		
		Card16* pd = (Card16*)(dest);
		Card8*  ps = (Card8*)(src->data);
		
		while (dy--) {
			
			int    n = dx;
			Card8* p = ps;
			
			while (n--) {
				*pd++ = table[*p++];
			}
			
			ps += src->bwidth;
		}
		
//		asm ("movel %0,a4\ntrap #0" : : "g" (table) : "a4" );
		
		return;
	}
	
	Card8 table[256];
	Card8 *t = table;
	
	for (int i = 0; i < 16; i++) {
		for (int j = 0; j < 16; j++) {
			*t++ = (tbl[i] << 4) + tbl[j];
		}
	}
	
	t = table;
	
	do_asm_convc (src->data, dest, src->box.d.dy, src->bwidth, t);
}

void
ColorMan::LoadMonoImage (const void *mdata)
{
	//	Decode monochrome image
	
	Image header = *((Image *)(mdata));
	
	this->mimage.box.d.dx	= header.dx;
	this->mimage.box.d.dy	= header.dy;
	this->mimage.box.o.x	= 0;
	this->mimage.box.o.y	= 0;
	
	Card16 rbytes	= (header.dx + 7) / 8;
	
	this->mimage.bwidth = rbytes;
	
	this->mimage.data = new char[rbytes * this->mimage.box.d.dy];
	if (this->mimage.data == 0) {
		this->err = 1;
		return;
	}
	
	const void *encoded = mdata + LG_IMAGE;
	
	if (header.coding != 0)
		this->gra->DecodeImage (this->mimage.data, encoded, & header);
	else
		memcpy (this->mimage.data, encoded, header.length);
}

void
ColorMan::LoadColorImage (const void *gcdata)
{
	//	Decode color image
	//	Does it work ... always ?
	
	Image header = *((Image *)(gcdata));
	
	this->cimage.box.d.dx	= header.dx;
	this->cimage.box.d.dy	= header.dy;
	this->cimage.box.o.x	= 0;
	this->cimage.box.o.y	= 0;
	
	if (header.clutlen) {
		this->clutimg = (TCClut *)(new char[header.clutlen]);
		memcpy (this->clutimg, gcdata + LG_IMAGE, header.clutlen);
	}
	
	Card16 rbytes	= ((header.dx + 7) >> 3) * 4;
	
	this->cimage.bwidth  = rbytes;
	this->cimage.datalen = rbytes * this->cimage.box.d.dy;
	this->cimage.data    = new char[this->cimage.datalen + 64];
	
	if (this->cimage.data == 0) {
		this->err = 1;
		return;
	}
	
	this->cimage.data += 64;
	
	const void *encoded = gcdata + LG_IMAGE + header.clutlen;
	
	if (header.coding != 0) {
		Card16 old_dx    = header.dx;
		Card8  old_depth = header.depth;
		header.dx   *= 4;
		header.depth = 1;
		this->gra->DecodeImage (this->cimage.data, encoded, & header);
		header.dx    = old_dx;
		header.depth = old_depth;
	} else
		memcpy (this->cimage.data, encoded, header.length);
}

void *
ColorMan::GetPDis ()
{
	void *desc;
	
	asm ( ".long 0x4e460092	\n\t"	//	LIB?GETPDIS
		  "movel a4,%0		\n\t"
		: "=g" (desc)
		:
		: "a4", "d7" );
	
	return desc;
}

void
ColorMan::Init (const void *mresdata, const void *cresdata, Card16 icnumber, const Box *icons_def)
{
	//	Load monochrome and color images
	
	this->err = 0;
	
	if (this->mimage.data) {
		delete this->mimage.data;
		this->mimage.data = 0;
	}
	
	if (this->cimage.data) {
		delete this->cimage.data;
		this->cimage.data = 0;
	}
	
	this->LoadMonoImage (mresdata);
	if (this->err) return;
	this->LoadColorImage (cresdata);
	
	this->icnumber  = icnumber;
	this->icons_def = icons_def;
}


ColorMan::~ColorMan ()
{
	if (this->gra) delete this->gra;
	if (this->tcolor) delete this->tcolor;
	if (this->cdata) delete this->cdata;
	if (this->clut256) delete this->clut256;
	if (this->clutimg) delete this->clutimg;
}


Bool
ColorMan::FindColor (const GColor& color, Card32& index)
{
	Card32 r = color.r >> 8;
	Card32 g = color.g >> 8;
	Card32 b = color.b >> 8;
	
	Card32 limit = 0xffffffff;
	
	int num = 16;
	if (this->depth == 8) num = 256;
	
	for (int i = 0; i < num; i++) {
		Int32 dr = r - (this->clut256->entry[i].color.r >> 8);
		Int32 dg = g - (this->clut256->entry[i].color.g >> 8);
		Int32 db = b - (this->clut256->entry[i].color.b >> 8);
		
		dr *= dr;
		dg *= dg;
		db *= db;
		
		Card32 value = (Card32)(dr + dg + db);
		
		if (value < limit) {
			limit = value;
			index = i;
		}
		
		if (value == 0) return TRUE;
	}
	
	if (limit < 2000)	// about 10 % global tolerance
		return TRUE;
	else
		return FALSE;
}


void
ColorMan::Update ()
{
	this->err = 0;
	
	//	Look if we have colors
	
	this->depth = this->gra->GetDepth ();
	
	if ( (this->depth < 4)
	  || (this->depth > 8)
	  || (this->force_mono)
	  || (this->clut256 == 0) ) {
		this->use_color = FALSE;
		return;
	}
	
	//	From here we assume that we have 16 or 256 colors...
	
	this->use_color = TRUE;
	
	//	Read 16 or 256 colors clut
	
	this->tcolor->SaveClut (TC_TABLE, this->clut256);
	
	//	Find relative colors
	
	GColor ncolor[SS_COL_NUM];
	GColor ecolor;
	
	Card32 index;
	
	for (int i = 0; i < SS_COL_NUM; i++) {
		this->tcolor->SrcCret (map_relcol[i], ncolor[i], ecolor, index);
	}
	
	//	Find image colors ; color 1 is a special background color
	
	Card32 colors[16];
	
	for (i = 0; i < 16; i++) {
		this->FindColor (this->clutimg->entry[i].color, colors[i]);
	}
	
	this->FindColor (ncolor[SSBG], colors[1]);
	
	//	Find standard colors
	
	for (i = 0; i < SS_COL_NUM; i++) {
		this->FindColor (ncolor[i], this->colors[i]);
	}
	
	this->FindColor (ncolor[SSLGRAY], colors[12]);
	this->FindColor (ncolor[SSMGRAY], colors[13]);
	this->FindColor (ncolor[SSDGRAY], colors[14]);
	
	//	Build conversion table
	
	for (i = 0; i < 16; i++) {
		this->ctable[i] = colors[i];
	}
	
	//	If color pixmap, convert color image
	
	if (this->cdata == 0) {
		int mul = (this->depth == 4) ? 1 : 2;
		this->cdata = new char[this->cimage.datalen * mul];
		if (this->cdata == 0) {
			this->use_color = FALSE;
			return;
		}
	}
	
	this->ConvC (&this->cimage, this->cdata, this->ctable);
}


void
ColorMan::DrawIcon (Card16 num, Pos pos)
{
	if (num >= this->icnumber) return;
	
	this->gra->SetMode (LOADDOT);
	
	if (this->UseColor ()) {
		int width = (this->depth == 4) ? this->cimage.bwidth : (this->cimage.bwidth * 2);
		this->gra->RasterColor (this->icons_def[num], pos, this->cdata, width);
	} else {
		this->gra->RasterMono (this->icons_def[num], pos, this->mimage.data, this->mimage.bwidth);
	}
}


void
ColorMan::GetIconSize (Card16 num, Dim& dim) const
{
	if (num >= this->icnumber) {
		dim.dx = 0;
		dim.dy = 0;
	} else {
		dim = this->icons_def[num].d;
	}
}


void
ColorMan::DrawRectDown (Box rect)
{
	Box b = rect;
	b.d.dy = 0;
	
	this->gra->SetMode (SETDOT);
	this->gra->SetFgPixel (this->colors[SSDGRAY]);
	this->gra->Line (b);
	
	b.o.y++;
	b.d.dy = rect.d.dy - 1;
	b.d.dx = 0;
	
	this->gra->Line (b);
	
	b.o.x += rect.d.dx - 1;
	
	this->gra->SetFgPixel (this->colors[SSWHITE]);
	this->gra->Line (b);
	
	b.o.x = rect.o.x + 1;
	b.o.y = rect.o.y + rect.d.dy - 1;
	b.d.dx = rect.d.dx - 2;
	b.d.dy = 0;
	
	this->gra->Line (b);
}


void
ColorMan::DrawLine (Box line)
{
	this->gra->SetMode (SETDOT);
	
	if (this->UseColor ()) {
		if (line.d.dx == 0) {
			this->gra->SetFgPixel (this->colors[SSDGRAY]);
			this->gra->Line (line);
			this->gra->SetFgPixel (this->colors[SSWHITE]);
			line.o.x++;
		} else if (line.d.dy == 0) {
			this->gra->SetFgPixel (this->colors[SSDGRAY]);
			this->gra->Line (line);
			this->gra->SetFgPixel (this->colors[SSWHITE]);
			line.o.y++;
		}
	}
	
	this->gra->Line (line);
	this->RestoreColors ();
}

void
ColorMan::SetColorBack (SSColor color)
{
	this->gra->SetBgPixel (this->colors[color]);
}


void
ColorMan::SetColorFore (SSColor color)
{
	 this->gra->SetFgPixel (this->colors[color]);
}


void
ColorMan::RestoreColors ()
{
	this->gra->SetFgPixel (this->colors[SSBLACK]);
	this->gra->SetBgPixel (this->colors[SSBG]);
}

