/*
 *		drawcolima.cc
 *
 *		Dessine une image en couleur.
 *
 *		(C) 1993-1999,	Pierre ARNAUD, OPaC, CH-1437 SUSCEVAZ
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "page-struct.h"

extern void Say (const char*);
extern void Say (long);


// Classe pour lire les pixels dans une image couleur 
// profondeur variable.

class Pix {
	
	const Card8* data;
	const Card8* clut;
	Card8*		my_clut;
	int			x;
	int			dx;
	int			pos_byte;
	int			pos_bit;
	int			width;
	int			depth;
	char		invert_mono;

public:
	Pix (const void* d, const char* c, int wx, int bpp, int w, char invert = 0)
		{
			int num_color = 1 << bpp;
			data = (Card8*)(d);
			clut = (Card8*)(c) + 32;
			
			if ( (bpp == 4)
			  || (bpp == 8) ) {
				my_clut = new Card8[num_color * 4];
				if (my_clut == 0) abort ();
			} else {
				my_clut = 0;
			}
			x  = 0;
			dx = wx;
			pos_byte = 0;
			pos_bit  = 0;
			depth = bpp;
			width = w;
			invert_mono = invert;
			
			if (my_clut) {
				for (int i = 0; i < num_color; i++) {
					const Card8* ptr = clut + 10*i + 4;
					Card32 index = ((const Card32*)(ptr))[-1] * 4;
					my_clut[index+0] = ptr[0];
					my_clut[index+1] = ptr[2];
					my_clut[index+2] = ptr[4];
				}
			}
			
			// Say ("Width = "); Say (w); Say (", bpp = "); Say (bpp); Say ("\r");
		}
	~Pix () { if (my_clut) delete my_clut; }
	void Next (int& red, int& green, int& blue);
	void Next (int& grey);
};


void
Pix::Next (int& red, int& green, int& blue)
{
	red   = 0;
	green = 0;
	blue  = 0;
	
	int i;
	
	switch (depth) {
		
		case 32:
			pos_byte++;
			// continue sinon comme 24 bit/pixel
		
		case 24:
			red   = data[pos_byte++];
			green = data[pos_byte++];
			blue  = data[pos_byte++];
			break;
		
		case 1:
			if (data[pos_byte] & (128 >> pos_bit)) {
				red = green = blue = 0xFF;
			}
			if (invert_mono) {
				red = green = blue = (red ? 0x00 : 0xFF);
			}
			
			if (pos_bit == 7) {
				pos_bit = 0;
				pos_byte++;
			} else {
				pos_bit++;
			}
			break;
		
		case 4:
			if (pos_bit) {
				i = data[pos_byte] & 0x0F;
				pos_bit = 0;
				pos_byte++;
			} else {
				i = (data[pos_byte] >> 4) & 0x0F;
				pos_bit++;
			}
			
			{
				const Card8* color = my_clut+i*4;
				red   = *color++;
				green = *color++;
				blue  = *color++;
			}
			break;
		
		case 8:
			{
				const Card8* color = my_clut+data[pos_byte++]*4;
				red   = *color++;
				green = *color++;
				blue  = *color++;
			}
			break;
		
		default:
			break;
	}
	
	x++;
	
	if (x >= dx) {
		x = 0;
		pos_byte = 0;
		pos_bit = 0;
		data += width;
	}
}

void
Pix::Next (int& grey)
{
	int r, g, b;
	
	Next (r, g, b);
	
	grey = (300 * r + 586 * g + 114 * b) / 1000;
	
	if (grey > 255) grey = 255;
}


// Dessine une image couleur

void
PagePS::DrawColorImage (int x1, int y1, int x2, int y2, int pdx, int pdy,
						float angle, int bpp, int width,
						const void* clut, const void* image)
{
	char invert_mono = 0;
	
	if (bpp == 0) {
		bpp         = 1;
		invert_mono = 1;
	}
	
	SendGlyphs ();
	
	int dx = x2 - x1;
	int dy = y2 - y1;
	
	if (bpp == 1) {
		
		output->Color (0, 0, 0);		//	utilise du noir pour dessiner les points...
    
	    Bool odd   = (pdy & 1) ? TRUE : FALSE;
    	int  bytes = (pdx + 7) >> 3;
    	
    	if (odd) pdy++;
    	
	    output->Send (pdx);		// largeur
	    output->Send (pdy);		// hauteur
	    output->Send (dx);		// largeur en points de PAGE
	    output->Send (dy);		// hauteur en points de PAGE
	    output->Send (angle);	// angle de rotation
	    output->Send (x1);		// origine x
	    output->Send (y1);		// origine y
	    output->Write ("im\n");	// commande
		
		if (odd) pdy--;
    	
    	while (pdy--) {
    		
    		char copy[bytes];
    		memcpy (copy, image, bytes);
    		
    		if (invert_mono) {
    			for (int i = 0; i < bytes; i++) {
    				copy[i] = ~ copy[i];
    			}
    		}
    		
    		output->Send (image, bytes, FALSE);
    		image += width;
    	}
    	
		if (odd) {
			char zero[bytes];
			bzero (zero, bytes);
			output->Send (zero, bytes, FALSE);
		}
		
		output->Write ("\n");
		return;
	}
	
	if (output->Level () == 1) {
		
		if (is_color) {
			
			// dessine l'image couleur pour une imprimante Level 1 couleur !!!
			
			output->Send (pdx);			// largeur
			output->Send (pdy);			// hauteur
			output->Send (dx);			// largeur en points de PAGE
			output->Send (dy);			// hauteur en points de PAGE
			output->Send (angle);		// angle de rotation
			output->Send (x1);			// origine x
			output->Send (y1);			// origine y
			output->Write ("imc\n");	// commande
			output->UseColor ();
			Pix pix (image, (const char*)(clut), pdx, bpp, width, invert_mono);
			
			long i = pdx*pdy;
			
			while (i--) {
				
				char buffer[3];
				int r, g, b;
				
				pix.Next (r, g, b);
				
				buffer[0] = r;
				buffer[1] = g;
				buffer[2] = b;
				
				output->Send (buffer, 3, FALSE);
			}
			
			output->Write ("\n");
			
		} else {
			
			// dessine l'image couleur avec les moyens du bord, c'est--dire en gris
			
			output->Send (pdx);			// largeur
			output->Send (pdy);			// hauteur
			output->Send (dx);			// largeur en points de PAGE
			output->Send (dy);			// hauteur en points de PAGE
			output->Send (angle);		// angle de rotation
			output->Send (x1);			// origine x
			output->Send (y1);			// origine y
			output->Write ("im8\n");	// commande
			
			Pix pix (image, (const char*)(clut), pdx, bpp, width, invert_mono);
			
			long i = pdx*pdy;
			unsigned int grey, log, hig;
			char buffer[64];
			
			while (i--) {
				
				pix.Next (grey);
				
				log = (grey & 0x0F);
				hig = (grey >> 4);
				
				buffer[0] = hig + (hig < 10 ? '0' : ('a' - 10));
				buffer[1] = log + (log < 10 ? '0' : ('a' - 10));
				
				output->Write (buffer, 2);
			}
			
			sprintf (buffer, "%% %dx%d => %d (width=%d)\n", pdx, pdy, pdx*pdy, width);
			
			output->Write ("\n");
			output->Write (buffer);
		}
		
	} else {
		
		// dessine l'image couleur en Level 2
		
		if (bpp == 4) {
			
			int i;
			char idx_clut[16*3];
			
			for (i = 0; i < 16; i++) {
				
				const Card8* color	= (const Card8*)(clut)+32+10*i+4;
				Card32		 index	= ((const Card32*)(color))[-1];
				
				idx_clut[index*3+0] = color[0];
				idx_clut[index*3+1] = color[2];
				idx_clut[index*3+2] = color[4];
				
				if ( (color[0] == color[2])
				  && (color[0] == color[4]) ) {
					//	gray level...
				} else {
					output->UseColor ();
				}
			}
			
			output->Send (dx); output->Send (dy);
			output->Send (angle);
			output->Send (x1); output->Send (y1);
			
			output->Write ("gsave translate rotate scale\n");
			output->Write ("[ /Indexed /DeviceRGB 15\n");
			output->Send (idx_clut, 16*3, TRUE);
			output->Write (" ] setcolorspace\n");
			
			output->Write ("<< /ImageType 1 ");
			output->Write ("/Width "); output->Send (pdx);
			output->Write ("/Height "); output->Send (pdy);
			output->Write ("/ImageMatrix [ "); output->Send (pdx);
			output->Write ("0 0 "); output->Send (pdy);
			output->Write ("0 0]\n");
			output->Write ("/DataSource currentfile 1 (%%EndBinary) /SubFileDecode filter\n");
			output->Write ("/BitsPerComponent 4\n");
			output->Write ("/Decode [0 15]\n");
			output->Write ("/Interpolate true\n>> image\n");
			
			pdx += 1;
			pdx /= 2;
			
			while (pdy--) {
				output->Send (image, pdx, FALSE);
				image += width;
			}
			
			output->Write ("\n%%EndBinary\ngrestore\n");
			return;
		}
		
		if (bpp == 8) {
			
			int i;
			char idx_clut[256*3];
			
			for (i = 0; i < 256; i++) {
				
				const Card8* color	= (const Card8*)(clut)+32+10*i+4;
				Card32		 index	= ((const Card32*)(color))[-1];
				
				idx_clut[index*3+0] = color[0];
				idx_clut[index*3+1] = color[2];
				idx_clut[index*3+2] = color[4];
				
				if ( (color[0] == color[2])
				  && (color[0] == color[4]) ) {
					//	gray level...
				} else {
					output->UseColor ();
				}
			}
			
			output->Send (dx); output->Send (dy);
			output->Send (angle);
			output->Send (x1); output->Send (y1);
			
			output->Write ("gsave translate rotate scale\n");
			output->Write ("[ /Indexed /DeviceRGB 255\n");
			output->Send (idx_clut, 256*3, TRUE);
			output->Write (" ] setcolorspace\n");
			
			output->Write ("<< /ImageType 1 ");
			output->Write ("/Width "); output->Send (pdx);
			output->Write ("/Height "); output->Send (pdy);
			output->Write ("/ImageMatrix [ "); output->Send (pdx);
			output->Write ("0 0 "); output->Send (pdy);
			output->Write ("0 0]\n");
			output->Write ("/DataSource currentfile 1 (%%EndBinary) /SubFileDecode filter\n");
			output->Write ("/BitsPerComponent 8\n");
			output->Write ("/Decode [0 255]\n");
			output->Write ("/Interpolate true\n>> image\n");
			
			while (pdy--) {
				output->Send (image, pdx, FALSE);
				image += width;
			}
			
			output->Write ("\n%%EndBinary\ngrestore\n");
			return;
		}
		
		output->Send (pdx);				// largeur
		output->Send (pdy);				// hauteur
		output->Send (dx);				// largeur en points de PAGE
		output->Send (dy);				// hauteur en points de PAGE
		output->Send (angle);			// angle de rotation
		output->Send (x1);				// origine x
		output->Send (y1);				// origine y
		output->Write ("imc\n");		// commande
		
		Pix pix (image, (const char*)(clut), pdx, bpp, width, invert_mono);
		
		long i = pdx*pdy;
		
		while (i--) {
			
			char buffer[3];
			int r, g, b;
			
			pix.Next (r, g, b);
			
			buffer[0] = r;
			buffer[1] = g;
			buffer[2] = b;
			
			output->Write (buffer, 3);
		}
		
		output->Write ("\n");
		output->UseColor ();
	}
}


