
/*
 *		drawglyph.cc
 *
 *		Dessine un petit motif graphique monochrome avec transparence
 *		(pour les caractres, par exemple).
 *
 *		(C) Copyright 1993-1995,  Pierre ARNAUD, OPaC, CH-1437 SUSCEVAZ
 */

#include <stdio.h>
#include <string.h>

#include "page-struct.h"
#include "latinpage.h"
#include "metrics/hr.h"
#include "metrics/hb.h"
#include "metrics/hi.h"
#include "metrics/tr.h"
#include "metrics/tb.h"
#include "metrics/ti.h"
#include "metrics/cr.h"
#include "metrics/cb.h"
#include "metrics/cbo.h"

extern Bool g_smart_text;
extern Bool debug;


Bool g_smart_text = TRUE;

/*
 *	Une description de glyph en provenance de PAGE est base sur
 *	des paramtres XCAR. Ceux-ci peuvent tre utiliss pour trouver
 *	une fonte PostScript adquate :
 */

struct SuperGlyph
{
	Card16				oy;
	Card16				ox;
	Card16				flags;
	Card16				code;
	const char*			font;
	Card32				rot;
	Card16				slant_y;
	Card16				slant_x;
	Card16				scale_y;
	Card16				scale_x;
	Card16				width;
	Card16				width_E;
};


/*
 *	Pour chaque fonte PostScript, il y a un nom de procdure associ
 *	ainsi qu'une table des chasses.
 */

struct NameToFontInfo {
	const char*			name;					//	nom de la fonte PostScript
	const char*			proc;					//	nom de la procdure
	const Card16*		array;					//	pointeur sur la table des chasses
};

static const NameToFontInfo
g_name_to_font_info[] = {
	{ "Helvetica",				"a",  g_HR_font_width },
	{ "Helvetica-Bold",			"b",  g_HB_font_width },
	{ "Helvetica-Oblique",		"c",  g_HI_font_width },
//	{ "Helvetica-BoldOblique",	"d",  ... },
	{ "Times-Roman",			"e",  g_TR_font_width },
	{ "Times-Bold",				"f",  g_TB_font_width },
	{ "Times-Italic",			"g",  g_TI_font_width },
//	{ "Times-BoldItalic",		"h",  ... },
	{ "Courier",				"i",  g_CR_font_width },
	{ "Courier-Bold",			"j",  g_CB_font_width },
//	{ "Courier-Oblique",		"k",  g_CO_font_width },
	{ "Courier-BoldOblique",	"l",  g_CBO_font_width },
	{ 0,						  0,  0 }
};

static const char*
g_name_font_map[] = {
	"Helvetia",					"Helvetica",
	"Helvetia_B",				"Helvetica-Bold",
	"Helvetia_BI",				"Helvetica-BoldOblique",
	"Helvetia_I",				"Helvetica-Oblique",
	"Classic",					"Times-Roman",
	"Classic_B",				"Times-Bold",
	"Classic_BI",				"Times-BoldItalic",
	"Classic_I",				"Times-Italic",
	"Machine",					"Courier",
	"Machine_B",				"Courier-Bold",
	"Machine_BI",				"Courier-BoldOblique",
//	"Machine_I",				"Courier-Oblique",
	"Machine_F",					"Courier",
	"Machine_BF",				"Courier-Bold",
	"Machine_BIF",				"Courier-BoldOblique",
//	"Machine_IF",				"Courier-Oblique",
	0,							0
};


/*
 *	Associ  l'instance du document en cours d'impression, on
 *	conserve un record priv qui nous permet d'optimiser les
 *	transferts en rduisant le nombre de changements de fontes
 *	indispensables.
 */

struct DocFontInfo
{
	const NameToFontInfo*		font;
	Card16						x, y;
	Card16						size;
	Card16						estimated_x;
};


/*
 *	Tente de dessiner un glyph au moyen d'une fonte de l'imprimante,
 *	si cela s'avre possible...
 */

Bool
PagePS::DrawFontGlyph (const SuperGlyph* glyph, int color)
{
	Bool   font_ok = this->glyph_font_ok;
	Card16 code	   = g_map_page_to_latin[glyph->code];
	
	const char*			  s_font = glyph->font;
	const char**		  d_font = g_name_font_map;
	const NameToFontInfo* info	 = g_name_to_font_info;
	char buffer[16];
	
	if ( (glyph->slant_y != 0)
	  || (glyph->slant_x != 0)
	  || (glyph->scale_y != 100)
	  || (glyph->scale_x != 100)
	  || (glyph->rot != 0)
	  || (code == 0) ) {
		return FALSE;
	}
	
	//	Cherche la fonte PostScript correspondant  la fonte PAGE ou XCAR
	//	propose en entre...
	
	while (d_font[0]) {
		if (strcmp (s_font, d_font[0]) == 0) {
			break;
		}
		d_font += 2;
	}
	
	if (d_font[0] == 0) {
		return FALSE;
	}
	
	//	Cherche  prsent les informations plus dtailles sur la fonte
	//	PostScript retenue.
	
	while (info->name) {
		if (strcmp (info->name, d_font[1]) == 0) {
			break;
		}
		info++;
	}
	
	if ( (info->name == 0)
	  || (info->array[code] == 0) ) {
		return FALSE;
	}
	
	DocFontInfo* dfi = (DocFontInfo*)(this->glyph_font_cache);
	
	if (dfi == 0) {
		dfi = new DocFontInfo;
		this->glyph_font_cache = dfi;
		if (dfi == 0) return FALSE;
		
		dfi->font = 0;
		dfi->x	  = 0;
		dfi->y	  = 0;
		dfi->size = 0;
		dfi->estimated_x = 0;
	}
	
	
	//	Les informations sur la fonte sont enfin disponibles... On peut
	//	procder  la suite !
	
	this->SendGlyphs ();
	
	float rset = this->color[color].red	  / 65535.0;
	float gset = this->color[color].green / 65535.0;
	float bset = this->color[color].blue  / 65535.0;
	
	this->output->Color (rset, gset, bset);
	
	float  font_E_size = (info->array['E'] / 1000.0);
	float  need_E_size = (glyph->width_E);
	Card16 scale_font  = (Card16)(need_E_size/font_E_size * 1270.0/this->dpi_x);
	
	float  font_real_size = (info->array[code] / 1000.0)
						  * need_E_size / font_E_size
						  * 1270.0 / this->dpi_x;
	
	buffer[0] = code;
	buffer[1] = 0;
	
	if (this->output->Level () == 1) {
		if ((code == '(') || (code == ')') || (code == '\\')) {
			buffer[0] = '\\';
			buffer[1] = code;
			buffer[2] = 0;
		} else if ((code < 32) || (code > 126)) {
			sprintf (buffer, "\\%03o", code);
		}
	}
	
	if ( (font_ok)
	  && (dfi->font == info)
	  && (dfi->size == scale_font) ) {
		
		//	Optimise le cas o la fonte est la mme qu'avant, qu'aucune autre
		//	primitive graphique n'a t mise entre temps et que l'on est sur
		//	la mme page qu'avant.
		//
		//	Si on est sur la mme ligne, on s'pargne un transfert inutile de
		//	donnes (en effet, `y' est inutile).
		
		if (dfi->y == glyph->oy) {
			if (g_smart_text) {
				int delta = dfi->estimated_x - glyph->ox;
				int move  = glyph->ox-dfi->x;
				if (delta < 0) delta = -delta;
				if (delta*10 > move) {			//	accepte jusqu' 10% d'erreur en [x] !
					this->output->EmitShow ();
//					this->output->Write ("%>\n");
				}
				this->output->Show (buffer, glyph->ox, glyph->oy, font_real_size, glyph->width);
			} else {
				this->output->Send (buffer);
				this->output->Send (glyph->ox-dfi->x);
				this->output->Write ("s\n");
			}
		} else {
			this->output->Send (buffer);
			this->output->Send (glyph->ox);
			this->output->Send (glyph->oy);
			this->output->Write ("S\n");
		}
		
	} else {
		
		this->output->Send (buffer);
		this->output->Send (scale_font);
		this->output->Send (glyph->ox);
		this->output->Send (glyph->oy);
		
		strcpy (buffer, "$0");
		strcat (buffer, info->proc);
		strcat (buffer, "\n");
		this->output->Write (buffer);
	}
	
	this->glyph_font_ok = TRUE;
	
	dfi->font = info;
	dfi->x	  = glyph->ox;
	dfi->y	  = glyph->oy;
	dfi->size = scale_font;
	dfi->estimated_x = dfi->x + (int)font_real_size - 1;
	
	return TRUE;
}



void
PagePS::DrawHugeGlyph (int x, int y, int dx, int dy, int width, int color,
				   long id1, long id2, const void* image)
{
	
	//	C'est un gros caractre (compte prs de 650 mm2 dans le cas
	//	d'une imprimante 300 dpi). Il ne faut pas le mettre dans le
	//	cache, mais l'envoyer comme une image.
	
	this->SendGlyphs ();
	
	float rset = this->color[color].red	  / 65535.0;
	float gset = this->color[color].green / 65535.0;
	float bset = this->color[color].blue  / 65535.0;
	
	this->output->Color (rset, gset, bset);
	
	this->output->Write ("/.s ");
	this->output->Send (width);
	this->output->Write (" string def\n");
	
	this->output->Write ("gsave ");
	this->output->Send (x);				// origine x
	this->output->Send (y);				// origine y
	this->output->Write (" translate\n");
	
	this->output->Send (dx);			// largeur en points
	this->output->Send (dy);			// hauteur en points
	this->output->Write (" true GlyphMatrix {currentfile .s ");
	this->output->Write ((this->output->Level () == 1) ? "readhexstring" : "readstring");
	this->output->Write (" pop} imagemask\n");
	
	while (dy--) {
		this->output->Send (image, width, FALSE);
		image += width;
	}
	
	this->output->Write ("\ngrestore\n");
}


/*
 *	Dessine un glyph et gre les cas suivants :
 *
 *	1. C'est un glyph qui est mul par une fonte interne de
 *	   l'imprimante.
 *
 *	2. C'est un glyph norme qui est gr comme une image en
 *	   monochrome pour ne pas saturer le cache.
 *
 *	3. C'est un petit glyph qui est simplement gliss dans le
 *	   cache adquat.
 */

void
PagePS::DrawGlyph (int x, int y, int dx, int dy, int width, int color,
				   long id1, long id2, const void* image, const void* glyph_ptr)
{
	if ( (glyph_ptr)
	  && (this->DrawFontGlyph ((const SuperGlyph*)(glyph_ptr), color)) ) {
		return;
	}
	
	if (debug) {
		if (glyph_ptr == 0) {
			this->output->Write ("% no-font\n");
		} else {
			const SuperGlyph* sg = (const SuperGlyph*)(glyph_ptr);
			this->output->Write ("% ");
			this->output->Write (sg->font);
			this->output->Send (sg->slant_x);
			this->output->Send (sg->slant_y);
			this->output->Send (sg->scale_x);
			this->output->Send (sg->scale_y);
			this->output->Send (sg->rot);
			this->output->Write ("\n");
		}
	}
	
	if (dx * dy > 90000) {
		this->DrawHugeGlyph (x, y, dx, dy, width, color, id1, id2, image);
		return;
	}
	
	Glyph glyph;
	
	glyph.dx = dx;
	glyph.dy = dy;
	glyph.width = width;
	glyph.id1 = id1;
	glyph.id2 = id2;
	
	cache.Store (glyph, x, y, color, image);
}


void
PagePS::KillFontCache ()
{
	DocFontInfo* info = (DocFontInfo*)(this->glyph_font_cache);
	if (info) delete info;
	this->glyph_font_cache = 0;
}


void
PagePS::PurgeFontCache ()
{
	//	Ne fait rien d'autre que de mettre le fanion  zro...
	//	Ceci est utile pour la mthode DrawFontGlyph.
	
	this->glyph_font_ok = FALSE;
}


void
PagePS::SendGlyphs ()
{
	if (this->glyph_font_ok) {
		this->PurgeFontCache ();
	}
	
	
	// Passe en revue tous les glyphs stocks dans le cache et
	// envoie les de faon rationnelle vers l'imprimante...
	
	Glyph* glyph;
	GlyphInst* inst;
	float r,g,b;
	
	r = g = b = 0.0;
	
	output->Color (r, g, b);
	
	while (cache.first_cache) {
		
		glyph = cache.first_cache;
		inst = cache.first_inst;
		
		// met le glyph vers l'imprimante
		
		output->Send (glyph->data, glyph->dy * glyph->width, TRUE);
		output->Send (glyph->dx);
		output->Send (glyph->dy);
		output->Send (glyph->width);
		
		output->Write ("sg\n");
		
		int old_x = 0;
		int old_y = 0;
		
		while (inst) {
			if (inst->glyph == glyph) {
				float rset = this->color[inst->color].red / 65535.0;
				float gset = this->color[inst->color].green / 65535.0;
				float bset = this->color[inst->color].blue / 65535.0;
				
				output->Send (inst->x - old_x);
				output->Send (inst->y - old_y);
				output->Color (rset, gset, bset);
				
				output->Write ("g\n");
				
				old_x = inst->x;
				old_y = inst->y;
			}
			
			inst = inst->next;
		}
		
		output->Write ("eg\n");
		
		// supprime le glyph en question
		
		cache.first_cache = glyph->next;
		glyph->next = 0;
		delete glyph;
	}
	
	cache.Purge ();
}


Glyph::Glyph ()
{
	count = 0;
	next = 0;
	data = 0;
}

Glyph::~Glyph ()
{
	if (data) delete data;
	if (next) delete next;
}

int
operator == (const Glyph& a, const Glyph& b)
{
	return (a.id1 == b.id1) && (a.id2 == b.id2);
}

GlyphInst::GlyphInst ()
{
	next = 0;
	glyph = 0;
}

GlyphInst::~GlyphInst ()
{
	if (next) delete next;
}


void
Cache::Purge ()
{
	{
		GlyphInst* mine = first_inst;
		GlyphInst* next;
		
		while (mine) {
			next = mine->next;
			mine->next = 0;
			delete mine;
			mine = next;
		}
	}
	
	{
		Glyph* mine = first_cache;
		Glyph* next;
		
		while (mine) {
			next = mine->next;
			mine->next = 0;
			delete mine;
			mine = next;
		}
	}
	
	first_inst = 0;
	first_cache = 0; 
}


// Place un glyph dans la page et dans le cache s'il n'y est pas
// encore. C'est lors du PagePS::NewPage que les glyphs sont
// rellement transmis  l'imprimante.

void
Cache::Store (const Glyph& glyph, int x, int y, int color, const void* image)
{
	Glyph* list = first_cache;
	Glyph* found = 0;
	
	while (list) {
		if (*list == glyph) {
			found = list;
			break;
		}
		
		list = list->next;
	}
	
	if (found == 0) {
		
		// Cre un nouveau glyph dans le cache, puisque rien n'a t trouv !
		
		found = new Glyph;
		if (found == 0) return;
		
		*found = glyph;
		found->data = new char[found->dy*found->width];
		
		if (found->data == 0) {
			delete found;
			return;
		}
		
		bcopy (image, found->data, found->dy*found->width);
		
		found->next = first_cache;
		first_cache = found;
	}
	
	// Insre une instance du glyph...
	
	GlyphInst* inst = new GlyphInst;
	if (inst == 0) return;
	
	found->count++;
	
	inst->next = first_inst;
	inst->glyph = found;
	inst->x = x;
	inst->y = y;
	inst->color = color;
	
	first_inst = inst;
}
