
/*
 *		drawline.cc
 *
 *		Dessin de lignes conformes aux dsirs de PAGE.
 *
 *		(C) 1993-1997,	Pierre ARNAUD, OPaC, CH-1437 SUSCEVAZ
 */

extern "C" {
#include <stdlib.h>
#include <math.h>
}

#include "page-struct.h"



// Dessine une ligne avec un trait quelconque, tel que dfini
// dans PAGE. Si `reset' vaut TRUE, on reprend le traitill au
// dbut. Par dfaut, le traitill continue l o on l'avait
// abandonn (pour les polygones et les courbes de Bzier).

void
PagePS::DrawLine (int x1, int y1, int x2, int y2, int pen, Bool reset = FALSE)
{
	SendGlyphs ();
	
	// Si le pinceau utilis est compatible avec les traitills de
	// l'imprimante, laisse l'imprimante grer le dessin des traits
	// et des trous.
	
	Bool hv = (x1 == x2) || (y1 == y2);
	
	if (reset) {
		dash_pos = 0.0;
		last_dash_pos = 0.0;
	}
	
	if (SelectDash (pen, hv)) {
		
		DrawLinePiece (x1, y1, x2, y2, pen);
		
		int dx = x2-x1;
		int dy = y2-y1;
		
		dash_pos += sqrt ((dx*dx)+(dy*dy));
		
		return;
	}
	
	// En cas de changement de pinceau, il faut prendre la peine
	// de rinitialiser la table des traitills dfinis de faon
	// absolue...
	
	if (pen != dash.pen) {
		
		int count = 0;
		dash.len = 0;
		
		while (count < 8) {
			
			int gap_drawn = this->pen[pen].gap[count];
			int gap_empty = this->pen[pen].gap[count+1];
			
			if (gap_empty <= 0) break;
			
			if (gap_drawn == 0) {
				if (--gap_empty < 0) {
					gap_empty = 0;
				}
			}
			
			dash.len += gap_drawn;
			dash.len += gap_empty;
			
			dash.gap[count]   = gap_drawn;
			dash.gap[count+1] = gap_empty;
			
			count += 2;
		}
		
		if (count < 8) dash.gap[count] = -1;
		
		reset = TRUE;
		dash.pen = pen;
	}
	
	if (reset) dash.pos = dash.num = 0;
	
	// Traitill, il faut grer cela soi-mme, car les tratills
	// de PAGE ne correspondent pas aux traitills de PostScript.
	
	int dx = x2 - x1;
	int dy = y2 - y1;
	Bool neg_x = (dx < 0);
	Bool neg_y = (dy < 0);
	int len_x = (neg_x ? -dx : dx);
	int len_y = (neg_y ? -dy : dy);
	int x = x1;
	int y = y1;
	int inc, len;
	
	if (len_x > len_y) {
		
		// Le trait reste prs de l'horizontale, il faut prendre `x'
		// comme modle pour l'incrment...
		
		inc = len_x;
		
		for (;;) {
			
			int old_x2;
			int gap = dash.gap[dash.num] - dash.pos;
			len = (inc > gap) ? gap : inc;
			
			old_x2 = x2 = x + (neg_x ? -len : len);
			y2 = y1 + ((x2 - x1) * dy) / dx;
			
			if (len == 0) x2 += (neg_x ? -1 : 1);
			
			dash.pos += len;
			
			if (Even (dash.num)) DrawLinePiece (x, y, x2, y2, pen);
			
			x = old_x2;
			y = y2;
			
			inc -= len;
			if (inc == 0) break;
			
			if (++dash.num > 7) dash.num = 0;
			if (dash.gap[dash.num] < 0) dash.num = 0;
			
			dash.pos = 0;
		}
	
	} else {
		
		// Le trait reste prs de la verticale, il faut prendre `y'
		// comme modle pour l'incrment...
		
		inc = len_y;
		
		for (;;) {
			
			int old_y2;
			int gap = dash.gap[dash.num] - dash.pos;
			len = (inc > gap) ? gap : inc;
			
			old_y2 = y2 = y + (neg_y ? -len : len);
			x2 = x1 + ((y2 - y1) * dx) / dy;
			
			if (len == 0) y2 += (neg_y ? -1 : 1);
			
			dash.pos += len;
			
			if (Even (dash.num)) DrawLinePiece (x, y, x2, y2, pen);
			
			x = x2;
			y = old_y2;
			
			inc -= len;
			if (inc == 0) break;
			
			if (++dash.num > 7) dash.num = 0;
			if (dash.gap[dash.num] < 0) dash.num = 0;
			
			dash.pos = 0;
		}
	}
}




// Dessine juste un morceau de trait plein en tenant juste
// compte de l'aspect du trait (extrmits).

void
PagePS::DrawLinePiece (int x1, int y1, int x2, int y2, int pen)
{
	if (this->pen[pen].width == 0)
		return;
	
	float rset,gset,bset;
	
	rset = this->color[this->pen[pen].color].red / 65535.0;
	gset = this->color[this->pen[pen].color].green / 65535.0;
	bset = this->color[this->pen[pen].color].blue / 65535.0;
	
	output->Color (rset, gset, bset);
	
	if (this->pen[pen].flags & PEN_CIRCLE) {
		
		output->Send (this->pen[pen].width);
		output->Send (x2); output->Send (y2);
		output->Send (x1); output->Send (y1);
		output->Send (PS_ROUND_LINE);
		
	} else if (this->pen[pen].flags & PEN_ELLIPSE) {
		
		output->Send (this->pen[pen].width);
		output->Send (this->pen[pen].height);
		output->Send (x2); output->Send (y2);
		output->Send (x1); output->Send (y1);
		output->Send (PS_ELLIP_LINE);
		
	} else if (this->pen[pen].flags & PEN_SQUARE) {
		
		if ((x2 == x1) || (y2 == y1) || (!page_mode)) {
			output->Send (this->pen[pen].width);
			output->Send (x2); output->Send (y2);
			output->Send (x1); output->Send (y1);
			output->Send (PS_SQUARE_LINE);
			return;
		}
		
		long xl1, xl2, xr1, xr2;
		long yt1, yt2, yb1, yb2;
		
		long w1 = this->pen[pen].width >> 1;
		long w2 = this->pen[pen].width - w1;
		
		xl1 = x1 - w1;	  yb1 = y1 - w1;
		xr1 = x1 + w2;	  yt1 = y1 + w2;
		xl2 = x2 - w1;	  yb2 = y2 - w1;
		xr2 = x2 + w2;	  yt2 = y2 + w2;
		
		output->Send (xl1); output->Send (yb1);
		output->Send (xr1); output->Send (yt1);
		output->Send (xl2); output->Send (yb2);
		output->Send (xr2); output->Send (yt2);
		
		output->Send (xl1); output->Send (yb1);
		output->Send (xr1); output->Send (yb1);
		output->Send (xr2); output->Send (yb2);
		output->Send (xl2); output->Send (yb2);
		
		output->Send (xl1); output->Send (yt1);
		output->Send (xr1); output->Send (yt1);
		output->Send (xr2); output->Send (yt2);
		output->Send (xl2); output->Send (yt2);
		
		output->Send (xl1); output->Send (yb1);
		output->Send (xl1); output->Send (yt1);
		output->Send (xl2); output->Send (yt2);
		output->Send (xl2); output->Send (yb2);
		
		output->Send (xr1); output->Send (yb1);
		output->Send (xr1); output->Send (yt1);
		output->Send (xr2); output->Send (yt2);
		output->Send (xr2); output->Send (yb2);
		
		output->Send (PS_MISC_LINE);
		
	} else if (this->pen[pen].flags & PEN_RECT) {
		
		if ((x2 == x1) || (y2 == y1) || (!page_mode)) {
			output->Send (this->pen[pen].width);
			output->Send (this->pen[pen].height);
			output->Send (x2); output->Send (y2);
			output->Send (x1); output->Send (y1);
			output->Send (PS_RECT_LINE);
			return;
		}
		
		long xl1, xl2, xr1, xr2;
		long yt1, yt2, yb1, yb2;
		
		long w1 = this->pen[pen].width >> 1;
		long w2 = this->pen[pen].width - w1;
		long h1 = this->pen[pen].height >> 1;
		long h2 = this->pen[pen].height - h1;
		
		xl1 = x1 - w1;	  yb1 = y1 - h1;
		xr1 = x1 + w2;	  yt1 = y1 + h2;
		xl2 = x2 - w1;	  yb2 = y2 - h1;
		xr2 = x2 + w2;	  yt2 = y2 + h2;
		
		output->Send (xl1); output->Send (yb1);
		output->Send (xr1); output->Send (yt1);
		output->Send (xl2); output->Send (yb2);
		output->Send (xr2); output->Send (yt2);
		
		output->Send (xl1); output->Send (yb1);
		output->Send (xr1); output->Send (yb1);
		output->Send (xr2); output->Send (yb2);
		output->Send (xl2); output->Send (yb2);
		
		output->Send (xl1); output->Send (yt1);
		output->Send (xr1); output->Send (yt1);
		output->Send (xr2); output->Send (yt2);
		output->Send (xl2); output->Send (yt2);
		
		output->Send (xl1); output->Send (yb1);
		output->Send (xl1); output->Send (yt1);
		output->Send (xl2); output->Send (yt2);
		output->Send (xl2); output->Send (yb2);
		
		output->Send (xr1); output->Send (yb1);
		output->Send (xr1); output->Send (yt1);
		output->Send (xr2); output->Send (yt2);
		output->Send (xr2); output->Send (yb2);
		
		output->Send (PS_MISC_LINE);
		
	} else {
		abort ();				// condition impossible !
	}
}



// Slectionne un traitill. Retourne TRUE si le traitill est gr
// par l'imprimante, FALSE si il doit tre mul  la main.

Bool
PagePS::SelectDash (int pen, Bool hv)
{
	PenFlags flags = this->pen[pen].flags;
	int xpen = 0;
	
	if (hv) xpen = 256;
	
	if ( (flags & PEN_ELLIPSE) || (flags & PEN_RECT) || (flags & PEN_CONT)
	  || (this->page_mode && (hv == FALSE)) ) {
		
		// On ne peut pas utiliser les traitills dans le cas de
		// pinceaux non symtriques, dans le cas de traits continus
		// et dans le cas de traits " la PAGE".
		
		if (!null_dash)
			output->Write ("cd ");
		
		last_dash_pen = pen;
		null_dash = TRUE;
		
		return (flags & PEN_CONT) == PEN_CONT;
	}
	
	// Traitill " la PostScript", aussi pour PAGE si trait horizontal
	// ou vertical.
	
	if (((pen + xpen) != last_dash_pen) || (last_dash_pos != dash_pos)) {
		
		last_dash_pen = pen + xpen;
		last_dash_pos = dash_pos;
		null_dash = FALSE;
		
		output->Write ("[");
		
		for (int i = 0; i < 8; i += 2) {
			
			int gap_drawn = this->pen[pen].gap[i];
			int gap_empty = this->pen[pen].gap[i+1];
			
			if (gap_empty <= 0) break;
			
			output->Send (gap_drawn ? : 1);
			
			if (gap_drawn == 0)
				if (--gap_empty < 0)
					gap_empty = 0;
			
			output->Send (gap_empty);
		}
		
		output->Write ("]");
		output->Send (dash_pos);
		output->Write ("sd ");
	}
	
	return TRUE;
}

