/*
 *		page-struct.h
 *
 *		Dfinition des symboles et des structures utilises
 *		dans PAGE pour dfinir les pinceaux, trames, couleurs
 *		et traits.
 *
 *		(C) 1993-1996,	Pierre ARNAUD, OPaC, CH-1437 SUSCEVAZ
 */

#ifndef _PAGE_STRUCT_H_
#define _PAGE_STRUCT_H_

typedef unsigned char  Bool;
#define FALSE		   ((Bool)(0))
#define TRUE		   ((Bool)(1))

typedef unsigned char  Card8;
typedef unsigned short Card16;
typedef unsigned long  Card32;
typedef signed char	   Int8;
typedef signed short   Int16;
typedef signed long	   Int32;

static inline Bool Even (int x)			{ return !(x & 0x01); }
static inline Bool Odd (int x)			{ return (x & 0x01); }

extern Bool InitLibrary ();
extern void KillLibrary ();

struct SuperGlyph;

extern "C" {
	int _xfos_create (const char*, void*, long*);
	int _xfos_open (const char*, void*, long*);
	int _xfos_close (void*);
	int _xfos_rdbyte (void*, long*, void*);
	int _xfos_wrbyte (void*, long*, const void*);
}

#define			MAXPEN			40
#define			MAXTRAME		40
#define			MAXCOLOR		40
#define			MAXCAR			16

// Dfinitions ayant trait aux pinceaux

typedef Card32 PenFlags;

#define PEN_CIRCLE		1
#define PEN_ELLIPSE		2
#define PEN_SQUARE		4
#define PEN_RECT		8
#define PEN_CONT		256
#define PEN_DASH		512


struct PenDesc {
	PenFlags	flags;
	short		height;
	short		width;
	short		gap[8];					// trait - trou - .. - trait - trou
	short		color;
	char		reserved[6];
};


// Dfinitions ayant trait aux trames

enum TrameFlags {
	TRAME_GREY			= 0x00000001,
	TRAME_HACH			= 0x00000002,
	TRAME_DEGLIN		= 0x00000004,	// dgrad linaire
	TRAME_DEGCIR		= 0x00000008,	// dgrad circulaire
	TRAME_DEGELL		= 0x00000010,	// dgrad elliptique
	TRAME_DEGCON		= 0x00000020,	// dgrad cnique
	TRAME_SMLSQR		= 0x00000100,
	TRAME_BIGSQR		= 0x00000200,
	TRAME_BIGCIR		= 0x00000400,
	TRAME_QUINC			= 0x00008000,
	TRAME_TRANSP		= 0x00010000,
	TRAME_BIDIR			= 0x00020000,	// dgrad bidirectionnel : aller-retour
	TRAME_2ND			= 0x01000000,
	TRAME_2NDHOR		= 0x02000000,
	TRAME_2NDOBL		= 0x04000000,
	TRAME_2NDVER		= 0x08000000,
	TRAME_2NDITA		= 0x10000000
};

struct TrameDesc {
	TrameFlags	flags;
	Card16		grey_percent;
	short		dot_size;
	short		angle;
	short		dist;
	short		width_1;
	short		width_2;
	short		color_set;
	short		color_clr;
	char		reserved[4];
	
	// Extension pour les dgrads
	
	short		deg_start;
	short		deg_end;
	short		deg_angle;
	short		deg_cy, deg_cx;
	char		deg_pt_size;
	char		deg_style;				// style de la trame DGTRAME*
	char		deg_reserved[8];
};

#define DGTRAMEBIG		0				// gros point
#define DGTRAMELITTLE	1				// trame de petits points
#define DGTRAMEHLINE	2				// ligne horizontale
#define DGTRAMEVLINE	3				// ligne verticale
#define DGTRAMETRIAN	4				// triangles



// Dfinitions ayant trait aux couleurs

struct ColorDesc {
	char		name[16];
	Card16		red;
	Card16		green;
	Card16		blue;
	Card16		black;
	char		reserved[8];
};


/*
 *		Dfinitions pour la classe de gnration de
 *		code PostScript Level 1 ou Level 2.
 */

enum PSops {
	PS_ROUND_LINE,						// ligne avec bouts arrondis
	PS_ELLIP_LINE,						// ligne avec bouts elliptiques
	PS_SQUARE_LINE,						// ligne avec bouts carrs
	PS_RECT_LINE,						// ligne avec bouts rectangulaires
	PS_MISC_LINE,						// ligne quelconque dfinie par 4 rgions
	
	PS_ROUND_BEZIER,					// courbe de Bzier avec bouts arrondis
	PS_ELLIP_BEZIER,					// courbe de Bzier avec bouts elliptiques
	PS_SQUARE_BEZIER,					// courbe de Bzier avec bouts carrs
	PS_RECT_BEZIER,						// courbe de Bzier avec bouts rectangulaires
	
	PS_POLYNEW,							// polygone: dbut de dfinition
	PS_POLYLINE,						// polygone: rajoute une ligne
	PS_POLYBEZIER,						// polygone: rajoute une courbe de Bzier
	PS_POLYFILL							// polygone: remplit
};



// Classe pour crire dans un fichier level 1 ou level 2 (peu
// importe  ce niveau).

class Output
{
	Bool		is_color;
	Bool		do_close;
	void*		file;
	float		or,og,ob;
	
	char		show_buffer[256];
	int			show_len;
	int			show_ox;
	int			show_oy;
	int			show_dx;
	float		show_w;
	int			show_num;
	
protected:
	char		buffer[256];
	
	void RealEmitShow ();
	
public:
	Output (const char* name);
	Output (long channel);
	virtual ~Output ();
	
	virtual Bool Ok () { return (file != 0); }
	
	virtual int Level () = 0;
	virtual void Send (int) = 0;
	virtual void Send (float) = 0;
	virtual void Send (double x) { this->Send ((float)(x)); }
	virtual void Send (PSops) = 0;
	virtual void Send (const char*) = 0;
	virtual void Send (const void*, int, Bool) = 0;
	
	virtual void Color (float r, float g, float b);
	virtual void Show (const char*, int x, int y, float w, int dx);
	virtual void ResetColor ();
	
	void EmitShow ()			{ if (this->show_len) this->RealEmitShow (); }
	void UseColor ()			{ this->is_color = TRUE; }
	Bool IsColorUsed () const	{ return this->is_color; }
	
	Bool Write (const void* data, int size);
	Bool Write (const char* string);
};


class OutputLevel1 : public Output {
	
public:
	OutputLevel1 (const char* name) : Output (name) {}
	OutputLevel1 (long channel) : Output (channel) {}
	virtual int Level () { return 1; }
	virtual void Send (int);
	virtual void Send (float);
	virtual void Send (PSops);
	virtual void Send (const char*);
	virtual void Send (const void*, int, Bool);
};


class OutputLevel2 : public Output {
	
public:
	OutputLevel2 (const char* name) : Output (name) {}
	OutputLevel2 (long channel) : Output (channel) {}
	virtual int Level () { return 2; }
	virtual void Send (int);
	virtual void Send (float);
	virtual void Send (PSops);
	virtual void Send (const char*);
	virtual void Send (const void*, int, Bool);
};


/*
 *		Un `glyph' reprsente une petite image monochrome
 *		qui sert gnralement  reprsenter des caractres.
 */

struct Glyph {
	
	Glyph*		next;			// lien avec le glyph suivant
	
	int			dx, dy;			// dimension en points
	int			width;			// largeur en bytes
	
	long		id1, id2;		// identificateurs uniques
	long		count;			// compteur d'utilisations
	void*		data;			// copie des donnes
	
	Glyph ();
	~Glyph ();
	
	friend int operator == (const Glyph& a, const Glyph& b);
};




/*
 *		Un enregistrement `GlyphInst' permet de positionner un
 *		glyph au sein d'une page et de dfinir sa couleur.
 */

struct GlyphInst {
	
	GlyphInst*	next;			// lien avec le glyph suivant dans la page
	Glyph*		glyph;			// lien avec le glyph  reprsenter
	
	int			x, y;			// position dans la page
	short		color;			// couleur du glyph
	
	GlyphInst ();
	~GlyphInst ();
};


/*
 *		Classe grant le cache pour les glyphs.
 */

struct Cache {
	
	GlyphInst*	first_inst;		// queue des instances dans la page
	Glyph*		first_cache;	// queue des glyphs dans le cache
	
	Cache () { first_inst = 0; first_cache = 0; }
	~Cache () { delete first_inst; delete first_cache; }
	
	void Purge ();
	void Store (const Glyph& glyph, int x, int y, int color, const void* data);
};


/*
 *		Dfinition de la classe faisant office de module de
 *		librairie PAGE --> PostScript.
 */

class PagePS {
	
	friend static Bool ServerLoop ();
	
	struct Dash {
		short	pos;
		short	num;
		short	len;
		short	pen;
		short	gap[8];
	};
	
	struct BBox {
		short	x1,y1,x2,y2;
		Bool	empty;
		BBox () { empty = TRUE; x1=y1=x2=y2 = 0; }
		void Add (int x, int y)
			{
				if (empty) {
					x1 = x2 = x;
					y1 = y2 = y;
					empty = FALSE;
				} else {
					x1 = x1 <? x;
					y1 = y1 <? y;
					x2 = x2 >? x;
					y2 = y2 >? y;
				}
			}
	};
	
	Bool		failure : 8;			// chec si TRUE
	Bool		is_color : 8;			// TRUE => level 1 couleur
	Bool		is_no_multi : 8;		// TRUE => pas plus d'une page logique par feuille physique
	Bool		is_multi_ok : 8;		// TRUE => profit de mettre plusieurs pages par feuille
	
	int			eo_page_x;				// changement  chaque page paire/impaire
	int			eo_page_y;				// changement  chaque page paire/impaire
	
	int			page_count;				// compteur de pages
	int			real_page_count;		// compteur de pages physiques (feuilles)
	int			page_setup;				// compteur de pages pour SetupPage
	int			page_start;				// compteur de pages pour StartPage
	int			def_num;				// nombre de pages par dfaut
	Bool		start_new_page;			// TRUE => dmarre une page physique
	
	BBox		polybox;				// bounding box du polygone
	Dash		dash;					// tat courant du trait
	
	PenDesc		pen[MAXPEN];			// pinceaux
	TrameDesc	trame[MAXTRAME];		// trames
	ColorDesc	color[MAXCOLOR];		// couleurs
	
	char		buffer[256];			// tampon
	
	Cache		cache;					// cache pour les glyphs
	Output*		output;					// lien avec le dispositif de sortie
	int			dpi_x, dpi_y;
	int			page_dx, page_dy;
	
	Bool		stroke_adjust;
	Bool		page_mode;
	Bool		new_glyph_mode;
	int			last_dash_pen;
	Bool		null_dash;
	double		dash_pos;
	double		last_dash_pos;
	
	//	Champs dfinissables par PPS_INITPAGE :
	
	int			screen_mul;
	float		screen_angle;
	float		scale_page_x;
	float		scale_page_y;
	Bool		turn_page;
	
	//	Champs rservs  PPS_DRGLYPH
	
	Bool		glyph_font_ok;
	void*		glyph_font_cache;
	
public:
	int			id;
	int			count;
	
protected:
	void DrawLinePiece (int, int, int, int, int);
	void SendGlyphs ();
	
public:
	PagePS ();
	~PagePS ();
	
	Bool SelectDash (int num, Bool hv);
	
	void SetResolution (int x, int y) { dpi_x = x; dpi_y = y; }
	void SetPageSize (int x, int y) { page_dx = x; page_dy = y; }
	void DefaultNumPage (int num) { def_num = num; }
	void SetNoMulti () { is_no_multi = TRUE; }
	
	void InitPenDesc (const PenDesc* desc, int i);
	void InitTrameDesc (const TrameDesc* desc, int i);
	void InitColorDesc (const ColorDesc* desc, int i);
	
	void SetOutput (Output* o) { output = o; }
	Output* GetOutput () { return output; }
	Bool HadFailure () { return failure; }
	
	void NewDocument (const char* doc, const char* author, const char* for_whom, const char* date,
					  const char* for_dest, const char* epsf, const char* rules);
	void NewPage (int num);
	void SetupPage ();
	void StartPage ();
	void PreparePage () { if (page_count != page_start) this->StartPage (); }
	
	void DrawLine (int, int, int, int, int pen, Bool reset = FALSE);
	void DrawBezier (int, int, int, int, int, int, int, int, int pen, Bool reset = FALSE);
	
	void PolyOpen ();
	void PolyLine (int, int, int, int);
	void PolyBezier (int, int, int, int, int, int, int, int);
	void PolyClose (int, int, int, int, int);
	
	void DrawGlyph (int, int, int, int, int, int, long, long, const void*, const void* glyph);
	void DrawMonoImage (int, int, int, int, int, int, float, int, int, const void*);
	void DrawColorImage (int, int, int, int, int, int, float, int, int, const void*, const void*);
	
	Bool DrawFontGlyph (const SuperGlyph*, int color);
	void DrawHugeGlyph (int, int, int, int, int, int, long, long, const void*);
	
	void KillFontCache ();
	void PurgeFontCache ();
	
	void IsColor (Bool x) { is_color = x; }
	Bool IsColor () const { return is_color; }
	
	void NewGlyphMode (Bool x) { new_glyph_mode = x; }
	Bool NewGlyphMode () const { return new_glyph_mode; }
	void PageMode (Bool x);
	Bool PageMode () const { return page_mode; }
	
	void StrokeAdjust (Bool x) { stroke_adjust = x; }
	Bool StrokeAdjust () const { return stroke_adjust; }
	
	void ScreenParam (int m, float a) { screen_mul = m; screen_angle = a; }
	void TurnPage (Bool x) { turn_page = x; }
	void ScalePage (float x, float y) { scale_page_x = x; scale_page_y = y; }
};

#endif

