/*
 *		ppm.h
 *
 *		Dfinition des structures utilises lors des dialogues avec le
 *		serveur d'impression PostScript (PostScript Printer Manager).
 *
 *		(C) 1993-1995,  Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#ifndef _PPM_H_
#define _PPM_H_

#include "ppd.h"


/*
 *	Types exports par le "PostScript Printer Manager" (appel plus
 *	simplement PPM).
 */

struct PPM_Res;

struct PPM_Job;
struct PPM_Status;
struct PPM_Message;
struct PPM_Error;
struct PPM_Output;
struct PPM_Report;
struct PPM_Answer;

struct PPM_GetPPD;
struct PPM_OpenConn;
struct PPM_CloseConn;
struct PPM_RemoteConn;
struct PPM_PrnName;

struct PPM_Request;


/*
 *	Dfinition de ressources sonores, textuelles et visuelles
 *	pour dcrire de manire claire un message provenant depuis
 *	l'imprimante.
 */

struct PPM_Res {
	Card32				key;			/* clef d'accs aux ressources							*/
	Card32				audio;			/* ressource audio associe (ou -1)						*/
	Card32				text;			/* ressource texte associe (ou -1)						*/
	Card32				image;			/* ressource image associe (ou -1)						*/
};


/*
 *	Enumration des requtes possibles pour le systme
 *	d'impression PostScript.
 */

typedef Card32	PPM_RequestEnum;

#define	PPM_RQ_LIST_PRN		0			/* liste les imprimantes								*/
#define	PPM_RQ_RENAME_PRN	1			/* renomme une imprimante								*/
	
#define	PPM_RQ_INFO_PRN		2			/* retourne des informations sur une imprimante			*/
	
#define	PPM_RQ_LIST_DPIS	3			/* liste les rsolutions								*/
#define	PPM_RQ_LIST_TRAYS	4			/* liste les bacs d'entre								*/
#define	PPM_RQ_LIST_BINS	5			/* liste les bacs de sortie								*/
#define	PPM_RQ_LIST_FONTS	6			/* liste les fontes										*/
	
#define	PPM_RQ_GET_PPD		7			/* retourne code PostScript depuis base de donnes PPD	*/
	
#define	PPM_RQ_DEF_TRAY		8			/* dfinit un bac d'entre								*/
#define	PPM_RQ_DEF_BIN		9			/* dfinit un bac de sortie								*/
	
#define	PPM_RQ_OPEN_CONN	10			/* ouvre une connexion									*/
#define	PPM_RQ_CLOSE_CONN	11			/* ferme une connexion									*/
#define	PPM_RQ_PRINT		12			/* imprime un fichier									*/



/*
 *	Enumration des rponses possibles depuis le serveur
 *	d'impression PostScript.
 */

typedef Card32	PPM_AnswerEnum;

#define	PPM_ANS_STARTING_JOB 0			/* signale dbut d'un job								*/
#define	PPM_ANS_ENDING_JOB	1			/* signale fin d'un job									*/
#define	PPM_ANS_STATUS		2			/* statut : "idle", "busy", "waiting", ...				*/
#define	PPM_ANS_MESSAGE		3			/* message : "exitserver: permanent [..] changed", ...	*/
#define	PPM_ANS_ERROR		4			/* erreur												*/
#define	PPM_ANS_OUTPUT		5			/* donnes retournes par l'imprimante					*/
#define	PPM_ANS_REPORT		6			/* rponse du systme d'impression						*/



/*
 *	Enumration des types de "jobs" reconnus.
 */

typedef	Card32	PPM_JobEnum;

#define	PPM_JOB_DOCUMENT	1			/* document												*/
#define	PPM_JOB_QUERY		2			/* interrogation										*/
#define	PPM_JOB_INTERACTIVE 3			/* dialogue interactif									*/
#define	PPM_JOB_S_PRN_LIST	4			/* Smaky: liste les imprimantes							*/



/*
 *	Enumration des statuts reconnus. Ces messages sont retourns
 *	 intervalles rguliers par le systme d'impression.
 */

typedef Card32	PPM_StatusEnum;

#define	PPM_STATUS_IDLE		0			/* libre												*/
#define	PPM_STATUS_BUSY		1			/* occupe												*/
#define	PPM_STATUS_WAIT		2			/* en attente											*/
#define	PPM_STATUS_PRINT	3			/* en cours d'impression								*/
#define	PPM_STATUS_INIT		4			/* en cours d'initialisation							*/
#define	PPM_STATUS_WARMING	5			/* en cours d'chauffement								*/
#define	PPM_STATUS_ERROR	6			/* erreur												*/
#define	PPM_STATUS_USERINT	7			/* intervention de l'utilisateur ncessaire				*/
#define	PPM_STATUS_MISC		8			/* divers (statut non reconnu)							*/



/*
 *	Description d'un job (qui en est l'auteur,  qui le rsultat
 *	est destin et comment il s'appelle).
 */

struct PPM_Job {
	
	PPM_JobEnum			job;			/* nature du travail									*/
	
	char				for_whom[64];	/* nom de l'utilisateur associ au document				*/
	char				author[64];		/* logiciel auteur du document							*/
	char				title[64];		/* titre du document									*/
	char				routing[64];	/* nom de la station initiatrice						*/
	char				printer[128];	/* nom complet de l'imprimante concerne				*/
	char				docdata[64];	/* nom du type de donnes (Binary...)					*/
};


/*
 *	Description d'un statut d'imprimante (nom du job associ,
 *	information et source des donnes).
 */

struct PPM_Status {
	
	PPM_StatusEnum		status;			/* type													*/
	PPM_Res				resource;		/* ressources associes									*/
	
	char				job[64];		/* nom du travail en cours								*/
	char				message[48];	/* message reu de l'imprimante							*/
	char				source[16];		/* source du document									*/
	char				args[128];		/* arguments associs (pour complter les ressources)	*/
};


/*
 *	Description d'un message d'imprimante. Un message est une
 *	erreur non fatale et correspond  un "warning".
 */

struct PPM_Message {
	
	PPM_Res				resource;		/* ressources associes									*/
	
	char				message[64];	/* message reu de l'imprimante							*/
	char				args[128];		/* arguments associs (pour complter les ressources)	*/
};


/*
 *	Description d'une erreur d'imprimante.
 */

struct PPM_Error {
	
	PPM_Res				resource;		/* ressources associes									*/
	
	char				message[64];	/* message reu de l'imprimante							*/
	char				args[128];		/* arguments associs (pour complter les ressources)	*/
};


/*
 *	Description de donnes en provenance d'une imprimante. S'il y a
 *	plus de donnes qu'il n'y a de place, elles seront fragmentes.
 */

struct PPM_Output {
	
	Card16				size;			/* taille du message dans ce bloc						*/
	Bool				more;			/* TRUE => message incomplet							*/
	Card8				dummy;
	
	char				output[512];	/* message (ou fragment de message) pas termin par 0	*/
};


/*
 *	Description d'une rponse du gestionnaire d'impression. Les
 *	donnes doivent tre interprtes par l'appelant.
 */

struct PPM_Report {
	
	PPM_RequestEnum		request;		/* requte correspondante								*/
	
	Card16				data_size;		/* taille des donnes dans ce bloc						*/
	Bool				more;			/* TRUE => rponse incomplte							*/
	Card8				dummy;

	Card16				record_num;		/* nombre de records total dans cette rponse			*/
	Card16				record_size;	/* taille d'un record									*/
	
	char				data[512];		/* donnes												*/
};


/*
 *	Description d'une rponse gnrique.
 */

struct PPM_Answer {
	
	PPM_AnswerEnum		what;			/* nature de la rponse									*/
	
	Card32				num;			/* numro de rgnration du job (ou 0)					*/
	Card32				id;				/* identificateur de la requte							*/
	void*				answer_gate;	/* BAR pour mettre la rponse							*/
	Bool				free_gate;		/* TRUE => la BAR devient  nouveau libre				*/
	Card8				dummy;
	
	union {
		PPM_Job			job;			/* description du travail (dbut/fin)					*/
		PPM_Status		status;			/* statut retourn par l'imprimante						*/
		PPM_Message		message;		/* message de l'imprimante								*/
		PPM_Error		error;			/* erreur retourne par l'imprimante					*/
		PPM_Output		output;			/* valeurs retournes par l'imprimante					*/
		PPM_Report		report;			/* informations retournes par le print manager			*/
	} answer;
};


/*
 *	Description d'une requte d'interrogation de la base de donnes
 *	PPD (PostScript Printer Description).
 */

struct PPM_GetPPD {
	
	Card16				num;			/* nombre de requtes									*/
	char				ppd[512];		/* buffer pour code des interrogations (sp. par <0>)	*/
};


/*
 *	Description d'une connexion. Cette structure ne peut tre utilise
 *	que localement. La structure distante "PPM_RemoteConn" ne doit jamais
 *	tre utilise directement par un client.
 */

struct PPM_OpenConn {
	
	const char*			author;			/* nom du logiciel auteur de la connexion				*/
	Card32				pid;			/* numro de rgnration du processus (*)				*/
};

/*
 *	(*) REMARQUE :		Il est possible de donner un pid nul, auquel cas tous les
 *						messages en provenance de l'imprimante seront communiqus.
 */


/*
 *	Ferme une connexion. La connexion est identifie par son gate d'accs.
 */

struct PPM_CloseConn {
	
	void*				answer_gate;	/* gate utilis pour la connexion						*/
};


/*
 *	Description d'une connexion distante. Rserv pour le soft interne.
 */

struct PPM_RemoteConn {
	
	char				author[64];		/* nom du logiciel + pid + machine id					*/
};


/*
 *	Nom de l'imprimante + type de l'imprimante + nom du rseau.
 */

struct PPM_PrnName {
	
	char				name[32];		/* nom (par ex. "Foudre de Guerre")						*/
	char				type[32];		/* type (par ex. "microLaser9 Turbo")					*/
	char				zone[32];		/* zone (par ex. "Salle de classe")						*/
	char				comm[64];		/* commentaire											*/
};



/*
 *	Description d'une requte pour le systme d'impression.
 */

struct PPM_Request {
	
	PPM_RequestEnum		what;			/* nature de la requte									*/
	
	Card32				machine_id;		/* identificateur de la machine					==>		*/
	Card32				id;				/* identificateur de la requte					==>		*/
	void*				answer_gate;	/* BAR pour attendre la rponse					==>		*/
	Card32				error;			/* erreur en retour								==>		*/
	char				routing[64];	/* nom de la machine initiatrice				==>		*/
	
	PPM_PrnName			name;			/* nom de l'imprimante									*/
	
	union {
		PPM_PrnName		name;			/* nom pour renommer									*/
		PPD_InTray		tray;			/* dfinition du bac  papier en entre					*/
		PPD_OutBin		bin;			/* dfinition du bac  papier en sortie					*/
		PPM_GetPPD		get_ppd;		/* interrogation de la base de donnes PPD				*/
		PPM_OpenConn	open_conn;		/* ouverture d'une connexion localement					*/
		PPM_CloseConn	close_conn;		/* fermeture d'une connexion							*/
		PPM_RemoteConn	remote_conn;	/* ouverture d'une connexion distante (ne pas utiliser) */
		char			print[256];		/* imprime un fichier									*/
	} request;
};



/*
 *		L'mission de requtes se fait toujours sur la BAR nomme "#:PPMrequests".
 *		Le programme SMA_PPMCLI.CODE envoie celles-ci  travers le rseau vers le
 *		serveur d'impression.
 *
 *		Les rponses parviennent aussi par le rseau et sont ensuite redistribues
 *		par SMA_PPMCLI.CODE vers les diffrentes BAR de rception.
 *
 *		Il y a au maximum 8 BARs de rception disponibles simultanment ( cause
 *		de limitations dans le NTR). Il est impratif de fermer toute connexion
 *		ayant t ouverte avec succs, sans quoi le nombre de BARs sera rduit.
 *
 *		Impression
 *		----------
 *
 *		Le logiciel imprime le document en passant par PRINTA (ou n'importe quel
 *		autre mcanisme) et produit par consquent un fichier PostScript qui va
 *		tre stock dans un dossier du serveur d'impression.
 *
 *		Au moment o le fichier est imprim, un certain nombre de messages sont
 *		produits par l'imprimante et par le systme d'impression. Ceux-ci ne sont
 *		dlivrs par rseau que si le logiciel ouvre une connexion PPM.
 *
 *		A la fin de l'impression, il faut fermer la connexion PPM.
 *
 *
 *		Requtes
 *		--------
 *
 *		Les requtes sont envoyes par le rseau au serveur PPM qui y rpond le
 *		plus rapidement possible. La BAR reue ne peut tre utilise que le temps
 *		de lire la rponse. Elle redevient disponible pour d'autres usages aprs.
 */

#endif

