/*
 *	ppmatalk.cc
 *
 *	Interface du PostScript Printer Manager avec le rseau Appletalk (ou le
 *	Prunetalk du Smaky), l'impression sur Centronics, ligne srie, etc.
 *
 *	(C)	Copyright 1993-2000, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 SUSCEVAZ
 */

/*
 *	Les routines ppm_.. ne peuvent pas tre accdes depuis
 *	l'extrieur du module. Elles constituent l'interface rel
 *	entre le serveur Prunetalk (crit en MEX et tournant comme
 *	processus spar) et les fonctions haut niveau du PPM.
 *
 *	Toutes les routines sont prvues pour du mono-tche. Il est
 *	donc exclu de vouloir faire excuter ce code  2 processus.
 *		
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

extern int errno;

#include <c++/fos.h>
#include <c++/ntrel.h>

#include "misc.h"
#include "ppm.h"
#include "ppm_atalk.h"
#include "printcon.h"

static const BUF_SIZE			= 1024;					// taille du buffer
static const MY_FLOW_QUANTUM	= 8;					// lit et crit 4096 bytes au maximum
static const SMALL_DELTA_T		= 5;					// .1 seconde

Bool  no_gs		  = FALSE;
Bool  no_message  = FALSE;
Bool  use_ctrl_D  = TRUE;
Bool  force_040A  = FALSE;

extern void send_clock_message (const char* text, const char* network, const char* title);
extern void ppm_loop_printing (int& open);


Card32 PPMan::num_documents = 0;
Card32 PPMan::num_queries   = 0;
Card32 PPMan::num_others    = 0;
Card32 PPMan::job_file_num  = 0;


/*
 *	Initialise le PostScript Printer Manager.
 */

PPMan::PPMan ()
{
	this->count = -1;
}


/*
 *	Dtruit le PostScript Printer Manager.
 */

PPMan::~PPMan ()
{
}


/*
 *	Cherche l'imprimante correspondant  la description
 *	textuelle (nom, type, zone). Retourne un pointeur sur
 *	le record de dfinition ou 0 en cas d'erreur.
 */

NBPElem*
PPMan::FindPrinter (const PPM_PrnName* name)
{
	Int32 index = this->count;
	
	while (index-- > 0) {
		if ( (strcmp (name->name, this->list[index].object) == 0)
		  && (strcmp (name->type, this->list[index].type) == 0)
		  && (strcmp (name->zone, this->list[index].zone) == 0) ) {
			return & this->list[index];
		}
	}
	
	//	Si l'imprimante ne peut pas tre trouve dans la table locale,
	//	cela signifie qu'elle n'tait pas connue  la dernire recherche.
	
	return 0;
}


/*
 *	Vrifie si une adresse sur le rseau est toujours valable.
 *	Dans le cas contraire, il faudra redemander la liste.
 *	Retourne TRUE si l'adresse est valable.
 */

Bool
PPMan::CheckPrinter (const NBPElem* elem)
{
	NBPCommand cmd;
	NtrBar*    gate   = ppm_init_atalk ();
	void*      answer = 0;
	
	if (gate == 0) {
		return FALSE;
	}
	
	cmd.command       = NBP_CONFIRM_NAME;
	cmd.timeout_essai = 0;
	cmd.name          = elem[0];
	
	gate->Offer ();
	gate->EndOffer (& cmd, answer);
	
	//	Retourne TRUE si l'imprimante a t trouve et qu'elle a rpondu
	//	prsent.
	
	return (cmd.error) ? FALSE : TRUE;
}


/*
 *	Met  jour la liste des imprimantes PostScript disponibles sur
 *	le rseau AppleTalk. Cette opration ne devrait pas tre faite
 *	trop souvent, car elle prend envrion 4 secondes.
 */

void
PPMan::UpdatePrinterList ()
{
	NBPCommand cmd;
	NtrBar*    gate = ppm_init_atalk ();
	void*      answer = 0;
	
	report_msg (REPORT_INFO, "Liste les imprimantes");
	
	cmd.command       = NBP_LOOK_UP;
	cmd.list          = & this->list[0];
	cmd.nb_max_rep    = NBP_LISTELEM_NUM;
	cmd.nb_rempli     = 0;
	cmd.nb_essais     = 0;
	cmd.timeout_essai = 0;
	
	strcpy (cmd.name.object, "=");					// liste tout le monde
	strcpy (cmd.name.type, "LaserWriter");			// pourvu que ce soient des PostScript printers
	strcpy (cmd.name.zone, "*");					// sur notre rseau local AppleTalk
	
	Ntr::SetTim (50*10);
	
	if ( (gate == 0)
	  || (gate->Offer ())
	  || (gate->EndOffer (& cmd, answer)) ) {
		
		//	AppleTalk pas install correctement, imprimante absente ou autre
		//	cause mystrieuse...
		
		this->count = -1;
		
	} else {
		
		//	On sait combien d'imprimantes ont rpondu prsent  l'appel.
		
		this->count = cmd.nb_rempli;
	}
	
	Ntr::SetTim (0xFFFF);
	
	if (this->count >= 0) {
		char buffer[200];
		sprintf (buffer, "Trouv %d imprimantes", this->count);
		report_msg (REPORT_INFO, buffer);
	}
}



/*
 *	Retourne le nombre d'imprimantes connectes  Appletalk sur
 *	la zone courante. Refait la liste si ncessaire.
 */

int
PPMan::GetNumPrinter ()
{
	if (this->count == -1) {
		this->UpdatePrinterList ();
	}
	
	return this->count;
}


/*
 *	Retourne le nom d'une imprimante dfinie par son rang dans la
 *	liste des imprimantes. Retourne TRUE en cas de succs.
 */

Bool
PPMan::GetPrinterInfo (int num, PPM_PrnName* name)
{
	if (num > this->GetNumPrinter ()) {
		return FALSE;
	}
	
	strcpy (name->name, this->list[num].object);
	strcpy (name->type, this->list[num].type);
	strcpy (name->zone, this->list[num].zone);
	
	return TRUE;
}



/*
 *	Regarde si l'impression a t interrompue.
 */

Bool
PPMan::IsStopRequested ()
{
	return (getenv ("PSTALK_ABORT")) ? TRUE : FALSE;
}


/*
 *	Encode un ensemble de 4 bytes en 5 caractres ASCII au moyen du
 *	filtre Ascii85Encode.
 */

inline int
Ascii85Encode (Card32 bin, unsigned char* asc)
{
	if (bin == 0) {
		asc[0] = 'z';
		asc[1] = 0;
		return 1;
	}
	
	asc[0] = bin / (85*85*85*85); bin -= asc[0] * 85*85*85*85;
	asc[1] = bin / (85*85*85);	  bin -= asc[1] * 85*85*85;
	asc[2] = bin / (85*85);		  bin -= asc[2] * 85*85;
	asc[3] = bin / (85);		  bin -= asc[3] * 85;
	asc[4] = bin;
	
	asc[0] += '!';
	asc[1] += '!';
	asc[2] += '!';
	asc[3] += '!';
	asc[4] += '!';
	asc[5]	= 0;
	
	return 5;
}


/*
 *	Analyse un bout de fichier PostScript pour en retirer des
 *	informations sur le type du job.
 */

Bool
PPMan::ParsePostScript (const PPM_PrnName* printer, const char* name, JobDef* job, int& error)
{
	Card32 size  = BUF_SIZE;
	Card16 canal = 0;
	char   buffer[BUF_SIZE+1] = { 0 /* ... */ };
	
	error = FOS::Open (name, FOS_OPEN_READ, canal);
	
	if (error) {
		return FALSE;
	}
	
	FOS::Read (canal, size, buffer);
	FOS::Close (canal);
	
	if ( (buffer[0] != '%')
	  || (buffer[1] != '!') ) {
		
		error = 0x0001;
		report_msg (REPORT_ERROR, "Fichier non PostScript");
		return FALSE;
	}
	
	buffer[11] = 'x';					// remplace la version par "x"
	buffer[13] = 'x';					// remplace la rvision par "x"
	
	// Remplit la description du "job" que l'on va bientt commencer.
	// On vient de s'assurer que c'est bien un fichier PostScript que
	// nous avons sous la main.
	
	job->job.job = (strncmp ("%!PS-Adobe-x.x Query", buffer, 20) == 0)
				   ? PPM_JOB_QUERY
				   : PPM_JOB_DOCUMENT;
	
	if (strncmp ("%!PS-Smaky-x.x PrnList", buffer, 22) == 0) {
		job->job.job = PPM_JOB_S_PRN_LIST;
	}
	
	switch (job->job.job) {
		
		case PPM_JOB_DOCUMENT:
			this->num_documents++;
			this->job_file_num++;
			break;
		
		case PPM_JOB_S_PRN_LIST:
		case PPM_JOB_QUERY:
			this->num_queries++;
			this->job_file_num++;
			break;
		
		default:
			this->num_others++;
			this->job_file_num++;
			break;
	}
	
	extract (job->job.for_whom, buffer, "%%For:");
	extract (job->job.author,   buffer, "%%Creator:");
	extract (job->job.title,    buffer, "%%Title:");
	extract (job->job.routing,  buffer, "%%Routing:");
	extract (job->job.docdata,  buffer, "%%DocumentData:");
	
	strcpy (job->job.printer, printer->name);
	strcat (job->job.printer, ".");
	strcat (job->job.printer, printer->type);
	strcat (job->job.printer, "@");
	strcat (job->job.printer, printer->zone);
	
	job->num          = this->job_file_num;
	job->user_id      = 0;
	job->user_machine = 0;
	job->user_gate    = 0;
	
	return TRUE;
}


/*
 *	Imprime sur un stream Centronics ou RS-232.
 */

Bool
PPMan::PrintPostScriptStream (const PPM_PrnName* printer, const char* name,
							  JobDef* job, int& error)
{
	FILE* output = 0;
	FILE* input  = 0;
	
	Card32 count = 20;
	char   pname[200];
	
	strcpy (pname, printer->zone);
	
	if (pname[strlen (pname)-1] != ':') {
		strcat (pname, ":");
	}
	
	while (use_ctrl_D && count--) {
		
		//	Essaie 20 fois d'ouvrir le port d'impression et d'envoyer avec succs
		//	un ^D... S'il ne passe pas, c'est que l'imprimante a un srieux
		//	problme, car on a attendu 2 minutes environ.
		
		Card32     size    = 8;
		Card16     canal   = 0;
		const char data[9] = { 4, 10, '%', '!', 'P', 'S', 10, 4, 0 };
		int        tmper   = 0;
		char       buffer[200];
		
		error = FOS::Open (pname, FOS_OPEN_WRITE, canal);
		
		if (error) {
			sprintf (buffer, "Ouverture de %s impossible", pname);
			report_msg (REPORT_ERROR, buffer);
			return FALSE;
		}
		
		error = FOS::Write (canal, size, data);
		tmper = FOS::Close (canal);
		
		sprintf (buffer, "Write %s = %04x:%04x", pname, error, tmper);
		report_msg (REPORT_INFO, buffer);
		
		if ( (error == 0)
		  && (tmper == 0) ) {
			break;
		}
		
		if (this->IsStopRequested ()) {
			error = 0x0001;
			return FALSE;
		}
		
		report_msg (REPORT_ERROR, "Imprimante pas prte");
		Ntr::Dels (3);
	}
	
	if (count == 0) {
		report_msg (REPORT_ERROR, "Abandonne l'impression");
		error = 0x0001;
		return FALSE;
	}
	
	const char* debug_out = getenv ("PSTALK_DBGFILE");
	
	if (debug_out) {
		output = fopen (debug_out, "wb");
		report_msg (REPORT_INFO, "Ouvert fichier de debug");
	} else {
		output = fopen (pname, "wb");
	}
	
	input  = fopen (name, "rb");
	
	if ( (input == 0)
	  || (output == 0) ) {
		
		report_msg (REPORT_ERROR, "Problme avec un fichier");
		report_msg (REPORT_ERROR, name);
		
		if (input)  { fclose (input);  }
		if (output) { fclose (output); }
		
		error      = 0x0001;
		
		return FALSE;
	}
	
	// Le cas `use_ctrl_D == FALSE' signifie en clair que l'on ne veut
	// pas que nous ne touillions le stream de sortie avec des caractres
	// de contrle. Donc, pas de CTRL-D au dbut et  la fin du job et
	// aucune conversion de type "quoting" des donnes. On n'utilise pas
	// non plus de filtre Level 2...
	
	// Si l'utilisateur a spcifi PSTALK_ASCII85, on force le filtre ASCII85
	// dans tous les cas. Cela permet de garantir qu'aucune donne binaire ne
	// sera mise.
	
	// Voila le vrai `job'  imprimer :
	
	if (getenv ("PSTALK_TURBO")) {
		
		const int bufsize = 100*1024;
		char*     buffer  = new char[bufsize];
		
		if (buffer) {
			
			int size = 0;
			int kb   = 0;
			
			for (;;) {
				size = fread (buffer, 1, bufsize, input);
				if (size == 0) break;
				fwrite (buffer, size, 1, output);
				
				char msg[200];
				sprintf (msg, "%5d KB transfrs.", kb);
				report_msg (REPORT_INFO, msg);
				kb += bufsize / 1024;
			}
			
			delete buffer;
			goto end_of_job;
		}
	}
	
	if ( ( (getenv ("PSTALK_BIN"))
	    || (strcmp (job->job.docdata, "Binary") != 0) )
	  && ( (getenv ("PSTALK_ASCII85") == 0) ) ) {
		
		if (force_040A) {
			fprintf (output, "%c%c", 0x04, 0x0a);
		}
		
		if (use_ctrl_D) {
			
			// Emet une commande pour passer en mode de communication dit
			// standard (CTRL-A est un "Quote"). Ceci n'est ncessaire que
			// sur une imprimante Level 2 : c'est gal si a choue sur les
			// autres !
			
			fprintf (output, "%%!PS-Adobe-3.0\n"
							 "%%%%Title: (Set up Binary Protocol - Level 2)\n"
							 "%%%%EndComments\n"
							 "systemdict /languagelevel known\n"
							 "{ languagelevel 2 eq } { false } ifelse\n"
							 "{ currentsystemparams\n"
							 "  /CurInputDevice 2 copy known {\n"
							 "    get <<\n"
							 "      /Protocol /Binary\n"
							 "      /Interpreter /PostScript\n"
							 "    >> setdevparams\n"
							 "  } { pop pop } ifelse\n"
							 "} if\n"
							 "%%EOF\n");
			
			fflush (output);
		}
		
		// Tient compte des squences binaires dans le cas de level 2
		// car ^C, ^D et autres ne passent pas, sinon.
		
		int c, i = 0;
		
		while ((c = fgetc (input)) != EOF) {
			switch (c) {
				case 0x01: case 0x03: case 0x04: case 0x05:
				case 0x11: case 0x13: case 0x14: case 0x1C:
					if (use_ctrl_D) {
						fputc (0x01, output);
						fputc (0x40+c, output);
						break;
					}
					//	fall through...
				
				default:
					fputc (c, output);
					break;
			}
			
			i++;
			
			if ((i & 0x7FF) == 0) {
				
				if (this->IsStopRequested ()) {
					break;
				}
				
				char buffer[200];
				sprintf (buffer, "%5d KB transfrs.", i/1024);
				report_msg (REPORT_INFO, buffer);
			}
		}
		
	} else {
		
		if (force_040A) {
			fprintf (output, "%c%c", 0x04, 0x0a);
		}
		
		fprintf (output, "%%!PS-Adobe-3.0\n");
		fprintf (output, "%%%%Title: %s (ASCII 85 encoded)\n", job->job.title);
		fprintf (output, "%%%%Creator: %s\n", job->job.author);
		fprintf (output, "%%%%For: %s\n", job->job.for_whom);
		fprintf (output, "%%%%Routing: %s\n", job->job.routing);
		fprintf (output, "%%%%EndComments\n\n");
		fprintf (output, "currentfile /ASCII85Decode filter cvx exec\n");
		fflush (output);
		
		char*  buf  = 0;
		int buflen  = 1024*100*2;
		
		while (buf == 0) {
			
			buflen /= 2;
			
			if (buflen <= 1024) {
				break;
			}
			
			buf = new char[buflen];
		}
		
		char tmp[1024];
		
		if (buf == 0) {
			buflen = 1024;
			buf    = tmp;
		}
		
		if (buflen) {
			char buffer[200];
			sprintf (buffer, "Buffer de %d KB.", buflen/1024);
			report_msg (REPORT_INFO, buffer);
		}
		
		int	   num  = 0;
		int	   cnt  = 0;
		int    len  = 0;
		Bool   stop = FALSE;
		Card32 bin  = 0;
		Card8  ascii[8];
		
		char*  ptr  = buf;
		int    free = buflen;
		
		for (;;) {
			
			bin = 0;
			len = fread (&bin, 1, 4, input);
			
			if (len < 4) { cnt = len; break; }
			
			len  = Ascii85Encode (bin, ascii);
			cnt += len;
			
			if (cnt >= 75) {
				ascii[len++] = '\n';
				cnt = 0;
			}
			
			if (len > free) {
				fwrite (buf, 1, buflen-free, output);
				ptr  = buf;
				free = buflen;
			}
			
			memcpy (ptr, ascii, len);
			
			ptr  += len;
			free -= len;
			
			if ((num++ & 0x7FF) == 0) {
				if (this->IsStopRequested ()) {
					stop = TRUE;
					break;
				}
				
				char buffer[200];
				sprintf (buffer, "%5d KB transfrs.", num/1024*4);
				report_msg (REPORT_INFO, buffer);
			}
		}
		
		if (free != buflen) {
			fwrite (buf, 1, buflen-free, output);
		}
		
		if ( (stop == FALSE) && (cnt > 0) ) {
			Ascii85Encode (bin, ascii);
			if (ascii[0] == 'z') strcpy ((char*)(ascii), "!!!!!");
			ascii[cnt+1] = 0;
			fprintf (output, "%s", ascii);
		}
		
		if (buf != tmp) {
			delete buf;
		}
		
		fprintf (output, "~>\n");
		
		//	Copy the color hint to the output (non-encoded) stream.
		
		fseek (input, -40, SEEK_END);
		char bufend[50] = { 0 };
		fread (bufend, 1, 40, input);
		ptr = strstr (bufend, "%ColorHint:");
		
		if (ptr) {
			strstr (ptr, "\n") [0] = 0;
			fprintf (output, "%s\n", ptr);
		}
		
		fprintf (output, "%%%%EOF\n");
	}
	
	if (force_040A) {
		fprintf (output, "%c%c", 0x0a, 0x04);
	}
end_of_job:
	error = 0x0000;
	errno = 0;
	fflush (output);
	fclose (output);
	fclose (input);
	
	if (errno) {
		char buffer[200];
		sprintf (buffer, "Job mal termin %04x", errno);
		report_msg (REPORT_ERROR, buffer);
	} else {
		report_msg (REPORT_INFO, "Job termin, ok.");
	}

	if (use_ctrl_D) {
		output = fopen (pname, "wb");
		fputc (4, output);				/* CTRL-D => EOT */
		fflush (output);
		fclose (output);
	}
		
	if (this->IsStopRequested ()) {
		report_msg (REPORT_ERROR, "Impression avorte");
		putenv ("PSTALK_ABORT");
		error = 0x0001;
	}
	
	FOS::Delete (name);
	
	return error ? FALSE : TRUE;
}



/*
 *	Imprime via GhostScript.
 */

Bool
PPMan::PrintGhostScript (const PPM_PrnName* printer, const char* name,
						 JobDef* job, int& error)
{
	//	Impression du fichier via GhostScript. On excute simplement
	//	le programme GhostScript en attendant qu'il se termine...
	
	static int fax_count = 0;
	
	char   buffer[400];
	Bool   call_fax = FALSE;
	
	if (getenv ("GSLOG") == 0) {
		FOS::Assign ("#GSLOG:", "#MM0:TMP:");
	}
	
	report_msg (REPORT_INFO, "Lance GhostScript");
	
	if (strcmp (printer->zone, "GS-SMAKY.PS") == 0) {
		
		//	Cas particulier: c'est une imprimante propre au SMAKY qu'il
		//	faut utiliser. On va donc utiliser les infos passes par le
		//	commentaire.
		
		const char* comment = (printer->comm[0] == '!') ? printer->comm + 1 : "";
		
		sprintf (buffer, "gs -dNOPAUSE -q -sDEVICE=smaky -sOutputFile=#PHYS_PRINT: "
						 "%s #mm0:usr:local:lib:ghostscript:smaky:%s %s -c quit "
						 "1>#gslog:gs.out 2>#gslog:gs.err",
						 comment, printer->zone, name);
		
	} else if (strncmp (printer->zone, "GS-FAX", 6) == 0) {
		
		if (getenv ("FAXSPOOL") == 0) {
			FOS::Execute ("faxsend", FOS_EX_LEAVEKEY);
			report_msg (REPORT_INFO, "Lance FaxSend");
			Ntr::Dels (5);
		}
		
		if (getenv ("FAXSPOOL") == 0) {
			report_msg (REPORT_ERROR, "FaxSpool pas initialis");
			return FALSE;
		}
		
		fax_count++;
		call_fax = TRUE;
		
		sprintf (buffer, "gs -dNOPAUSE -q "
						 "#mm0:usr:local:lib:ghostscript:smaky:%s %s -c quit "
						 "1>#gslog:gs.out 2>#gslog:gs.err",
						 printer->zone, name);
		
	} else {
		
		sprintf (buffer, "gs -dNOPAUSE -q "
						 "#mm0:usr:local:lib:ghostscript:smaky:%s %s -c quit "
						 "1>#gslog:gs.out 2>#gslog:gs.err",
						 printer->zone, name);
		
	}
	
	//	Excute GhostScript...
	
	error = (no_gs) ? 0 : FOS::Execute (buffer, FOS_EX_LEAVEKEY);
	
	FOS::Delete (name);
	
	if ( (error == 0)
	  && (call_fax) ) {
		
		sprintf (buffer, "faxspool \"%s\"", job->job.for_whom);
		error = FOS::Execute (buffer, FOS_EX_LEAVEKEY);
	}
	
	return TRUE;
}


static void
convspace (char* ptr)
{
	while (*ptr) {
		if (*ptr == ' ') *ptr = '_';
		ptr++;
	}
}


/*
 *	Imprime un fichier. Ce fichier peut tre soit un document,
 *	soit une requte. Remplit la description du "job" pour que
 *	l'appelant sache de quoi il s'agit.
 */

int
PPMan::PrintFile (const PPM_PrnName* printer, const char* name, JobDef* job)
{
	int  error = 0;
	char buffer[BUF_SIZE+1] = { 0 /* ... */ };
	
	if (this->ParsePostScript (printer, name, job, error) == FALSE) {
		return error;
	}
	
	if (job->job.job == PPM_JOB_S_PRN_LIST) {
		report_msg (REPORT_INFO, "Requte Smaky: PrnList");
		this->UpdatePrinterList ();
		int n   = this->GetNumPrinter ();
		int num = 0;
		FILE* fout = fopen ((job->job.routing[0]) ? job->job.routing : "/tmp/prnlist.info", "wt");
		if (fout) {
			fprintf (fout, "Total: %d\n\n", n);
			while (n--) {
				char name[128];
				char type[128];
				char zone[128];
				
				strcpy (name, this->list[num].object);	convspace (name);
				strcpy (type, this->list[num].type);	convspace (type);
				strcpy (zone, this->list[num].zone);	convspace (zone);
				
				fprintf (fout, "%s.%s@%s\n", name, type, zone);
				num++;
			}
			fclose (fout);
		}
		job->file = 0;
		return 0;
	}
	
	sprintf (buffer, "Impression par %s.%s@%s",
					 printer->type, printer->name,
					 printer->zone);
	report_msg (REPORT_INFO, buffer);
	
	
	//	Analyse ce qu'il faut faire avec l'imprimante !
	
	if (strcmp (printer->type, "Smaky") == 0) {
		
		//	Impression soit en copiant directement sur Centronics, soit en copiant
		//	directement sur RS-232 ?
		
		if ( (strcmp (printer->name, "Centronics") == 0)
		  || (strcmp (printer->name, "Serial") == 0) ) {
			
			this->PrintPostScriptStream (printer, name, job, error);
			return error;
		}
		
		//	Impression par la voie dtourne qui se nomme GhostScript ?
		
		if (strcmp (printer->name, "GhostScript") == 0) {
			
			this->PrintGhostScript (printer, name, job, error);
			return error;
		}
		
		return 1;
	}
	
	NBPElem* elem = this->FindPrinter (printer);
	
	if ( (elem == 0)
	  || (this->CheckPrinter (elem) == FALSE) ) {
		
		//	L'imprimante n'tait pas connue, il faut refaire la
		//	liste des imprimantes et tenter notre chance une
		//	nouvelle fois...
		
		report_msg (REPORT_ERROR, "Imprimante inconnue");
		
		this->UpdatePrinterList ();
		elem = this->FindPrinter (printer);
		
		if (elem == 0) {
			return 2;
		}
		
		//	On ne supporte pas une imprimante se trouvant dans
		//	une autre zone Appletalk...
	}
	
	if (this->CheckPrinter (elem) == FALSE) {
		return 2;
	}
	
	//	Ouvre  nouveau le fichier et place celui-ci dans la liste
	//	des boults  imprimer.
	
	FILE* file    = fopen (name, "rb");
	int   channel = PrintCon::Open (file, job, &elem->address);
	
	if (channel == -1) {
		job->file = 0;
		fclose (file);
		return 1;
	}
	
	job->file = file;
	return 0;
}


/*
 *	Une itration dans la gestion des imprimantes et des rponses
 *	dsires par les divers clients.
 */

void
ppm_loop_printing (int& open)
{
	PrintCon::AllCheckWrite ();
	PrintCon::AllCheckStatus ();

	if (PPMan::IsStopRequested ()) {
		
		for (int i = 0; i < MAX_PRINTCON; i++) {
			PrintCon::Abort (i);
		}
		
		putenv ("PSTALK_ABORT");
		report_msg (REPORT_INFO, "Impression en cours d'avortement");
	}
	
	
	//	Slection multiple sur les diverses sources possibles (c'est--dire
	//	les connexions avec les imprimantes et les gates de rception et
	//	mission de commandes).
	
	Card32 ident  = 0;
	void*  result = 0;
	NtrSel select;
	
	PrintCon::AllSelect (& select);
	PPMServer::AllSelect (& select);
	
	select.AddDeltaTimeout (SMALL_DELTA_T);
	
	if (select.Sync (ident, result) == 0) {
		PrintCon::AllAfterSelect (ident, (Card32)(result), open);
		PPMServer::AllAfterSelect (ident, (Card32)(result));
	}
};



void
send_clock_message (const char* text, const char* network, const char* title)
{
	char routing[100];
	char message[256];
	strncpy (routing, network, sizeof (routing));
	char* ptr = routing;
	while (*ptr && (*ptr != ' ')) ptr++;
	*ptr = 0;
	
	sprintf (message, "sma_psmsg/b/f%s 4 3 %s Imp *** %s *** %s ***",
					  (routing[0] == '@') ? "/d" : "",
					  (routing[0] == '@') ? routing : "",
					  text, title);
	
	if (!no_message) {
		FOS::Execute (message, FOS_EX_LEAVEKEY);
	}
}


