/*
 *	ppmserver.cc
 *
 *	Routines de gestion du serveur "PostScript Printer Manager".
 *
 *	(C)	 Copyright 1993-1995, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <c++/fos.h>
#include <c++/ntrel.h>
#include "ppm.h"
#include "ppm_atalk.h"
#include "printcon.h"
#include "misc.h"

#include "genericppd.h"

static const char SERVER_NAME[]        = "sma_psserv";
static const char MSG_EXITSERVER[]     = "%%[ exitserver: permanent state may be changed ]%%";
static const char MSG_FLUSHING[]       = "%%[ Flushing: rest of job (to end-of-file) will be ignored ]%%";
static const char MSG_FONTMISSING[]    = " not found, using Courier";
static const int  START_SERVER_TIMEOUT = 50*10;	// attend 10 secondes entre 2 tentatives d'excution


/*
 *	Variables locales utilises pour la gestion de la communication avec
 *	le serveur d'impression PSSERV.
 */

static NtrBar*          gate           = 0;
static volatile Card32  gate_id        = 0;
static int              sel_accept_num = -1;
static char             bufmsg[5000];

/*
 *	Variables globales du serveur d'impression.
 */

PPMServer* PPMServer::list_first = 0;
PPMServer* PPMServer::list_last = 0;


/*
 *	Extrait le message contenu entre les "%%[ " et " ]%%" d'un message
 *	d'erreur pour ne conserver que l'essentiel.
 */

static void
extract_err (char* buffer, const char* text)
{
	int count = 63;
	text += 4;
	
	while (! ( (text[0] == ' ')
	        && (text[1] == ']')
	        && (text[2] == '%')
	        && (text[3] == '%') )
	      && (count--) ) {
		
		*buffer++ = *text++;
	}
	
	*buffer++ = '\n';
	*buffer   = 0;
}



/*
 *	Nettoie une chane et remplace les "_" par des espaces.
 */

static void
clean_space (char* buffer)
{
	char c;
	
	while (c = *buffer) {
		if (c == '_') *buffer = ' ';
		buffer++;
	}
}


PPMServer::PPMServer (PPM_Request* rq)
{
	request = *rq;
	
	next = 0;
	prev = list_last;
	
	list_last ? list_last->next : list_first = this;
	list_last = this;
}


PPMServer::~PPMServer ()
{
	next ? next->prev : list_last = prev;
	prev ? prev->next : list_first = next;
}



/*
 *	Dfinit le gate de rception de requtes pour la
 *	slection mutliple.
 */

void
PPMServer::AllSelect (NtrSel* select)
{
	if ( (gate_id == 0)
	  && (gate == 0) ) {
		
		static Card32 time_stamp = 0;
		
		Card32 time  = Ntr::GetTimeMS ();
		int    error = 0;
		
		if (time_stamp + START_SERVER_TIMEOUT > time) {
			return;
		}
		
		sprintf (bufmsg, "%s bar=%x", SERVER_NAME, & gate_id);
		
		error = FOS::Execute (bufmsg, FOS_EX_LEAVEKEY);
		Ntr::Dels (5);
		
		if (gate_id == 0) {
			if (error) {
				sprintf (bufmsg, "Pas pu excuter le serveur : %04x", error);
				report_msg (REPORT_ERROR, bufmsg);
			}
			return;
		}
		
		report_msg (REPORT_INFO, "Dmarrage du serveur en cours");
		FOS::Execute ("rm #psspool:*.ps", FOS_EX_LEAVEKEY);
		
		time_stamp = time;
		return;
	}
	
	if (gate == 0) {
		gate = new NtrBar (gate_id);
	}
	
	sel_accept_num = select->GetSelNum ();
	select->AddAcceptBar (gate_id, sel_accept_num);
}



/*
 *	Analyse le rsultat de la slection multiple. Gre 
 *	une ventuelle requte reue en entre.
 */

void
PPMServer::AllAfterSelect (Card32 index, Card32 req)
{
	PPMServer* server = 0;
	
	if ( (gate == 0)
	  || (sel_accept_num == -1) ) {
		return;
	}
	
	if (sel_accept_num == index) {
		
		PPM_Request* rq = (PPM_Request*) req;
		
		switch (rq->what) {
			
			case PPM_RQ_PRINT:
				server = new PPMServer (rq);
				break;
			
			default:
				break;
		}
		
		gate->EndAccept (0);
	}
}


/*
 *	Emet une rponse ( une personne en particulier ou a
 *	tout le monde).
 */

void
PPMServer::Emit (JobDef* job, PPM_Answer& answer)
{
	Card16 emit = 0;
	Card32 len  = 0;
	bufmsg[0]   = 0;
	char* print = bufmsg;
	
	switch (answer.what) {
		
		case PPM_ANS_STARTING_JOB:
			sprintf (print, "Dbut de job `%s'.", answer.answer.job.title);
			len   = strlen (print);
			break;
		
		case PPM_ANS_ENDING_JOB:
			sprintf (print, "Fin de job `%s'.", answer.answer.job.title);
			len   = strlen (print);
			break;
		
		case PPM_ANS_STATUS:
			print = answer.answer.status.message;
			len   = strlen (print);
			break;
		
		case PPM_ANS_MESSAGE:
			print = answer.answer.message.message;
			len   = strlen (print);
			emit  = get_report_file ();
			break;
		
		case PPM_ANS_ERROR:
			print = answer.answer.error.message;
			len   = strlen (print);
			emit  = get_report_file ();
			break;
		
		case PPM_ANS_OUTPUT:
			print = answer.answer.output.output;
			len   = answer.answer.output.size;
			emit  = get_report_file ();
			break;
		
		default:
			break;
	}
	
	
	if (emit) {
		FOS::Write (emit, len, print);
	} else {
		if (print != bufmsg) {
			strncpy (bufmsg, print, len);
			bufmsg[len] = 0;
		}
		report_msg (REPORT_INFO, bufmsg);
	}
	
	if (job->user_id == 0) {
		
		// Aucun utilisateur ne s'est abonn  la distribution
		// de ces messages, alors cela ne vaut pas la peine de
		// les distribuer...
		
		return;
	}
	
	answer.num         = job->num;
	answer.id          = 0;
	answer.answer_gate = 0;
	answer.free_gate   = FALSE;
}


/*
 *	Transmet le message `dbut de job'.
 */

void
PPMServer::EmitStartJob (JobDef* job)
{
	PPM_Answer answer;
	
	answer.what = PPM_ANS_STARTING_JOB;
	answer.answer.job = job->job;
	
	PPMServer::Emit (job, answer);
}


/*
 *	Transmet le message `fin de job'.
 */

void
PPMServer::EmitEndJob (JobDef* job)
{
	PPM_Answer answer;
	
	answer.what = PPM_ANS_ENDING_JOB;
	answer.answer.job = job->job;
	
	PPMServer::Emit (job, answer);
}


/*
 *	Transmet un statut en provenance de l'imprimante vers les
 *	clients intresss.
 */

void
PPMServer::EmitStatus (JobDef* job, const char* text)
{
	PPM_Answer answer;
	
	answer.what = PPM_ANS_STATUS;
	
	strcpy (answer.answer.status.job, job->job.title);
	strcpy (answer.answer.status.message, text);
	strcpy (answer.answer.status.source, "");
	
	bzero (answer.answer.status.args, 128);
	
	answer.answer.status.resource.key   = 0;
	answer.answer.status.resource.audio = 0xFFFFFFFF;
	answer.answer.status.resource.text  = 0xFFFFFFFF;
	answer.answer.status.resource.image = 0xFFFFFFFF;
	answer.answer.status.status         = PPM_STATUS_MISC;
	
	PPMServer::Emit (job, answer);
}



/*
 *	Transmet un message en provenance de l'imprimante vers les
 *	clients intresss.
 */

void
PPMServer::EmitMessage (JobDef* job, const char* text)
{
	PPM_Answer answer;
	
	answer.what = PPM_ANS_MESSAGE;
	
	if (strncmp (text, MSG_EXITSERVER, strlen (MSG_EXITSERVER)) == 0) {
		
		answer.answer.message.resource.key   = 0;
		answer.answer.message.resource.audio = 0xFFFFFFFF;
		answer.answer.message.resource.text  = RS_MSG_EXITSERVER;
		answer.answer.message.resource.image = 0xFFFFFFFF;
		extract_err (answer.answer.message.message, text);
		bzero (answer.answer.message.args, 128);
		
	} else if (strncmp (text, MSG_FLUSHING, strlen (MSG_FLUSHING)) == 0) {
		
		answer.answer.message.resource.key   = 0;
		answer.answer.message.resource.audio = 0xFFFFFFFF;
		answer.answer.message.resource.text  = RS_MSG_FLUSHING;
		answer.answer.message.resource.image = 0xFFFFFFFF;
		extract_err (answer.answer.message.message, text);
		bzero (answer.answer.message.args, 128);
		
	} else {
		
		answer.answer.message.resource.key   = 0;
		answer.answer.message.resource.audio = 0xFFFFFFFF;
		answer.answer.message.resource.text  = RS_MSG_FONTMISSING;
		answer.answer.message.resource.image = 0xFFFFFFFF;
		extract_err (answer.answer.message.message, text);
		bzero (answer.answer.message.args, 128);
		
		char* font = answer.answer.message.args;
		char c;
		int count = 127;
		
		while ((c = *text++) && (c != ' ') && (c != '\n') && (c != '\r') && (count--))
			*font++ = c;
	}
	
	PPMServer::Emit (job, answer);
}


/*
 *	Transmet un message d'erreur en provenance de l'imprimante
 *	vers les clients intresss.
 */

void
PPMServer::EmitError (JobDef* job, const char* text)
{
	PPM_Answer answer;
	
	answer.what = PPM_ANS_ERROR;
	
	extract_err (answer.answer.error.message, text);
	bzero (answer.answer.error.args, 128);
	
	PPMServer::Emit (job, answer);
}


/*
 *	Transmet des donnes (message, erreur, rponse)  un client
 *	intress par ces vnements.
 */

void
PPMServer::EmitOutput (JobDef* job, const void* data, long size)
{
	const char* text = (const char*)(data);
	
	if ( (strncmp (text, MSG_EXITSERVER, strlen (MSG_EXITSERVER)) == 0)
	  || (strncmp (text, MSG_FLUSHING, strlen (MSG_FLUSHING)) == 0)
	  || (strstr (text, MSG_FONTMISSING)) ) {
		
		EmitMessage (job, text);
		return;
	}
	
	if ((text[0] == '%') && (text[1] == '%') && (text[2] == '[') && (text[3] == ' ')) {
		
		EmitError (job, text);
		return;
	}
	
	// Emet la rponse.
	
	PPM_Answer answer;
	
	while (size) {
		
		// Transmet la rponse en morceaux (on ne peut pas transmettre plus
		// que 512 bytes dans un message, limitation due  la diffusion).
		
		answer.what = PPM_ANS_OUTPUT;
		answer.answer.output.size = (size > 512) ? 512 : size;
		answer.answer.output.more = (size >= 512) ? TRUE : FALSE;
		bcopy (data, answer.answer.output.output, answer.answer.output.size);
		
		PPMServer::Emit (job, answer);
		
		size -= answer.answer.output.size;
		data += answer.answer.output.size;
	}
}


/*
 *	Imprime un document : remplit le descripteur PPM avec les informations
 *	adquates et retourne le nom du fichier  imprimer.
 */

Bool
PPMServer::PrintDocument (PPM_PrnName& printer, char*& file)
{
	int len = (list_first) ? strlen (list_first->request.request.print) : 0;
	file    = len ? new char[len+1] : 0;
	
	//	Y a-t-il un nom de fichier  imprimer ?
	
	if (file == 0) {
		return FALSE;
	}
	
	strcpy (printer.name, list_first->request.name.name); clean_space (printer.name);
	strcpy (printer.type, list_first->request.name.type); clean_space (printer.type);
	strcpy (printer.zone, list_first->request.name.zone); clean_space (printer.zone);
	strcpy (printer.comm, list_first->request.name.comm);
	strcpy (file,         list_first->request.request.print);
	
	//	Retire le document de la liste des fichiers  imprimer.
	
	delete list_first;
	
	return TRUE;
}

