/*
 *	printcon.cc
 *
 *	Connexion avec une imprimante via Appletalk.
 *
 *	(C)  Copyright 1993-1995, Pierre ARNAUD, CH-1437 SUSCEVAZ
 */

#include <stdio.h>
#include <string.h>

#include <c++/ntrel.h>
#include "misc.h"
#include "ppm.h"
#include "ppm_atalk.h"
#include "printcon.h"


/*
 *  Constantes diverses utilises pour la communication avec
 *  le Printer Access Protocol (PAP).
 */

#define	MY_FLOW_QUANTUM		8						// lit et crit 4096 bytes au maximum
#define STATUS_DELTA_T		100						// toutes les 2 secondes
#define STATUS_WR_DELTA_T	150						// 3 sec. aprs une criture

/*
 *  Variables globales utilises pour grer les connexions avec
 *  les imprimantes et pour le dialogue avec PAP.
 */

NtrBar*   PrintCon::pap_gate;						// gate de communication avec PAP
PrintCon* PrintCon::printcons[MAX_PRINTCON];		// table des connexions



/*********************************************************
 *														 *
 *		Mthodes prives de gestion de la connexion      *
 *														 *
 *********************************************************/


/*
 *  Ouvre la connexion : initialise un certain nombre de paramtres.
 */

PrintCon::PrintCon (FILE* f, JobDef* j)
{
    this->pending.status_pending = FALSE;
    this->pending.write_pending  = FALSE;
    this->pending.read_pending   = FALSE;
    this->pending.to_write       = FALSE;
    this->pending.demande_status = FALSE;
    this->pending.fin_envoi      = FALSE;
    this->pending.impr_terminee  = FALSE;
    this->pending.un_status_recu = FALSE;
    
    this->sel_in_num    = -1;
    this->sel_out_num   = -1;
    this->quantum       = 0;
    this->written       = 0;
    this->eof           = FALSE;
    this->closed        = FALSE;
    this->eof_recu      = FALSE;
    this->not_open_yet  = TRUE;
    this->not_ready_yet = TRUE;
    
    this->file = f;
    this->job  = j;
    
    Ntr::GetTimeMS (this->pending.prochain_status);
}


/*
 *  Initialise les variables relatives  l'imprimante  utiliser
 *  et dfinit la commande  envoyer  PAP pour tablir une
 *  connexion.
 */

void
PrintCon::Init (const NBPAddress* address)
{
    this->cmd_pap.command                  = PAP_OPEN_CONN;
    this->cmd_pap.u.open_conn.dest_net     = address->network;
    this->cmd_pap.u.open_conn.dest_node    = address->node;
    this->cmd_pap.u.open_conn.dest_socket  = address->socket;
    this->cmd_pap.u.open_conn.flow_quantum = MY_FLOW_QUANTUM;
    this->cmd_pap.u.open_conn.status_open  = status;
}


/*
 *  Lit des donnes  envoyer  l'imprimante depuis le
 *  fichier spcifi lors de l'initialisation.
 */

int
PrintCon::ReadBuffer (Card32& size)
{
	int read = 0;
	
    size = quantum * 512;
    read = fread (this->buffer, 1, size, this->file);
    
    this->eof = (read == size) ? FALSE : TRUE;
    size = read;
    
    return this->eof;
}


/*
 *  Regarde s'il est possible d'crire un nouveau paquet
 *  et prpare la commande.
 */

void
PrintCon::CheckWrite ()
{
    if ( (this->not_open_yet)
      || (this->not_ready_yet) ) {
    	return;
    }
    
    if ( (! this->pending.write_pending)
      && (! this->pending.fin_envoi)
      && (! this->eof) ) {
        
        Card32 size = 0;
        
        this->ReadBuffer (size);
        
        if ( (this->eof)
          && (size == 0) ) {
            buffer[0] = ' ';
            buffer[1] = 0;
            size      = 1;
        }
        
        if (size) {
            this->cmd_pap.command                = PAP_WRITE;
            this->cmd_pap.eof                    = this->eof;
            this->cmd_pap.u.write.length_wr_data = size;
            this->cmd_pap.u.write.write_data     = buffer;
            this->pending.write_pending          = TRUE;
            this->pending.to_write               = TRUE;
        }
    }
}


/*
 *  Vrifie si un message de statut de l'imprimante est
 *  dsir ( intervalles rguliers) et prpare la commande.
 */

void
PrintCon::CheckStatus ()
{
	Card32 time = 0;
	
	Ntr::GetTimeMS (time);
	
    if ( (this->not_open_yet)
      || (this->not_ready_yet) ) {
        
        // Retarde la lecture du statut. Ca ne sert  rien de se
        // prcipiter sur le statut juste aprs l'ouverture de la
        // connexion !
        
        this->pending.prochain_status = time + STATUS_DELTA_T;
        
        return;
    }
    
    if (this->pending.prochain_status < time) {
        this->stat_cmd.command                 = PAP_GET_STATUS;
        this->stat_cmd.u.get_status.status_get = status;
        this->pending.demande_status           = TRUE;
        this->pending.prochain_status          = time + STATUS_DELTA_T;
    }
}


/*
 *  Remplit le record de slection multiple en fonction des oprations
 *  qui doivent tre effectues :
 *
 *  - Ouverture d'une connexion.
 *  - Attente de la confirmation de l'ouverture.
 *  - Lecture d'infos
 *  - Emission d'une nouvelle requte
 */

void
PrintCon::Select (NtrSel* select)
{
    if (this->closed) {
    	return;
    }
    
    this->sel_in_num   = -1;
    this->sel_out_num  = -1;
    this->sel_open_num = -1;
    
    if (this->not_open_yet) {
        
        this->sel_open_num = select->GetSelNum ();
        
        if (PrintCon::pap_gate == 0) {
            PrintCon::pap_gate = new NtrBar (PAP_COMMAND_GATE);
        }
        
        select->AddOfferBar (PrintCon::pap_gate, this->sel_open_num);
        return;
    }
    
    this->sel_out_num = select->GetSelNum ();
    select->AddAcceptBar (this->gates.out, this->sel_out_num);
    
    if (this->not_ready_yet) {
    	return;
    }
    
    //	S'il y a des requtes  mettre, il faut encore rajouter
    //	un record dcrivant le gate des requtes.
    
    if ( ( (this->pending.to_write)
        && (! this->pending.fin_envoi) )
      || (! this->pending.read_pending)
      || ( (this->pending.demande_status)
        && (! pending.status_pending) )
      || (this->eof_recu) ) {
        
        this->sel_in_num = select->GetSelNum ();
        select->AddOfferBar (gates.in, this->sel_in_num);
    }
}


/*
 *  Gre le rsultat d'une slection multiple (si elle nous concerne).
 *  Il faut notamment :
 *
 *  - Prendre note de l'ouverture de la connexion.
 *  - Prendre note de la confirmation de l'ouverture de la connexion.
 *  - Prendre note des informations retournes par l'imprimante.
 *  - Envoyer une nouvelle requte.
 */

void
PrintCon::AfterSelect (Card32 index, Card32 value)
{
    if (this->closed) {
    	return;
    }
    
    void* dummy = 0;
    
    if (this->not_open_yet) {
        
        //  Si la connexion avec l'imprimante n'a pas encore t ouverte
        //  au niveau Appletalk, nous allons le faire maintenant.
        
        if (index != this->sel_open_num) {
        	return;
        }
        
        //	Le buffer `cmd_pap' a t prpar par l'initialisation effectue
        //	dans la mthode `Init'...
        
        PrintCon::pap_gate->EndOffer (& this->cmd_pap, dummy);
        
        if (this->cmd_pap.error) {
            this->closed = TRUE;
            return;
        }
        
        // L'ouverture a russi. Il faudra encore attendre la confirmation
        // sur le gate `out' livr par PAP.
        
        this->not_open_yet  = FALSE;
        this->not_ready_yet = TRUE;
        
        this->gates   = this->cmd_pap.u.open_conn.io_gates;
        this->quantum = this->cmd_pap.u.open_conn.flow_quantum <? MY_FLOW_QUANTUM;
        
        return;
    }
    
    if (this->not_ready_yet) {
        
        // La connexion est ouverte mais nous attendons toujours une
        // confirmation. Si elle a t reue correctement, la connexion
        // deviendra utilisable.
        
        if (index != this->sel_out_num) {
        	return;
        }
        
        NtrBar bar (this->gates.out);
        bar.EndAccept (0);
        
        switch (value) {
            
            case PAP_STATUS_AVAIL:
                break;
            
            case PAP_CONN_OPEN:
                this->not_ready_yet = FALSE;
                break;
            
            default:
                this->closed = TRUE;
        }
        
        return;
    }
    
    
    // La connexion est tablie. Il faut maintenant soit lire une rponse
    // de PAP, soit envoyer une nouvelle requte  l'imprimante.
    
    if (index == this->sel_out_num) {
        
        NtrBar bar (this->gates.out);
        bar.EndAccept (0);
        
        switch (value) {
            
            case PAP_STATUS_AVAIL:
                if (this->status[0]) {
                    PPMServer::EmitStatus (this->job, this->status);
                }
                
                this->pending.status_pending = FALSE;
                
                if (this->pending.fin_envoi) {
                	this->pending.un_status_recu = TRUE;
                }
                
                break;
            
            case PAP_DATA_WRITTEN:
                this->pending.write_pending  = FALSE;
                break;
            
            case PAP_STATUS_NO_RESP:
            	report_msg (REPORT_ERROR, "L'imprimante ne donne pas son statut");
                this->pending.status_pending = FALSE;
                break;
            
            case PAP_DATA_READ:
                this->pending.read_pending   = FALSE;
                
                if (this->cmd_read.u.read.read_info->eof_read) {
                	this->eof_recu = TRUE;
                }
                
                if (this->eof_recu) {
                	report_msg (REPORT_INFO, "***EOF***");
                }
                
                if (this->cmd_read.u.read.read_info->length_rd_data > 0) {
                	long len = this->cmd_read.u.read.read_info->length_rd_data;
                    this->string_read[len] = 0;
                    PPMServer::EmitOutput (this->job, this->string_read, len);
                }
                
                break;
            
            case PAP_READ_GARBAGE:
                report_msg (REPORT_ERROR, "L'imprimante dlire");
                this->pending.read_pending = FALSE;
                break;
            
            default:
            	report_msg (REPORT_ERROR, "Erreur PAP");
            	break;
        }
        
        return;
    }
    
    
    // Les requtes sont mises selon les priorits suivantes :
    //
    // 1. Fermeture de la connexion (si eof reu depuis l'imprimante).
    // 2. Ecrire de nouvelles donnes vers l'imprimante (s'il y en a).
    // 3. Lire les donnes en provenance de l'imprimante (s'il y en a).
    // 4. Lire le statut de l'imprimante (de temps  autre).
    
    if (index == this->sel_in_num) {
        
        if (this->eof_recu) {
            
            this->cmd_pap.command = PAP_CLOSE_CONN;
            
            NtrBar bar (this->gates.in);
            bar.EndOffer (& this->cmd_pap, dummy);
            
            if (this->cmd_pap.error) {
                report_msg (REPORT_ERROR, "PAP refuse de fermer la connexion");
            }
            
            this->closed = TRUE;
            return;
        }
    	
        if ( (this->pending.to_write)
          && (! this->pending.fin_envoi) ) {
            
            NtrBar bar (this->gates.in);
            bar.EndOffer (& this->cmd_pap, dummy);
            
            this->pending.to_write = FALSE;
            
            if (this->cmd_pap.error) {
                report_msg (REPORT_ERROR, "PAP refuse d'crire");
            } else {
            	Card32 time = 0;
            	char   buffer[100];
            	
            	Ntr::GetTimeMS (time);
            	
                this->written += this->quantum;
                
                sprintf (buffer, "%5d KB transfrs.", this->written >> 1);
                report_msg (REPORT_INFO, buffer);
                
                //  Reset de la demande de status (pas besoin, puisque l'on
                //  peut imprimer, c'est que tout va bien).
                
                this->pending.prochain_status = time + STATUS_WR_DELTA_T;
                this->pending.demande_status  = FALSE;
            }
            
            return;
        }
        
        if (! this->pending.read_pending) {
            
            this->cmd_read.command                             = PAP_READ;
            this->cmd_read.u.read.read_info                    = &this->data_read;
            this->cmd_read.u.read.read_info->read_data         = this->string_read;
            this->cmd_read.u.read.read_info->flow_quantum_read = MY_FLOW_QUANTUM;
            
            NtrBar bar (this->gates.in);
            bar.EndOffer (& this->cmd_read, dummy);
            
            if (this->cmd_read.error) {
                report_msg (REPORT_ERROR, "PAP refuse de lire");
            } else {
                this->pending.read_pending = TRUE;
            }
            
            return;
        }
        
        if ( (this->pending.demande_status)
          && (!this->pending.status_pending) ) {
            
            NtrBar bar (this->gates.in);
            bar.EndOffer (& this->stat_cmd, dummy);
            
            if (this->stat_cmd.error == 0) {
                this->pending.status_pending = TRUE;
                this->pending.demande_status = FALSE;
            }
            
            return;
        }
    }
}


/**********************************************************
 *														  *
 *		Mthodes publiques de gestion des connexions	  *
 *														  *
 **********************************************************/


/*
 *  Ouvre une connexion avec une imprimante. Retourne le numro
 *  de la connexion ou -1 si plus aucune n'est disponible.
 */

int
PrintCon::Open (FILE* file, JobDef* job, const NBPAddress* address)
{
	for (int i = 0; i < MAX_PRINTCON; i++) {
		
		if (PrintCon::printcons[i] == 0) {
			
			PrintCon::printcons[i] = new PrintCon (file, job);
			PrintCon::printcons[i]->Init (address);
			
			PPMServer::EmitStartJob (job);
			return i;
		}
	}
	
	return -1;
}


/*
 *  Avorte une impression. On simule en fait un `EOF' de fichier,
 *  ce qui va arrter l'imprimante, puis fermer la connexion.
 */

void
PrintCon::Abort (int num)
{
	if ( (num < 0)
	  || (num >= MAX_PRINTCON) ) {
		return;
	}
	
	if (PrintCon::printcons[num]) {
		fseek (PrintCon::printcons[num]->file, 0, SEEK_END);
	}
}


/*
 *  Vrifie s'il y a des choses  crire dans les diverses
 *  connexions.
 */

void
PrintCon::AllCheckWrite ()
{
	for (int i = 0; i < MAX_PRINTCON; i++) {
		if (PrintCon::printcons[i]) {
			PrintCon::printcons[i]->CheckWrite ();
		}
	}
}


/*
 *  Vrifie s'il y a des statuts  lire depuis les diverses
 *  connexions.
 */

void
PrintCon::AllCheckStatus ()
{
	for (int i = 0; i < MAX_PRINTCON; i++) {
		if (PrintCon::printcons[i]) {
			PrintCon::printcons[i]->CheckStatus ();
		}
	}
}


/*
 *  Remplit les records de slection multiple pour traiter les
 *  demandes des diverses connexions.
 */

void
PrintCon::AllSelect (NtrSel* rec)
{
	for (int i = 0; i < MAX_PRINTCON; i++) {
		if (printcons[i]) {
			printcons[i]->Select (rec);
		}
	}
}


/*
 *  Gre le rsultat d'une slection. Retourne le nombre de connexions
 *  encore ouvertes.
 */

void
PrintCon::AllAfterSelect (Card32 index, Card32 value, int& num_open)
{
	int i;
	num_open = 0;
	
	for (i = 0; i < MAX_PRINTCON; i++) {
		if (PrintCon::printcons[i]) {
			PrintCon::printcons[i]->AfterSelect (index, value);
			if (PrintCon::printcons[i]->closed) {
				PPMServer::EmitEndJob (PrintCon::printcons[i]->job);
				delete PrintCon::printcons[i];
				PrintCon::printcons[i] = 0;
			}
		}
	}
	
	for (i = 0; i < MAX_PRINTCON; i++) {
		if (PrintCon::printcons[i]) {
			num_open++;
		}
	}
}


