/*
 *	activityspy.cxx
 *
 *	Memory, processor time and network spy for SMA_SMAKY
 *
 *	(C) 1994 Erik BRUCHEZ
 *
 */

#include "activityspy.h"
#include "gesmem.h"
#include "ntrel.h"
#include "smevents.h"
#include "lib.h"
#include "fos.h"
#include "str.h"
#include "clock.h"
#include "pixmapman.h"
#include "config.h"

#include "mon.h"

#define SPY_ACTIVITY_PRIO 6

#define DEFAULT_SAMPLE_PERIOD	10
#define DEFAULT_MEM_PERIOD_1	1
#define DEFAULT_MEM_PERIOD_2	6
#define DEFAULT_PROC_PERIOD_1	1
#define DEFAULT_PROC_PERIOD_2	3
#define DEFAULT_NET_PERIOD		1
#define DEFAULT_MOUSE_PERIOD_1	2
#define DEFAULT_MOUSE_PERIOD_2	3

extern Clock		clock;
extern PixmapMan	pman;

/*
 *	Find if `pos' is in one of the valid zones defined by
 *	`box' and `size'.
 *	This is used by SpyActivity in order to find if 
 *	sleep is needed.
 */

static Card8
FindZone (Box box, Card8 size, Card8 mz, Pos pos)
{
	Card16 s = (Card16)(size);
	if (mz & 1) {
		if ( ((Card16)(pos.x) < s) && ((Card16)(pos.y) < s) ) return 1;
	}
	Card16 px = box.d.dx - s;
	if (mz & 2) {
		if ( ((Card16)(pos.x) >= px) && ((Card16)(pos.y) < s) ) return 2;
	}
	Card16 py = box.d.dy - s;
	if (mz & 4) {
		if ( ((Card16)(pos.x) >= px) && ((Card16)(pos.y) >= py) ) return 3;
	}
	if (mz & 8) {
		if ( ((Card16)(pos.x) < s) && ((Card16)(pos.y) >= py) ) return 4;
	}
	return 0;
}


/*
 *	Send a "sleep command" to START
 *	Return the number of the actually assigned keyboard
 *	or zero.
 */

static Card16
Sleep ()
{
	static const char sleep_prog[] = "START.CODE";
	
	for (Card16 kbd_num = 1; kbd_num <= 32; kbd_num++) {
		Card16 proc_num;
		const char *proc_name;
		
		Lib::SrcKNP (kbd_num, proc_num, proc_name);
		
		if (Str::Comp (sleep_prog, proc_name)) {
			Card16 num = 0;
			
			Lib::VAssKey (num);
			
			Lib::AssKey (kbd_num);
			Lib::WrAKey ('S' + (8 << 8) + (0x53 << 16));
			
			/*
			 *	This doesn't work since the key arrives to the actually assigned
			 *	keyboard
			 *
			Card16 kbd_chan;
			
			Lib::SrcCHK (kbd_num, kbd_chan);
			
			char buffer[] = { 0x0a, 'S', 8, '\0' };
			
			if (kbd_chan)
				Fos::Command (kbd_chan, buffer);
			 */
			
			return num;
		}
	}
	return 0;
}


/*
 *	This process spies memory, processor time, network activity
 *	and mouse position (for sleeping).
 */

void
SpyActivity (Card32 param)
{
	volatile ActivitySpy *that = (volatile ActivitySpy *)(param);
	
	EvActivityChanged ev;
	
	//	Temporary statistics
	
	Card32	proctime;
	Card32	idletime;
	Card32	progtime;
	
	Card32	deltaproc2 = 0;
	Card32	deltaidle2 = 0;
	
	Ntr::GetSyTime (proctime, idletime);
	Ntr::GetPrTime (progtime);
	
	//	Local period counters
	
	Card32	mem_period_1 =		that->mem_period_1;
	Card32	mem_period_2 =		that->mem_period_2;
	
	Card32	proc_period_1 =		that->proc_period_1;
	Card32	proc_period_2 =		that->proc_period_2;
	
	Card32	net_period =		that->net_period;
	
	Card32	mouse_period_1 =	that->mouse_period_1;
	Card32	mouse_period_2 =	that->mouse_period_2;
	
	//	Mouse spying variables
	
	Pos		mpos = { 0, 0};
	Card8	mzone = 0;			//	0 <=> not in any zone, else 1..4
	Bool	sleep = FALSE;
	Card16	kbd_num = 0;		//	keyboard number when sleep command sent
								//	(valid when sleep == TRUE)
	
	Bool	mem_clut_chg = FALSE;	//	recall that clut changed
	Card8	prev_clut_chg = 0;		//	clut changed n sampling periods ago
	
	//	Main loop
	
	for (;;) {
		
		Ntr::Delms (that->sample_period);
		
		Bool send_event = FALSE;
		
		ev.mem_changed	= FALSE;
		ev.mem_histo_changed = FALSE;
		ev.cpu_changed	= FALSE;
		ev.cpu_histo_changed = FALSE;
		ev.net_changed	= FALSE;
		ev.clut_changed = FALSE;
		
		//	Memory sampling period elapsed ?
		
		if (!(--mem_period_1)) {
			
			Card32 free = MemAccount::GetFreeMemory ();
			Card16 percent = free * 100 / that->total;
			
			if (!(--mem_period_2)) {
				if (that->mem_grmode == GRAPH_MODE_HISTOGRAM) {
					
					//	Update histogram information
					
					Bool before = (that->mem_hmode == HISTO_MODE_DYNAMICAL);
					
					if (before) that->mem_percents[7] = percent;
					
					for (int i = 0; i < 7; i++)
						that->mem_percents[i] = that->mem_percents[i + 1];
					
					if (!before) that->mem_percents[7] = percent;
					
					ev.mem_histo_changed = TRUE;
					send_event = TRUE;
				}
				mem_period_2 = that->mem_period_2;
			}
			
			if (free != that->free) {
				
				//	Memory occupation has changed
				
				that->free = free;
				that->hole = MemAccount::GetBiggestHole ();
				
				if (that->mem_hmode == HISTO_MODE_DYNAMICAL) {
					that->mem_percents[7] = percent;
					ev.mem_histo_changed = TRUE;
				}
				
				ev.mem_changed = TRUE;
				send_event = TRUE;
			}
			
			//	Update local period counter
			
			mem_period_1 = that->mem_period_1;
		}
		
		//	Processor time sampling period elapsed ?
		
		if (!(--proc_period_1)) {
			
			Card32 ptime, itime, pgtime;
			
			Ntr::GetSyTime (ptime, itime);
			Ntr::GetPrTime (pgtime);
			
			Int32 deltaproc;
			const Card32 deltaidle = (itime > idletime) ? (itime - idletime) : 0;
			
			if (that->sma_smaky_too) {
				deltaproc = ptime - proctime;
			} else {
				const Card32 deltaprog = (pgtime > progtime) ? (pgtime - progtime) : 0;
				deltaproc = ptime - proctime - deltaprog;
			}
			
			if (deltaproc < 0) deltaproc = 0;
			
			const Card16 percent = deltaproc ? deltaidle * 100 / deltaproc : 100;
			
			proctime = ptime;
			idletime = itime;
			progtime = pgtime;
			
			deltaproc2 += deltaproc;
			deltaidle2 += deltaidle;
			
			if (!(--proc_period_2)) {
				
				if (that->cpu_grmode == GRAPH_MODE_HISTOGRAM) {
					
					//	Update information
					
					const Card16 p2 = deltaproc2 ? deltaidle2 * 100 / deltaproc2 : 100;
					Bool before = (that->cpu_hmode == HISTO_MODE_DYNAMICAL);
					
					if (before) that->cpu_percents[7] = p2;
					
					for (int i = 0; i < 7; i++)
						that->cpu_percents[i] = that->cpu_percents[i + 1];
					
					if (!before) that->cpu_percents[7] = p2;
					
					ev.cpu_histo_changed = TRUE;
					send_event = TRUE;
				}
				
				deltaproc2 = 0;
				deltaidle2 = 0;
				proc_period_2 = that->proc_period_2;
			}
			
			if (percent != that->cpu_percent) {
				
				//	Processor occupation has changed
				
				that->cpu_percent = percent;
				
				if (that->cpu_hmode == HISTO_MODE_DYNAMICAL) {
					that->cpu_percents[7] = percent;
					ev.cpu_histo_changed = TRUE;
				}
				
				ev.cpu_changed = TRUE;
				send_event = TRUE;
			}
			
			//	Update local period counter
			
			proc_period_1 = that->proc_period_1;
		}
		
		//	Network sampling period elapsed (if we care) ?
		
		if (that->net_registered) {
			if (!(--net_period)) {
				
				Card32 d[4];
				Bool changed[4];
				Bool chgd = FALSE;;
				
				//	Here, we'll notify an ACTIVITY CHANGE between
				//	the previous period and the current period
				
				for (int i = 0; i < 4; i++) {
					const Card32 nc = that->net_count[i];
					
					d[i] = nc - that->old_net_count[i];
					that->old_net_count[i] = nc;
					
					if (d[i] != that->old_net_activity[i]) {
						changed[i] = TRUE;
						chgd = TRUE;
						that->old_net_activity[i] = d[i];
					} else
						changed[i] = FALSE;
				}
				
				if (chgd) {
					
					for (int i = 0; i < 4; i++) {
						ev.changed[i] = changed[i];
						ev.activity[i] = d[i];
					}
					
					ev.net_changed = TRUE;
					send_event = TRUE;
				}
				
				net_period = that->net_period;
			}
		}
		
		//	Sleep ?
		
		if (that->mzones) {
			if (!(--mouse_period_1)) {
				Pos pos;
				//Card8 state;
				
				if (!sleep) {
					
					//	Read the mouse
					
					Ntr::SetTim (0);
					//Lib::SpyMouse ((Card32 *)(&pos), &state);
					pos.x = 100;//DEBUG
					pos.y = 100;
					Ntr::SetTim (0xffff);
					
					//	Test position and conditions
					
					if ( (pos.x == mpos.x) && (pos.y == mpos.y) ) {
						
						//	Mouse didn't move
						
						Card8 zone = FindZone (that->screen_box, that->zone_size,
											   that->mzones, pos);
						
						if ( (zone) && (zone == mzone) ) {
							if (!(--mouse_period_2)) {
								kbd_num = Sleep ();
								if (kbd_num) {
									Ntr::Delms (that->sleep_delay);
									if (clock.IsMuted ()) sleep = TRUE;
								}
								mouse_period_2 = that->mouse_period_2;
							}
						} else {
							mzone = zone;
							mouse_period_2 = that->mouse_period_2;
						}
						
					} else {
						
						//	Mouse moved
						
						mzone = FindZone (that->screen_box, that->zone_size,
										  that->mzones, pos);
						mpos = pos;
						mouse_period_2 = that->mouse_period_2;
					}
				} else {	// sleep
					
					//	Trick in order to guess if the computer is awake !
					
					if (!clock.IsMuted ()) {
						if (kbd_num) Lib::AssKey (kbd_num);
						mzone = 0;
						mouse_period_2 = that->mouse_period_2;
						sleep = FALSE;
					}
				}
				
				mouse_period_1 = that->mouse_period_1;
			}
		}
		
		//	Clut changed ?
		
		Bool clut_chg = FALSE;
		
		if (!mem_clut_chg) {
			for (int i = 0; i < 4; i++)
				ev.dm_clut_chg[i] = FALSE;
		}
		
		for (int i = 0; i < 4; i++) {
			if (pman.ClutChanged (i)) {
				pman.ClutChangedAck (i);
				clut_chg = TRUE;
				ev.dm_clut_chg[i] = TRUE;
				break;
			}
		}
		
		if (clut_chg) {
			if (prev_clut_chg == 0) {
				ev.clut_changed = TRUE;
				send_event = TRUE;
			} else
				mem_clut_chg = TRUE;
			
			prev_clut_chg = 1;
		} else {
			if ( (mem_clut_chg) && (prev_clut_chg != 1) ) {
				ev.clut_changed = TRUE;
				send_event = TRUE;
				mem_clut_chg = FALSE;
				prev_clut_chg = 1;
			} else {
				if (prev_clut_chg)
					if (++prev_clut_chg > 2) prev_clut_chg = 0;
			}
		}
		
		//	Send event if needed
		
		if (send_event && that->evbox && !that->is_muted)
			that->evbox->PostEvent (ev);
	}
}


ActivitySpy::ActivitySpy ()
{
	//	Simple statistics
	
	this->total = MemAccount::GetTotalMemory ();
	this->hole = MemAccount::GetBiggestHole ();
	this->free = MemAccount::GetFreeMemory ();
	
	this->cpu_percent = 100;
	
	//	Periods
	
	this->sample_period = DEFAULT_SAMPLE_PERIOD;
	this->mem_period_1 = DEFAULT_MEM_PERIOD_1;
	this->mem_period_2 = DEFAULT_MEM_PERIOD_2;
	this->proc_period_1 = DEFAULT_PROC_PERIOD_1;
	this->proc_period_2 = DEFAULT_PROC_PERIOD_2;
	this->net_period = DEFAULT_NET_PERIOD;
	this->mouse_period_1 = DEFAULT_MOUSE_PERIOD_1;
	this->mouse_period_2 = DEFAULT_MOUSE_PERIOD_2;
	
	//	Modes
	
	this->sma_smaky_too = FALSE;
	this->mem_grmode = GRAPH_MODE_BAR;
	this->cpu_grmode = GRAPH_MODE_HISTOGRAM;
	this->mem_hmode = HISTO_MODE_NORMAL;
	this->cpu_hmode = HISTO_MODE_DYNAMICAL;
	this->sleep_delay = 75;
	
	//	Statistics (histograms)
	
	for (int i = 0; i < 8; i++) {
		this->mem_percents[i] = 100;
		this->cpu_percents[i] = 100;
	}
	
	//	Network management
	
	this->net_registered = FALSE;
	this->net_driver_name = NULL;
	this->net_rec = NULL;
	this->net_count = NULL;
	
	for (i = 0; i < 4; i++) {
		this->old_net_count[i] = 0;
		this->old_net_activity[i] = 0;
	}
	
	//	Mouse management
	
	this->mzones = 0xf;
	Box b = { { 0, 0 }, { 10000, 10000 } };
	this->screen_box = b;
	this->zone_size = 8;
	
	this->evbox = 0;
	this->is_muted = TRUE;
	
	Card32 pid  = 0;
	Card16 pino = 0;
	
	Ntr::CreTask (SpyActivity, 1000, SPY_ACTIVITY_PRIO,
				  "SpyActivity", (Card32)(this), 0, & pid, & pino);
}


ActivitySpy::~ActivitySpy ()
{
}


void
ActivitySpy::NetDriverRegister (const char *net_driver_name,  NetCountRec *rec)
{
	if ( (net_driver_name == NULL)
	  || (net_driver_name[0] == '\0')
	  || (rec == NULL)
	  || (this->net_registered) ) return;
	
	this->net_driver_name = net_driver_name;
	this->net_count       = (Card32 *)(rec);
	this->net_registered  = TRUE;
}


void
ActivitySpy::MuteOn ()
{
	this->is_muted = TRUE;
}


void
ActivitySpy::MuteOff ()
{
	this->is_muted = FALSE;
}


void
ActivitySpy::SetCPUGraphMode (GraphMode mode)
{
	if (mode == GRAPH_MODE_HISTOGRAM)
		for (int i = 0; i < 8; i++)
			this->cpu_percents[i] = 100;
	
	this->cpu_grmode = mode;
}


void
ActivitySpy::SetMemGraphMode (GraphMode mode)
{
	if (mode == GRAPH_MODE_HISTOGRAM) {
		for (int i = 0; i < 8; i++)
			this->mem_percents[i] = 100;
	} else {
		this->free = MemAccount::GetFreeMemory ();
		this->hole = MemAccount::GetBiggestHole ();
	}
	
	this->mem_grmode = mode;
}


void
ActivitySpy::GetMemStats (Card32& total, Card32& hole, Card32& free)
{
	total = this->total;
	hole = this->hole;
	free = this->free;
}

/*
void
ActivitySpy::CardToAsc (Card32 value, char *result)
{
	int i = 12;
	
	//	Convert the number
	
	do {
		result[--i] = ((char)(value % 10)) + '0';
		value /= 10;
	} while (value);
	
	//	Fill-up with spaces
	
	while (i) result[--i] = ' ';
}
*/

void
ActivitySpy::CardToAsc (Card32 value, char *result)
{
	//	Warning, we use special characters !
	//	3-12 : digits
	//	13 :   space of same width as digits
	//	14 :   space of same width as tick (')
	
	int j = 0;
	const int n = 10 - 1;
	
	value %= 100000000;
	
	//	Convert the number
	
	do {
		if ( (j == 3) || (j == 7) ) result[n - j++] = 39;
		result[n - j++] = ((char)(value % 10)) + 3;
		value /= 10;
	} while (value);
	
	//	Fill-up with spaces
	
	while (j < (n + 1)) {
		if ( (j == 3) || (j == 7) ) result[n - j++] = 14;
		result[n - j++] = 13;
	}
}


void
ActivitySpy::GetMemStats (char *buffer)
{
	const int n = 10;
	
	this->CardToAsc (this->free, buffer);
	
	buffer += n;
	*buffer++ = ' ';
	*buffer++ = '/';
	*buffer++ = ' ';
	
	this->CardToAsc (this->hole, buffer);
	
	buffer += n;
	*buffer = '\0';
}


void
ActivitySpy::GetCPUStats (char *buffer)
{
	const char txt[] = "Processeur : ";
	const Card32 len = Str::Len (txt);
	
	Str::Copy (buffer, txt);
	buffer += len;
	
	Card16 s = 100 - this->cpu_percent;
	
	//	s is a value in percent
	
	*buffer++ = 127;	// 2 spaces
	*buffer++ = 127;
	*buffer = '0';		// zero...
	
	char *p = buffer;
	
	for (int i = 0; i < 3; i++) {
		Card8 v = s % 10;
		s /= 10;
		*p-- = '0' + v;
		if (s == 0) break;
	}
	
	buffer++;
	*buffer++ = ' ';
	*buffer++ = '%';
	*buffer = '\0';
}


void
ActivitySpy::SetSamplePeriod (Card32 period)
{
	this->sample_period = period ? period : DEFAULT_SAMPLE_PERIOD;
}


void
ActivitySpy::SetFirstMemPeriod (Card32 period)
{
	this->mem_period_1 = period ? period : DEFAULT_MEM_PERIOD_1;
}


void
ActivitySpy::SetSecondMemPeriod (Card32 period)
{
	this->mem_period_2 = period ? period : DEFAULT_MEM_PERIOD_2;
}


void
ActivitySpy::SetFirstProcPeriod (Card32 period)
{
	this->proc_period_1 = period ? period : DEFAULT_PROC_PERIOD_1;
}


void
ActivitySpy::SetSecondProcPeriod (Card32 period)
{
	this->proc_period_2 = period ? period : DEFAULT_PROC_PERIOD_2;
}


void
ActivitySpy::SetNetPeriod (Card32 period)
{
	this->net_period = period ? period : DEFAULT_NET_PERIOD;
}


void
ActivitySpy::SetFirstMousePeriod (Card32 period)
{
	this->mouse_period_1 = period ? period : DEFAULT_MOUSE_PERIOD_1;
}


void
ActivitySpy::SetSecondMousePeriod (Card32 period)
{
	this->mouse_period_2 = period ? period : DEFAULT_MOUSE_PERIOD_2;
}


void
ActivitySpy::ParseConfiguration (const ConfigSet *config)
{
	if (this->mem_grmode != config->mem_grmode)
		this->SetMemGraphMode (config->mem_grmode);
	if (this->mem_hmode != config->mem_hmode)
		this->SetMemHistoMode (config->mem_hmode);
	if (this->cpu_grmode != config->cpu_grmode)
		this->SetCPUGraphMode (config->cpu_grmode);
	if (this->cpu_hmode != config->cpu_hmode)
		this->SetCPUHistoMode (config->cpu_hmode);
	if (this->sma_smaky_too != config->sma_smaky_cpu)
		this->SetCPUMode (config->sma_smaky_cpu);
	
	this->SetSamplePeriod (config->sample_period);
	this->SetFirstMemPeriod (config->mem_period_1);
	this->SetSecondMemPeriod (config->mem_period_2);
	this->SetFirstProcPeriod (config->cpu_period_1);
	this->SetSecondProcPeriod (config->cpu_period_2);
	this->SetNetPeriod (config->net_period);
	this->SetFirstMousePeriod (config->mouse_period_1);
	this->SetSecondMousePeriod (config->mouse_period_2);
}


void
ActivitySpy::SetBox (Box box)
{
	if (box.d.dx && box.d.dy)
		this->screen_box = box;
}


void
ActivitySpy::SetZones (Card8 bitmap)
{
	this->mzones = bitmap & 0xf;
}


void
ActivitySpy::SetZoneSize (Card8 size)
{
	if (size) this->zone_size = size;
}

