/*
 *	activityspy.h
 *
 *	Memory, processor time and network spy for SMA_SMAKY
 *
 *	(C) 1994 Erik BRUCHEZ
 *
 */

#ifndef _ACTIVITY_SPY_H_
#define _ACTIVITY_SPY_H_

#include "system.h"
#include "gra.h"

class EventBox;
struct NetCountRec;
struct ConfigSet;

enum GraphMode
{
	GRAPH_MODE_BAR = 0,
	GRAPH_MODE_HISTOGRAM = 1
};

enum HistoMode
{
	HISTO_MODE_NORMAL = 0,
	HISTO_MODE_DYNAMICAL =1
};

class ActivitySpy {

private:
	
	//	Simple statistics
	
	Card32	total;
	Card32	hole;
	Card32	free;
	
	Card16	cpu_percent;
	
	//	Histograms
	
	Card16	cpu_percents[8];
	Card16	mem_percents[8];
	
	//	Modes
	
	Bool		sma_smaky_too;
	GraphMode	mem_grmode;
	GraphMode	cpu_grmode;
	HistoMode	mem_hmode;
	HistoMode	cpu_hmode;
	
	//	Sample time information
	
	Card32	sample_period;
	Card32	mem_period_1;
	Card32	mem_period_2;
	Card32	proc_period_1;
	Card32	proc_period_2;
	Card32	net_period;
	Card32	mouse_period_1;
	Card32	mouse_period_2;
	
	Card16	sleep_delay;
	
	//	Network activity information
	
	Bool		net_registered;
	const char	*net_driver_name;
	NetCountRec	*net_rec;
	
	Card32		*net_count;
	Card32		old_net_count[4];
	Card32		old_net_activity[4];
	
	//	Mouse spying information
	
	Box			screen_box;
	Card8		mzones;
	Card8		zone_size;
	
	//	Misc
	
	Bool	is_muted;
	
	EventBox *evbox;
	
	void CardToAsc (Card32 value, char *result);
	
	//	Friend process
	
	friend void SpyActivity (Card32 param);
	
public:
	ActivitySpy ();
	~ ActivitySpy ();
	
	void SetEventBox (EventBox *evbox) { this->evbox = evbox; }
	
	//	Network activity settings
	
	void NetDriverRegister (const char *net_driver_name, NetCountRec *rec);
	Bool StillRegistered () { return this->net_registered; }
	
	//	Mouse spying parameters
	
	void SetBox (Box box);
	void SetZones (Card8 bitmap);
	void SetZoneSize (Card8 size);
	
	//	Muting
	
	void MuteOn ();
	void MuteOff ();
	
	//	Global configuration
	
	void ParseConfiguration (const ConfigSet *config);
	
	//	Modes
	
	void SetCPUGraphMode (GraphMode mode);
	GraphMode GetCPUGraphMode () { return this->cpu_grmode; }
	
	void SetMemGraphMode (GraphMode mode);
	GraphMode GetMemGraphMode () { return this->mem_grmode; }
	
	void SetCPUMode (Bool sma_smaky_too) { this->sma_smaky_too = sma_smaky_too; }
	Bool GetCPUMode () { return this->sma_smaky_too; }
	
	void SetCPUHistoMode (HistoMode mode) { this->cpu_hmode = mode; }
	HistoMode GetCPUHistoMode () { return this->cpu_hmode; }
	
	void SetMemHistoMode (HistoMode mode) { this->mem_hmode = mode; }
	HistoMode GetMemHistoMode () { return this->mem_hmode; }
	
	//	Periods settings
	
	void SetSamplePeriod (Card32 period);
	Card32 GetSamplePeriod () { return this->sample_period; }
	
	void SetFirstMemPeriod (Card32 period);
	Card32 GetFirstMemPeriod () { return this->mem_period_1; }
	
	void SetSecondMemPeriod (Card32 period);
	Card32 GetSecondMemPeriod () { return this->mem_period_2; }
	
	void SetFirstProcPeriod (Card32 period);
	Card32 GetFirstProcPeriod () { return this->proc_period_1; }
	
	void SetSecondProcPeriod (Card32 period);
	Card32 GetSecondProcPeriod () { return this->proc_period_2; }
	
	void SetNetPeriod (Card32 period);
	Card32 GetNetPeriod () { return this->net_period; }
	
	void SetFirstMousePeriod (Card32 period);
	Card32 GetFirstMousePeriod () { return this->mouse_period_1; }
	
	void SetSecondMousePeriod (Card32 period);
	Card32 GetSecondMousePeriod () { return this->mouse_period_2; }
	
	//	Statistics
		
	Card16 GetCPUStats (Card8 i) { return this->cpu_percents[i]; }
	Card16 GetMemStats (Card8 i) { return this->mem_percents[i]; }
	void GetMemStats (Card32& total, Card32& hole, Card32& free);
	void GetMemStats (char *buffer);
	void GetCPUStats (Card16& percent) { percent = this->cpu_percent; }
	void GetCPUStats (char *buffer);
};

#endif /* _ACTIVITY_SPY_H_ */

