/*
 *	clock.cxx
 *
 *	Clock interface for SMA_SMAKY
 *
 *	(C) Copyright 1994, 1995 Erik BRUCHEZ
 */


#include "clock.h"
#include "fos.h"
#include "ntrel.h"
#include "str.h"
#include "lib.h"
#include "smevents.h"
#include "config.h"

#include "mon.h"

#define DISPLAY_MESSAGE_PRIORITY 6
#define BEEP_MESSAGE_PRIORITY 6

/****************************************************************/

/*
 *	Conversion table for directory name
 */

static const char tr_dir[] = {
	'\0', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', '\0', 'X', 'U',
	'A', 'A', 'E', 'E', 'E', 'E', 'I', 'I',
	'O', 'U', 'U', 'A', 'O', 'C', '', '',
	'\0', '!', '"', '#', '$', '%', '&', '\047',
	'(', ')', '*', '+', ',', '-', '.', '\0',
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', ':', ';', '<', '=', '>', '?',
	'@', 'A', 'B', 'C', 'D', 'E', 'F', 'G',
	'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W',
	'X', 'Y', 'Z', '[', '\\', ']', '^', '_',
	'`', 'A', 'B', 'C', 'D', 'E', 'F', 'G',
	'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W',
	'X', 'Y', 'Z', '{', '|', '}', '~', 'X',
	
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X',
	'X', 'X', 'X', 'X', 'X', 'X', 'X', 'X'
};

/*
 *	Conversion table for message icon number
 */

static const Card8 tr_icon[] = {
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  1,  1,  1,  2,  2,  2,
	 3,  3,  3,  4,  4,  4,  5,  5,
	 5,  6,  6,  6,  7,  7,  7,  8,
	 8,  8,  9,  9,  9, 10, 10, 10,
	11, 11, 11, 12, 12, 12, 13, 13,
	13, 14, 14, 14, 15, 15, 15,  0,
	
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0,
	 0,  0,  0,  0,  0,  0,  0,  0
};

/****************************************************************/

ClockInterface::ClockInterface ()
{
	this->ReadStatus ();
}

ClockInterface::~ClockInterface ()
{
}

void
ClockInterface::ReadStatus ()
{
	Card16 channel;
	
	Fos::Open ("#CLOCK", FOS_OPRD, &channel);
	Fos::RStatus (channel, status.Size (), &this->status);
	Fos::Close (channel);
}

void
ClockInterface::ReadClock (ClockDateAndTime *date_and_time)
{
	Card16 channel;
	Card32 length = 7;
	
	Fos::Open ("#CLOCK", FOS_OPRD, &channel);
	Fos::RdByte (channel, &length, date_and_time);
	Fos::Close (channel);
}

void
ClockInterface::Command (const char *command)
{
	Card16 channel;
	
	Fos::Open ("#CLOCK", FOS_OPWR, &channel);
	Fos::Command (channel, (char *)(command));
	Fos::Close (channel);
}

void
ClockInterface::MuteOn ()
{
	this->Command ("M");
}

void
ClockInterface::MuteOff ()
{
	this->Command ("R");
}

void
ClockInterface::DisplayClock ()
{
	this->Command ("C");
}

void
ClockInterface::DisplayMemory ()
{
	this->Command ("N");
}

void
ClockInterface::DisplayDirectory ()
{
	this->Command ("D");
}

void
ClockInterface::RemoveMessage (Card16 number)
{
	char cmd[] = { 'K', number >> 8, number & 0xff, '\0' };
	
	this->Command (cmd);
}

/****************************************************************/

void
BeepMessage (Card32 param)
{
	volatile Clock *that = (volatile Clock *)(param);
	
	//	WARNING :	"beeps repetition every 5 scrollings" mode not
	//				implemented
	
	Card8 beeps = that->msg_info.beeps;
	beeps &= 0x7f;
	
	if (beeps == 0) return;
	
	//	Values from 1 to 6 make corresponding number of beeps
	//	Values greater than 6 make "infinite" loop
	
	for (int i = 0; i < beeps; ) {
		Lib::Beep (BIPBIG);
		Ntr::Delms (40);
		
		if (that->must_stop) return;
		
		if (beeps <= 6) i++;
	}
}


void
DisplayMessage (Card32 param)
{
	volatile Clock *that = (volatile Clock *)(param);
	
	EvMessageBegin evbegin;
	EvMessageScroll evscroll;
	EvMessageEnd evend;
	
	Bool fixed_step = that->scr_fixed_step;
	
	Card16 msg_num = that->msg_id;
	
	//	Post "message beginning" event
	
	evbegin.icon_number = tr_icon[that->msg_info.icon];
	evbegin.message_id = msg_num;
	Str::Copy (evbegin.text, that->msg_text);
	
//	AfText ("DisplayMessage\r"); //PA
	if (that->evbox) that->evbox->PostEvent (evbegin);
	
	//	Wait for message length info
	
	that->sync.Wait ();
	
	//	Launch beep process if necessary
	
	Bool must_beep = (that->msg_info.beeps & 0x7f) ? TRUE : FALSE;
	
	if (must_beep) {
		Card32 pid;
		Card16 pino;
		
		Ntr::CreTask (BeepMessage, 1000, BEEP_MESSAGE_PRIORITY,
					  "BeepMessage", (Card32)(that), 0, &pid, &pino);
	}
	
	//	Do scrolling
	
	Card8 delay = ( (that->msg_info.delay) && (that->msg_info.delay <= 8) ) ?
				  that->msg_info.delay : 8;
	Card8 scr_step = 9 - delay;
	Card8 times = that->msg_info.times ? that->msg_info.times : 0xff;
	
	if (fixed_step)
		scr_step = 8;
	else
		delay = 1;
	
	Bool stop = FALSE;
	
	//	Main loop
	
	for (Card8 i = 0; i < times; ) {
		
		for (Int16 pos = that->msg_max_scr; pos >= -that->msg_max_len; pos -= scr_step) {
			
			//	Test if we must end : this is the case when our message number
			//	has disappeared from the list
			
			Bool found = FALSE;
			
			for (int j = 0; j < that->maxmess; j++)
				if (msg_num == that->del[j]) {
					found = TRUE;
					break;
				}
			
			if ( (that->must_stop) || (!found) ) stop = TRUE;
			if (stop) break;
			
			//	Send event
			
			evscroll.position = pos;
			evscroll.message_id = msg_num;
			
//			AfText ("."); //PA
			if (that->evbox) that->evbox->PostEvent (evscroll);
			
			//	Wait
			
			Ntr::Delms (delay);
		}
		
		if (stop) break;
		
		if (times != 0xff) i++;
	}
	
	//	Kill message (we do it even if the message has been already
	//	killed by someone...)
	
	that->clintf.RemoveMessage (msg_num);
	
	//	Tell the BeepMessage task to abort itself (in case it didn't do
	//	so yet).
	
	that->must_stop = TRUE;
	
	//	Post "message end" event
	
	evend.message_id = msg_num;
	
	if (that->evbox) that->evbox->PostEvent (evend);
	
	//	Wait for son's end
	
	if (must_beep) Ntr::WFAbDesc ();
	
	//	Signal end of message management
	
	that->message_received = FALSE;
}


void
Clock::StopMessageDisplaying ()
{
	this->must_stop = TRUE;
}


void
Clock::SignalMsgGeom (Card16 len, Card16 scr)
{
//	AfText ("SignalMsgGeom: "); AfX4 (len); AfText (":"); AfX4 (scr); AfCR (); //PA: plante sans !
//	asm volatile (" " : : : "d7", "d4", "a3"); //PA
	
	//	Tell DisplayMessage that it can begin scrolling
	
	this->msg_max_len = len ? len : 1;
	this->msg_max_scr = scr ? scr : 1;
	this->sync.Signal ();
}

void
Clock::TimeElapsed (Card16 seconds)
{
	if (seconds == 0) return;
	
	//	Update time
	
	Card8 sec = this->date_and_time.seconds;
	
	while (seconds--) {
		sec++;
		if ( (sec & 0x0f) > 0x09) sec = (sec & 0xf0) + 0x10;
	}
	
	Bool read = FALSE;
	
	if (this->clintf.IsUpdated ()) {
		this->clintf.UpdateAck ();
		read = TRUE;
	}
	
	if ( (sec >= 0x60) || (read) ) {
		this->redraw_all = TRUE;// provis. (chaque minute !)
		this->clintf.ReadClock (&this->date_and_time);
	} else {
		this->date_and_time.seconds = sec;
	}
	
	//	Test messages if no message currently displayed
	
	volatile Clock *that = (volatile Clock *)(this);
	
	if (that->message_received) return;
	
	Bool found;
	Card8 buffer[MSG_BUFFER_SIZE];
	Card16 num;
	
	do {
		Card32 length = MSG_BUFFER_SIZE;
		Card8 mode;
		Card32 tmpnum;
		
		//	Try to obtain a message, with null timeout. If no timeout
		//	error, the message is copied into the buffer.
		
		Ntr::SetTim (0);
//-		AfText ("<"); AfX8 (this->bal->GetBAL ()); AfText (">");
		that->bal->GetMessage (buffer, length, tmpnum, mode);
		Ntr::SetTim (0xffff);
		
		//	If timeout, there is no message to read.
		
		if (that->bal->GetError () == ERTIMO) return;
		
		if (that->bal->GetError ()) {
			AfText ("SMA_SMAKY : error receiving message D7=");
			AfX4 (that->bal->GetError ());
			AfCR ();
			return;
		}
		
		if (mode & 0x01) {
			AfText ("SMA_SMAKY : error receiving message : ");
			AfText ("wanted pointer and received a number.");
			AfCR ();
			return;
		}
		
		//	We have the message. Now get message number.
		
		that->bal->GetNumMessage (tmpnum, mode);
		num = tmpnum & 0xffff;
		
		if ((mode & 0x01) == 0) {
			AfText ("SMA_SMAKY : error receiving message : ");
			AfText ("wanted number and received a pointer.");
			AfCR ();
			return;
		}
		
		//Lib::Beep (3);
		
		//	Test if the message is in the list of messages.
		//	If not so, forget it and look for another message.
		
		found = FALSE;
		
		for (int i = 0; i < that->maxmess; i++)
			if (num == that->del[i]) {
				found = TRUE;
				break;
			}
		
	} while (!found);	//	try another message
	
	//	Wake the Smaky up ?
	
	//...
	
	//	Read message parameters.
	
	ClockMessage *message = (ClockMessage *)(buffer);
	that->msg_info = *message;
	
	Str::NCopy (that->msg_text, message->text, MSG_BUFFER_SIZE - 1);
	that->msg_text[MSG_BUFFER_SIZE - 1] = '\0';
	
	that->msg_id = num;
	that->message_received = TRUE;
	that->must_stop = FALSE;
	
	//	Create the displaying task
	
	Card32 pid;
	Card16 pino;
	
	Ntr::CreTask (DisplayMessage, 1000, DISPLAY_MESSAGE_PRIORITY,
				  "DisplayMessage", (Card32)(that), 0, &pid, &pino);
}


Clock::Clock ()
{
	this->clintf.ReadClock (&this->date_and_time);
	
	this->mode = CLOCK_NORM;
	this->blinking_col = TRUE;
	this->display_seconds = TRUE;
	this->scr_fixed_step = FALSE;
	
	this->redraw_all = TRUE;
	
	this->dir[0] = '\0';
	this->mem[0] = '\0';
	this->cpu[0] = '\0';
	
	//	Init cycle
	
	this->cycle_len = LAST_CLOCK_MODE + 1;
	for (int i = 0; i <= LAST_CLOCK_MODE; i++)
		this->cycle[i] = (ClockMode)(i);
	this->cycle_pos = 0;
	
	//	Init BAL for messages
	
	void *tmpbal;
	
	this->clintf.GetMessagesInfo (tmpbal, this->del, this->maxmess);

//	AfText ("Clock: BAL id is "); AfX8 (tmpbal); AfCR ();
	
	this->bal = new NtrBAL (tmpbal);
	
	this->message_received = FALSE;
	this->must_stop = FALSE;
	
	this->evbox = NULL;
	
	this->msg_max_len = 100;
	this->msg_max_scr = 100;
}


Clock::~Clock ()
{
/*
 *	Assuming...
 *
 *	this->must_abort = TRUE;
 *	//Ntr::WFAbDesc ();			// warning... not here !
 */
}


void
Clock::Read (ClockDateAndTime *date_and_time)
{
	*date_and_time = this->date_and_time;
}


void
Clock::AfDec2 (char *buffer, Card8 n)
{
	*buffer++ = (char)((Card8)('0') + (n >> 4));
	*buffer = (char)((Card8)('0') + (n & 0x0f));
}


Card8
Clock::BCDToDec (Card8 bcd)
{
	return (bcd >> 4) * 10 + (bcd & 0x0f);
}


Bool
Clock::CmpDir (const char *s1, const char *s2)
{
	do {
		if (*s1 != tr_dir[(Card8)(*s2++)]) return FALSE;
		if (*s1++ == 0) return TRUE;
	} while (1);
}


void
Clock::TransformDir ()
{
	char *s = this->dir;
	
	while (*s = tr_dir[(Card8)(*s)]) s++;
}


Bool
Clock::Read (char *buffer, Bool large = FALSE)
{
	static const char *full_week_day[] =
	{
		"Lundi", "Mardi", "Mercredi",
		"Jeudi", "Vendredi", "Samedi", "Dimanche", "?"
	};
	
	static const char *small_week_day[] =
	{
		"Lun", "Mar", "Mer",
		"Jeu", "Ven", "Sam", "Dim", "?"
	};
	
	static const char *small_month[] =
	{
		"Jan ", "Fv ", "Mar ", "Avr ",
		"Mai ", "Juin", "Juil", "Aot",
		"Sep ", "Oct ", "Nov ", "Dc ", "?"
	};
	
	Card8 month = this->BCDToDec (this->date_and_time.month);
	Card8 weekday = this->BCDToDec (this->date_and_time.weekday);

	if (month > 12)  month = 12;		//	affiche "?" en cas d'erreur
	if (weekday > 7) weekday = 7;		//	affiche "?" en cas d'erreur
	
	switch (this->mode) {
	
		case CLOCK_NORM:
			
			//	Write week day
//-			AfText ("CLOCK_NORM:");
			if (large) {
				Str::Copy (buffer, full_week_day[weekday - 1]);
				while (*buffer++) ;
				buffer--;
			} else {
				Str::Copy (buffer, small_week_day[weekday - 1]);
				buffer += 3;
			}
			
			*buffer++ = ' ';
			
			//	Write day
			
			this->AfDec2 (buffer, this->date_and_time.day);
			
			/*
			 *	If we want to align month position
			 *
			if (*buffer == '0') *buffer = 127;
			
			buffer += 2;
			*buffer++ = ' ';
			 */
			
			if (buffer[0] == '0') {
				buffer[0] = buffer[1];
				buffer++;
			} else 			
				buffer += 2;
			
			*buffer++ = ' ';
			
			//	Write "small" month
			
			Str::Copy (buffer, small_month[month - 1]);
			buffer += 4;
			
			*buffer++ = ' ';
			
			//	Hours, minutes and seconds
			
			char s;
			
			if (this->blinking_col)
				s = (this->date_and_time.seconds % 2) ? ':' : ' ' ;
			else
				s = ':';
			
			this->AfDec2 (buffer, this->date_and_time.hours);
			
			if (*buffer == '0') *buffer = 127;
			
			buffer += 2;
			*buffer++ = s;
			this->AfDec2 (buffer, this->date_and_time.minutes);
			buffer += 2;
			*buffer++ = s;
			this->AfDec2 (buffer, this->date_and_time.seconds);
			buffer += 2;
			*buffer++ = '\0';
			
			break;
		
		case CLOCK_DATE:
//-			AfText ("CLOCK_DATE:");
			
			//	Write week day
			
			if (large) {
				Str::Copy (buffer, full_week_day[weekday - 1]);
				while (*buffer++) ;
				buffer--;
			} else {
				Str::Copy (buffer, small_week_day[weekday - 1]);
				buffer += 3;
			}
			
			*buffer++ = ' ';
			
			//	Write day
			
			this->AfDec2 (buffer, this->date_and_time.day);
			
			if (buffer[0] == '0') {
				buffer[0] = buffer[1];
				buffer++;
			} else 			
				buffer += 2;
			
			*buffer++ = ' ';
			
			//	Write "small" month
			
			Str::Copy (buffer, small_month[month - 1]);
			
			buffer += 4;
			*buffer++ = ' ';
			
			//	Write year
			
			if (this->date_and_time.year < 0x80) {	// to be verified !
				*buffer++ = '2';
				*buffer++ = '0';
			} else {
				*buffer++ = '1';
				*buffer++ = '9';
			}
			
			this->AfDec2 (buffer, this->date_and_time.year);
			buffer += 2;
			*buffer = '\0';
			
			break;
		
		case CLOCK_DIR:
//-			AfText ("CLOCK_DIR:");
			
			Card16 kbd_num;
			Card16 proc_num;
			const char *proc_name;
			const char *dir_name;
			
			Lib::VAssKey (kbd_num);
			Lib::SrcKNP (kbd_num, proc_num, proc_name);
			Fos::GSDir (dir_name, proc_num);
			
			if (!this->CmpDir (this->dir, dir_name)) {
				Str::NCopy (this->dir,dir_name, DIR_BUFFER_SIZE - 1);
				this->dir[DIR_BUFFER_SIZE - 1] = '\0';
				this->TransformDir ();
				this->redraw_all = TRUE;
			} else
				this->redraw_all = FALSE;
			
			Str::Copy (buffer, this->dir);
			
			break;
			
		case CLOCK_MEM:
//-			AfText ("CLOCK_MEM:");
			
			Str::Copy (buffer, this->mem);
			break;
		
		case CLOCK_CPU:
//-			AfText ("CLOCK_CPU:");
			
			Str::Copy (buffer, this->cpu);
			break;
			
		default:
//-			AfText ("default:");
			
			buffer[0] = '\0';
			
			break;
	}
	
//-	AfText ("Buffer: "); AfText (org_buffer); AfCR ();
	
	Bool r = this->redraw_all;
	this->redraw_all = FALSE;
	return r;
}


void
Clock::SetMemText (const char *text)
{
	Str::NCopy (this->mem, text, MEM_BUFFER_SIZE - 1);
	this->mem[MEM_BUFFER_SIZE - 1] = '\0';
}


void
Clock::SetCPUText (const char *text)
{
	Str::NCopy (this->cpu, text, CPU_BUFFER_SIZE - 1);
	this->cpu[CPU_BUFFER_SIZE - 1] = '\0';
}

void
Clock::SetColonsMode (Bool blinking)
{
	this->blinking_col = blinking;
}


void
Clock::SetSecondsMode (Bool display_seconds)
{
	this->display_seconds = display_seconds;
	this->redraw_all = TRUE;
}

void
Clock::SetScrollMode (Bool fixed_step)
{
	this->scr_fixed_step = fixed_step;
}


void
Clock::ParseConfiguration (const ConfigSet *config)
{
	this->SetColonsMode (config->blinking_col);
	
	if (config->cycle_len) {
		this->cycle_len = config->cycle_len;
		for (int i = 0; i < config->cycle_len; i++) {
			if (config->cycle[i] <= LAST_CLOCK_MODE)
				this->cycle[i] = config->cycle[i];
			else
				this->cycle[i] = CLOCK_NORM;
		}
		this->SetMode ((ClockMode)(this->cycle[0]));
		this->cycle_pos = 0;
	}
}


void
Clock::NextMode ()
{
	if (this->cycle_len > 1) {
		if (++this->cycle_pos >= this->cycle_len)
			this->cycle_pos = 0;
		this->SetMode ((ClockMode)(this->cycle[this->cycle_pos]));
	}
}

void
Clock::PrevMode ()
{
	if (this->cycle_len > 1) {
		if (this->cycle_pos-- == 0)
			this->cycle_pos = this->cycle_len - 1;
		this->SetMode ((ClockMode)(this->cycle[this->cycle_pos]));
	}
}

/****************************************************************/

