/*
 *	clock.h
 *
 *	Clock interface for SMA_SMAKY
 *
 *	(C) Copyright 1994,  Erik BRUCHEZ
 */


#ifndef _SMA_CLOCK_H_
#define _SMA_CLOCK_H_

#include "system.h"
#include "ntrel.h"

/****************************************************************/

/*
 *	This structure gives an access to the information permanently
 *	updated by the clock driver.
 *	One instance of this structure is read when an instance of
 *	the class 'ClockInterface' is created.
 */

struct ClockStatus
{
	volatile	Card16	status;
	volatile	Card8	*bits;
	volatile	void	*bal;
	volatile	Card16	*del;
	volatile	Card16	maxmess;
	
public:
	
	Bool IsBatteryOutOfOrder () { return (this->status == 'V'); }
	Bool IsUpdated () { return (this->bits[0] & 0x01) != 0; }
	Bool IsMuted () { return (this->bits[0] & 0x02) != 0; }
	
	void UpdateAck () { this->bits[0] &= ~0x01; }
	
	Bool IsClock () { return (this->bits[0] & 0x0c) == 0; }
	Bool IsMemory () { return (this->bits[0] & 0x04) != 0; }
	Bool IsDirectory () { return (this->bits[0] & 0x08) != 0; }
	
	void GetMessagesInfo (void*& bal, Card16*&del, Card16& maxmess)
	{
		bal = (void *)(this->bal);
		del = (Card16 *)(this->del);
		maxmess = (Card16)(this->maxmess);
	}
	
	int Size () { return sizeof (*this); }
};

/****************************************************************/

/*
 *	This structure contains the date and time information given
 *	by the clock driver.
 */

struct ClockDateAndTime
{
	volatile Card8	seconds;
	volatile Card8	minutes;
	volatile Card8	hours;
	volatile Card8	day;
	volatile Card8	month;
	volatile Card8	year;
	volatile Card8	weekday;
};

/****************************************************************/

/*
 *	This structure is returned by the clock driver on a BAL
 *	when a message is available.
 */

struct ClockMessage
{
	Card8	times;
	Card8	beeps;
	Card8	delay;
	Card8	icon;
	char	text[0];
};

/****************************************************************/

/*
 *	This class gives a simple access to the clock driver.
 */

class ClockInterface {

private:
	void ReadStatus ();
	void Command (const char *command);
	
	ClockStatus			status;
public:
	
	ClockInterface ();
	~ ClockInterface ();
	
	void ReadClock (ClockDateAndTime *date_and_time);
	
	//	Commands
	
	void MuteOn ();
	void MuteOff ();
	void DisplayClock ();
	void DisplayMemory ();
	void DisplayDirectory ();
	void RemoveMessage (Card16 number);
	
	//	Inquiries
	
	Bool IsBatteryOutOfOrder ()
	{
		return this->status.IsBatteryOutOfOrder ();
	}
	
	Bool IsMuted () { return this->status.IsMuted (); }
	Bool IsUpdated () { return this->status.IsUpdated (); }
	
	void UpdateAck () { this->status.UpdateAck (); }
	
	Bool IsClock () { return this->status.IsClock (); }
	Bool IsMemory () { return this->status.IsMemory (); }
	Bool IsDirectory () { return this->status.IsDirectory (); }
	
	void GetMessagesInfo (void*& bal, Card16*& del, Card16& maxmess)
	{
		this->status.GetMessagesInfo (bal, del, maxmess);
	}
};

/****************************************************************/

/*
 *	This class gives a high level access to the clock driver.
 *	It is automatically updated second by second.
 */

#define DIR_BUFFER_SIZE (100)
#define MEM_BUFFER_SIZE (50)
#define CPU_BUFFER_SIZE (50)
#define MSG_BUFFER_SIZE (200)

enum ClockMode
{
	CLOCK_NORM = 0,				// ex.: Mercredi 11 Jan 18:10:15
	CLOCK_DATE = 1,				// ex.: Mercredi 11 Jan 1995
	CLOCK_DIR = 2,				// ex.: #MM0:USERS:
	CLOCK_MEM = 3,				// ex.: 6'504'824 / 6'452'414
	CLOCK_CPU = 4,				// ex.: Processeur :  46 %
};

#define LAST_CLOCK_MODE (CLOCK_CPU)

class EventBox;
class NtrBAL;

struct ConfigSet;

class Clock {

private:
	ClockInterface		clintf;
	ClockDateAndTime	date_and_time;
	
	ClockMode			mode;
	
	Bool				blinking_col;
	Bool				display_seconds;
	Bool				scr_fixed_step;
	
	Bool				redraw_all;
	
	char				dir[DIR_BUFFER_SIZE];
	char				mem[MEM_BUFFER_SIZE];
	char				cpu[CPU_BUFFER_SIZE];
	
	Card8				cycle_len;
	ClockMode			cycle[12];
	Card8				cycle_pos;
	
	//	Messages management

	NtrBAL				*bal;
	
	Card16				*del;
	Card16				maxmess;
	
	char				msg_text[MSG_BUFFER_SIZE];
	ClockMessage		msg_info;
	Card16				msg_id;
	
	Bool				message_received;
	Bool				must_stop;
	
	EventBox			*evbox;
	
	Card16				msg_max_len;
	Card16				msg_max_scr;
	
	//	Strings utilities
	
	void AfDec2 (char *buffer, Card8 n);
	Card8 BCDToDec (Card8 bcd);
	Bool CmpDir (const char *s1, const char *s2);
	void TransformDir ();
	
	//	Friend process
	
	NtrSync				sync;
	
	friend void DisplayMessage (Card32 param);
	friend void BeepMessage (Card32 param);

public:
	Clock ();
	~ Clock ();
	
	void SetEventBox (EventBox *evbox) { this->evbox = evbox; }
	
	//	
	
	Bool IsMuted () { return this->clintf.IsMuted (); }
	
	void TimeElapsed (Card16 seconds);
	
	void Read (ClockDateAndTime *date_and_time);
	Bool Read (char *buffer, Bool large = FALSE);
	
	void SetMemText (const char *text);
	void SetCPUText (const char *text);
	
	//	Global configuration
	
	void ParseConfiguration (const ConfigSet *config);
	
	//	Modes
	
	void SetColonsMode (Bool blinking);
	Bool GetColonsMode () { return this->blinking_col; }
	
	void SetSecondsMode (Bool display_seconds);
	Bool GetSecondsMode () { return this->display_seconds; }
	
	void SetScrollMode (Bool fixed_step);
	Bool GetScrollMode ()  { return this->scr_fixed_step; }
	
	//	Modes for reading in the buffer
	
	void NextMode ();
	void PrevMode ();
	
	void SetMode (ClockMode mode) { this->mode = mode; }
	ClockMode GetMode () { return this->mode; }
	
	//	Messages management
	
	void SignalMsgGeom (Card16 len, Card16 scr);
	void StopMessageDisplaying ();
};

/****************************************************************/

#endif /* _SMA_CLOCK_H_ */

