/*
 *	gra.cxx
 *
 *	Gra2 and display descriptor interface for SMA_SMAKY
 *
 *	(C)	1994 Daniel MARMIER
 */

#include "system.h"
#include "gra.h"

#include "mon.h"

void
Gra::SetClipBox (Box clip)
{
	this->dd->clip_box = clip;
	this->dd->w.p.y = this->dd->wind_box.o.y + clip.o.y;
	this->dd->w.p.x = this->dd->wind_box.o.x + clip.o.x;
	this->dd->w.q.y = this->dd->w.p.y + clip.d.dy;
	this->dd->w.q.x = this->dd->w.p.x + clip.d.dx;
	
	this->err = 0;
}


void
Gra::Line (Box line)
{
	asm (	"movel	%0,a2\n\t"
			"movel	%1,d4\n\t"
			"movel	%2,d3\n\t"
			"movew	%3,d1\n\t"
			".word	0xaa86\n\t"		// GRA2_LINE
		:
		: "g" (dd), "g" (line.o), "g" (line.d), "g" (mode)
		: "d1", "d3", "d4", "d7", "a2" );
}


void
Gra::Rect (Box rect)
{
	Box b = rect;
	b.d.dy = 0;
	
	this->Line (b);
	
	b.d.dy = rect.d.dy;
	b.o.x += rect.d.dx - 1;
	b.d.dx = 0;
	
	this->Line (b);
	
	b.d.dx = -rect.d.dx;
	b.o.y += rect.d.dy - 1;
	b.d.dy = 0;
	
	this->Line (b);
	
	b.d.dy = -rect.d.dy;
	b.o.x = rect.o.x;
	b.d.dx = 0;
	
	this->Line (b);
}


void
Gra::RasterMono (Box box, Pos pos, const void* raster, Card16 rbytes)
{
	asm (	"movel	%0,a2		\n\t"
			"movel	%1,d3		\n\t"
			"movel	%2,d2		\n\t"
			"movel	%3,a3		\n\t"
			"movew	%4,d5		\n\t"
			"subl	a4,a4		\n\t"
			"movel	%5,d4		\n\t"
			"movew	%6,d1		\n\t"
			".word	0xaa80		\n\t"		// GRA2_RASTER
		:
		: "g" (dd), "g" (box.o), "g" (box.d), "g" (raster), "g" (rbytes),
		  "g" (pos), "g" (mode)
		: "d1", "d2", "d3", "d4", "d5", "d7", "a2", "a3", "a4" );
}

void
Gra::RasterColor (Box box, Pos pos, const void *raster, Card16 rbytes)
{
	const void* dd_abs = dd->abs;
	Card16      dd_iiy = dd->iiy;
	const void* ra_abs = raster; 
	Card16      ra_iiy = rbytes;
	
	Bool special = (rbytes == 0) && (dd_iiy != 0);
	
	if (special) {
		
		//	Source is in a smart display (which uses drawing functions and
		//	prohibits direct access). This is troublesome.
		//
		//	Swap source and destination; this works only if the destination
		//	is in memory.
		
//		AfText ("Swap display : ");
//		AfX8 (dd->abs); AfSpace (); AfX4 (dd->iiy); AfSpace ();
//		AfX8 (raster); AfSpace (); AfX4 (rbytes); AfCR ();
		
		dd->abs = raster;
		dd->iiy = rbytes;
		ra_abs  = dd_abs;
		ra_iiy  = dd_iiy;
		
		Pos src = box.o;
		
		box.o = pos;
		pos   = src;
	}
	
	if (special) {
		asm (	"movel	%0,a2		\n\t"
				"movel	%1,d4		\n\t"
				"movel	%2,d2		\n\t"
				"movel	%3,a4		\n\t"
				"movew	%4,d6		\n\t"
				"subl	a3,a3		\n\t"
				"movel	%5,d3		\n\t"
				"movew	%6,d1		\n\t"
				"oril	#0x1800,d1	\n\t"
				".word	0xaa99		\n\t"		// GRA2_CRASTER
				
			:
			: "g" (dd), "g" (box.o), "g" (box.d), "g" (ra_abs), "g" (ra_iiy),
			  "g" (pos), "g" (mode)
			: "d1", "d2", "d3", "d4", "d6", "d7", "a2", "a3", "a4" );
	} else {
		asm (	"movel	%0,a2		\n\t"
				"movel	%1,d3		\n\t"
				"movel	%2,d2		\n\t"
				"movel	%3,a3		\n\t"
				"movew	%4,d5		\n\t"
				"subl	a4,a4		\n\t"
				"movel	%5,d4		\n\t"
				"movew	%6,d1		\n\t"
				"oril	#0x1800,d1	\n\t"
				".word	0xaa99		\n\t"		// GRA2_CRASTER
				
			:
			: "g" (dd), "g" (box.o), "g" (box.d), "g" (ra_abs), "g" (ra_iiy),
			  "g" (pos), "g" (mode)
			: "d1", "d2", "d3", "d4", "d5", "d7", "a2", "a3", "a4" );
	}
	
	if (special) {
		
		//	Restore the proper graphic information in the descriptor.
		
		dd->abs = dd_abs;
		dd->iiy = dd_iiy;
	}
}

void
Gra::Icon (Pos pos, void* icon)
{
	Card8*	d = (Card8*)(icon);
	void*	data = icon + 2;
	Dim		dim;

	dim.dx = d[0];
	dim.dy = d[1];

	Card16 rbytes = (dim.dx + 7) / 8;
	
	asm (	"movel	%0,a2\n\t"
			"clrl	d3\n\t"
			"movel	%1,a3\n\t"
			"movew	%2,d5\n\t"
			"movel	%3,d4\n\t"
			"subl	a4,a4\n\t"
			"movel	%4,d2\n\t"
			"movew	%5,d1\n\t"
			".word	0xaa80\n\t"
		:
		: "g" (dd), "g" (data), "g" (rbytes), "g" (pos), "g" (dim), "g" (mode)
		: "d1", "d2", "d3", "d4", "d5", "d7", "a2", "a3", "a4" );
}

void
Gra::Clear (Box zone)
{
	asm (	"movel	%0,a2\n\t"
			"movel	%1,d4\n\t"
			"movel	%2,d3\n\t"
			".word	0xaa85\n\t"
		:
		: "g" (dd), "g" (zone.o), "g" (zone.d)
		: "d3", "d4", "d7", "a2" );
}

void
Gra::Set (Box zone)
{
	asm (	"movel	%0,a2\n\t"
			"movel	%1,d4\n\t"
			"movel	%2,d3\n\t"
			".word	0xaa92\n\t"
		:
		: "g" (dd), "g" (zone.o), "g" (zone.d)
		: "d3", "d4", "d7", "a2" );
}

void
Gra::Trame (Box zone, void *trame)
{
	asm (	"movel	%0,a2\n\t"
			"movel	%1,d4\n\t"
			"movel	%2,d3\n\t"
			"movew	%3,d1\n\t"
			"movel	%4,a4\n\t"
			".word	0xaa82\n\t"
		:
		: "g" (dd), "g" (zone.o), "g" (zone.d), "g" (mode), "g" (trame)
		: "d1", "d3", "d4", "d7", "a2", "a4" );
}

void
Gra::DecodeImage (void* decoded, const void* encoded, Image* header) const
{
	asm (	"movel	%1,a3\n\t"
			"movel	%2,a4\n\t"
			"movel	%3,a2\n\t"
			".word	0xaa96\n\t"		// GRA2_DECOIMA
			"movew	d7,%0\n\t"
			: "=g" (err)
			: "g" (decoded), "g" (encoded), "g" (header)
			: "d7", "a2", "a3", "a4" );
}

void
Gra::SetGencar (Gencar *gencar)
{
	if (gencar == 0) {
		AfText ("Warning: setting 0 gencar.\r");
	}
	this->dd->pgc = gencar;
	
	this->err = 0;
}

void
Gra::DrawChar (Pos& p, char c, Card8 hole = 0)
{
	Gencar *genc = (Gencar *)(this->dd->pgc);
	Card8 *widths = (Card8 *)(genc) + 256 + genc->widths;
	Card8 *matrices = (Card8 *)(genc) + 256 + genc->matrices;
	
	Pos q = p;
	q.y -= genc->ascent;
	
	Card8 d = (Card8)(c);
	
	if (d >= genc->first && d <= genc->last) {
		
		d -= genc->first;
		
		if (widths[d]) {
			
			Box origin = { { 0, 0}, { genc->mat_h, widths[d]} };
			
			this->RasterMono (origin, q, matrices + genc->mat_l * d, genc->mat_w);
		}
		
		p.x += widths[d] + hole;
	}
	
	this->err = 0;
}

inline void
Gra::SpeedRaster(const void *source, Card16 swidth, void *dest, Card16 dwidth,
				 Card16 posx, Card16 dx, Card16 dy)
{
	register const void *_source asm("a0");
	register void *_dest asm("a1");
	register Card32 _swidth asm("a2");
	register Card32 _dwidth asm("a3");
	register Card32 _posx asm("d0");
	register Card16 _dx asm("d1");
	register Card16 _dy asm("d2");
	
	_source = source;
	_dest = dest;
	_swidth = swidth;
	_dwidth = dwidth;
	_posx = posx;
	_dx = dx;
	_dy = dy;
	
	asm volatile ("
		bras	2f
	1:
		bfextu	a0@{#0:d1},d3
		bfins	d3,a1@{d0:d1}
		addl	a2,a0
		addl	a3,a1
	2:
		dbf		d2,1b"
	:
	: "g" (_source), "g" (_dest), "g" (_swidth), "g" (_dwidth), "g" (_posx), "g" (_dx), "g" (_dy)
	: "d2", "d3", "a0", "a1" );
}

void
Gra::DrawText (Pos p, const char *text, Card8 hole = 0)
{
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	const Card8 *widths = (const Card8 *)(genc) + 256 + genc->widths;
	const Card8 *matrices = (const Card8 *)(genc) + 256 + genc->matrices;
	
	p.y -= genc->ascent;
	
	Box origin = { { 0, 0}, { genc->mat_h, 0} };
	
	Card8 c;
	
	while (c = (Card8)(*text++)) {
		if (c >= genc->first && c <= genc->last) {
		
			c -= genc->first;
			origin.d.dx = widths[c];
			this->RasterMono (origin, p, matrices + genc->mat_l * c, genc->mat_w);
			p.x += widths[c] + hole;
        }
    }
    
    this->err = 0;
}

void
Gra::DrawText (Pos p, const char *text, void *buffer, Card16 bwidth)
{
	if (buffer == NULL) {
		this->err = 1;
		return;
	}
	
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	const Card8 *widths = (const Card8 *)(genc) + 256 + genc->widths;
	const Card8 *matrices = (const Card8 *)(genc) + 256 + genc->matrices;
	
	Card8 c;
	Card16 px = 0;
	
	while (c = (Card8)(*text++)) {
		if (c >= genc->first && c <= genc->last) {
		
			c -= genc->first;
			SpeedRaster (matrices + genc->mat_l * c, genc->mat_w, buffer, bwidth,
						 px, widths[c], genc->mat_h);
			px += widths[c];
        }
    }
    
    p.y -= genc->ascent;
	Box origin = { { 0, 0}, { genc->mat_h, px} };
	
	this->RasterMono (origin, p, buffer, bwidth);
    
    this->err = 0;
}

void
Gra::DrawCText (Box b, const char *text, JustMode mode = JUST_LEFT, Card8 hole = 0)
{
	const Card16 ascent = this->dd->pgc->ascent;
	const Card16 descent = this->dd->pgc->descent;
	
	b.o.y += (b.d.dy - ascent - descent) / 2 + ascent;
	
	this->DrawText (b.o, text, hole);
}

void
Gra::DrawCText (Box b, const char *text, void *buffer, Card16 bwidth, JustMode mode = JUST_LEFT)
{
	const Card16 ascent = this->dd->pgc->ascent;
	const Card16 descent = this->dd->pgc->descent;
	
	b.o.y += (b.d.dy - ascent - descent) / 2 + ascent;
	
	this->DrawText (b.o, text, buffer, bwidth);
}

Card32
Gra::TextWidth (const char *text)
{
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	const Card8 *widths = (const Card8 *)(genc) + 256 + genc->widths;
	
	Card8 c;
	Card32 width = 0;
	
	while (c = (Card8)(*text++)) {
		if (c >= genc->first && c <= genc->last) {
			c -= genc->first;
			width += widths[c];
        }
    }
    
    this->err = 0;
    return width;
}

Card32
Gra::CharWidth (char c)
{
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	const Card8 *widths = (const Card8 *)(genc) + 256 + genc->widths;
	
	Card32 width = 0;
	unsigned char d = (unsigned char)(c);
	
	if (d >= genc->first && d <= genc->last) {
		d -= genc->first;
		width += widths[d];
	}
    
    this->err = 0;
    return width;
}

void
Gra::DrawBText (const char *text, void *buffer, Card16 bwidth)
{
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	const Card8 *widths = (const Card8 *)(genc) + 256 + genc->widths;
	const Card8 *matrices = (const Card8 *)(genc) + 256 + genc->matrices;
	
	Card8 c;
	Card16 px = 0;
	
	while (c = (Card8)(*text++)) {
		if (c >= genc->first && c <= genc->last) {
		
			c -= genc->first;
			SpeedRaster (matrices + genc->mat_l * c, genc->mat_w, buffer, bwidth,
						 px, widths[c], genc->mat_h);
			px += widths[c];
        }
    }
    
    this->err = 0;
}

Card8
Gra::GetGencarHeight () const
{
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	return genc->mat_h;
}

Card16
Gra::GetGencarAscent () const
{
	const Gencar *genc = (const Gencar *)(this->dd->pgc);
	return genc->ascent;
}

