/*
 *	iconsman.cxx
 *
 *	User icons manager for SMA_SMAKY
 *
 *	(C)	1995 Erik BRUCHEZ
 *
 */

#include "iconsman.h"
#include "usericons.h"

/****************************************************************/

/*
 *	Static members
 */

Card8		IconsMan::max_icons_num = 0;
Card8		IconsMan::sizes_num = 0;
Card8		*IconsMan::sizes_list = NULL;
Card8		IconsMan::icons_space = 0;

IconInfo	*IconsMan::infos = NULL;

/****************************************************************/

/*
 *	Local data structure
 */

struct IconInfo
{
	Bool		is_free;
	Bool		is_visible;
	Card8		width;			//	in simple icons
	void		**data;			//	one data description per size
	
};

/****************************************************************/

IconsMan::IconsMan (Gra *gra)
{
	this->gra = gra;
	
	Pos p = { 0, 0 };
	
	this->pos = p;
	this->size = 0;
	
}


IconsMan::~IconsMan ()
{
	/*
	 *	Not implemented yet...
	 */
}

/****************************************************************/

void
IconsMan::SetIconsSizes (Card8 number, const Card8 *list)
{
	if (number) {
		IconsMan::sizes_num = number;
		IconsMan::sizes_list = new Card8[number];
		for (int i = 0; i < number; i++) IconsMan::sizes_list[i] = list[i];
	}
}


void
IconsMan::SetMaxIconsNum (Card8 num)
{
	if (num) {
		IconsMan::max_icons_num = num;
		
		IconsMan::infos = new IconInfo[num];
		
		IconInfo info;
		
		info.is_free = TRUE;
		info.is_visible = FALSE;
		info.width = 0;
		
		for ( int i = 0; i < num; i++) {
			IconsMan::infos[i] = info;
			IconsMan::infos[i].data = (void **)(new Card32[IconsMan::sizes_num]);// static...
			void **data = IconsMan::infos[i].data;
			for (int j = 0; j < IconsMan::sizes_num; j++) data[j] = NULL;
		}
	}
}


void
IconsMan::SetSpaceBetweenIcons (Card8 space)
{
	IconsMan::icons_space = space;
}

/****************************************************************/

/*
 *	Open an icon. A location in the row is reserved for the icon.
 *	The width is in "simple icon" units. It can't be greater
 *	than the maximal number of icons. If the allocation succeeded,
 *	a valid icon identifier is returned (zero in case of failure).
 */

Card32
IconsMan::Open (Card8 width)
{
	if ( (width) && (width <= IconsMan::max_icons_num) ) {
		for (int i = 0; i < (IconsMan::max_icons_num - width + 1); i++) {
			if (IconsMan::infos[i].is_free) {
				
				//	One location free...
				
				if (width > 1) {
					
					//	Look for the rest...
					
					Bool ok = TRUE;
					
					for (int j = 1; j < width; j++) {
						if (!(IconsMan::infos[i + j].is_free)) {
							ok = FALSE;
							break;
						}
					}
					
					if (ok == FALSE) continue;
				}
				
				//	Found
				
				for (int j = 0; j < width; j++) {
					IconsMan::infos[i + j].is_free = FALSE;
					IconsMan::infos[i + j].is_visible = FALSE;
					IconsMan::infos[i + j].width = 0;
					
					for (int k = 0; k < IconsMan::sizes_num; k++)
						IconsMan::infos[i + j].data[k] = NULL;
				}
				
				IconsMan::infos[i].width = width;
				
				return i + 1;
			}
		}
	}
	
	//	Allocation failed
	
	return 0;
}


Bool
IconsMan::IdValid (Card32 id)
{
	return ( (id)
		  && (id <= IconsMan::max_icons_num)
		  && !(IconsMan::infos[id - 1].is_free)
		  && (IconsMan::infos[id - 1].width) );
}


/*
 *	Close an icon. If the identifier is valid, deallocate the
 *	space occupated by the icon and free the icon data.
 */

void
IconsMan::Close (Card32 id)
{
	if (IconsMan::IdValid (id)) {
		id--;
		
		IconInfo *info = &IconsMan::infos[id];
		
		//	Delete data
		
		for (int i = 0; i < IconsMan::sizes_num; i++) {
			if (info->data[i]) {
				delete info->data[i];
				info->data[i] = NULL;
			}
		}
		
		//	Deallocate
		
		for (i = 0; i < info->width; i++)
			IconsMan::infos[id + i].is_free = TRUE;
	}
}


void
IconsMan::GetIconsSizes (Card8 buffer_size, Card8 *buffer)
{
	if (buffer_size && buffer && IconsMan::sizes_num && IconsMan::sizes_list)
		for (int i = 0; (i < buffer_size) && (i < IconsMan::sizes_num); i++)
			buffer[i] = IconsMan::sizes_list[i];
}


/*
 *	Set icon data. If the identifier is valid, set the icon data for
 *	the given size number. The data is copied. If icon is NULL, the
 *	data for the given size is disabled. The real size of the icon
 *	(found in the icon data) must be adequate.
 */

void
IconsMan::SetIcon (Card32 id, Card8 size_num, const void *icon)
{
	if (IconsMan::IdValid (id)) {
		id--;
		
		if (size_num < IconsMan::sizes_num) {
			
			//	Valid size
			
			IconInfo *info = &IconsMan::infos[id];
			
			//	Delete old data
			
			if (info->data[size_num]) {
				delete info->data[size_num];
				info->data[size_num] = NULL;
			}
			
			if (icon) {
				
				//	We must copy the data
				
				const Card8 *ic = (const Card8 *)(icon);
				Card32 size = (ic[0] + 7) / 8 * ic[1];
				
				if (size) {
					Card32 s = size + 2;
					info->data[size_num] = new char[s];
					if (info->data[size_num]) {
						Card8 *dst = (Card8 *)(info->data[size_num]);
						while (s--) *dst++ = *ic++;
					}
				}
			}
		}
	}
}


void
IconsMan::Hide (Card32 id)
{
	if (IconsMan::IdValid (id)) {
		id--;
		
		IconsMan::infos[id].is_visible = FALSE;
	}
}


void
IconsMan::Show (Card32 id)
{
	if (IconsMan::IdValid (id)) {
		id--;
		
		IconsMan::infos[id].is_visible = TRUE;
	}
}

/****************************************************************/

void
IconsMan::SelectSize (Card8 size_num)
{
	if (size_num < IconsMan::sizes_num)
		this->size = size_num;
}


void
IconsMan::SetFirstIconPos (Pos pos)
{
	this->pos = pos;
}


void
IconsMan::Draw ()
{
	
}


void
IconsMan::DrawIcon (Card32 id)
{
	if (IconsMan::IdValid (id)) {
		id--;
		
		Pos pos = this->pos;
		Card8 size = IconsMan::sizes_list[this->size];
		
		pos.x += (IconsMan::icons_space + size) * id;
		/*
		if ()
		
		this->gra->Icon (pos, );
		*/
	}
}

/****************************************************************/

