/*
 *	lookman.cxx
 *
 *	Little look manager for SMA_SMAKY
 *
 *	(C)	1995 Erik BRUCHEZ
 */

#include "lookman.h"
#include "fos.h"
#include "res.h"
#include "resources.h"
#include "ntrel.h"

#include "mon.h"

extern ResType rtyp_image;

extern "C" void* memcpy (void* dest, const void* source, unsigned long n);

#if 0
static Bool
getenv (const char* x)
{
	AfText (x);
	AfText (" -> ");
	Card16 err;
	asm (	"movel	%1,a3\n\t"
			"movel  #0,d3\n\t"
			".long	0x4E45001C\n\t"
			"movew	d7,%0\n\t"
		: "=g" (err)
		: "g" (x)
		: "d3", "d4", "d7", "a3", "a4" );
	
	AfX4 (err);
	AfCR ();
	return err == 0x0000;
}
#endif

#define WAIT_FOREVER Ntr::Dels (0xffff)

/****************************************************************/

/*
 *	Numbers of the system CRET relative colors.
 */

enum RelativeColor
{
	COLOR_PPWHITE	= 10,		//	white
	COLOR_PPLGRAY	= 7,		//	light gray
	COLOR_PPMGRAY	= 8,		//	middle gray
	COLOR_PPDGRAY	= 9,		//	dark gray
	COLOR_PPBLACK	= 11,		//	black
	
	COLOR_SSBGMENU	= 0x1201,	//	menu background
	COLOR_SSBGSK	= 0x1202,	//	softkeys background
	COLOR_SSINSK	= 0x1203,	//	softkeys interior
	COLOR_SSMEM		= 0x1204,	//	memory jauge
	COLOR_SSCPU		= 0x1205,	//	cpu jauge
	COLOR_SSRED		= 0x1206,	//	adornment 1 (red)
	COLOR_SSGREEN	= 0x1207,	//	adornment 2 (green)
	COLOR_SSBLUE	= 0x1208,	//	adornment 3 (blue)
	COLOR_SSCLTXT	= 0x1209,	//	clock text
	COLOR_SSCAPSTXT	= 0x120a,	//	"caps" text
	COLOR_SSSKTXT	= 0x120b,	//	softkeys text
	COLOR_SSMSGTXT	= 0x120c,	//	messages text
	COLOR_SSSKBARS	= 0x120d,	//	softkeys bars
	COLOR_SSNET		= 0x120e,	//	network arrows
	COLOR_SSBGMEM	= 0x120f,	//	memory jauge background
	COLOR_SSBGCPU	= 0x1210	//	cpu jauge background
};

/****************************************************************/

/*
 *	These are RGB default value for the relative colors used
 *	by SMA_SMAKY. The first element corresponds to the
 *	SSWHITE color, and the following elements are the
 *	following relative colors.
 */

static const GColor default_colors[SS_COL_NUM] = {
	{ 0xffff, 0xffff, 0xffff },	//	white
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	light gray
	{ 0xabab, 0xabab, 0xabab },	//	middle gray
	{ 0x8888, 0x8888, 0x8888 },	//	dark gray
	{ 0, 0, 0 },				//	black
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	menu background
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	softkeys background
	{ 0xffff, 0xffff, 0xffff },	//	softkeys interior
	{ 0, 0, 0 },				//	memory jauge
	{ 0, 0, 0 },				//	cpu jauge
	{ 0xffff, 0, 0 },			//	adornment 1 (red)
	{ 0, 0xffff, 0},			//	adornment 2 (green)
	{ 0, 0, 0xffff },			//	adornment 3 (blue)
	{ 0, 0, 0 },				//	clock text
	{ 0, 0, 0 },				//	"caps" text
	{ 0, 0, 0 },				//	softkeys text
	{ 0, 0, 0 },				//	messages text
	{ 0, 0, 0 },				//	softkeys bars
	{ 0, 0, 0 },				//	network arrows
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	memory jauge background
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	cpu jauge background
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	gray 1
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	gray 2
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	gray 3
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	gray 4
	{ 0xd5d5, 0xd5d5, 0xd5d5 },	//	gray 5
};

/****************************************************************/

/*
 *	This array maps the colors to their system relative
 *	color number;
 */

static const RelativeColor map_relcol[SS_COL_AUTO_NUM] = {
	COLOR_PPWHITE,		//	white
	COLOR_PPLGRAY,		//	light gray
	COLOR_PPMGRAY,		//	middle gray
	COLOR_PPDGRAY,		//	dark gray
	COLOR_PPBLACK,		//	black
	COLOR_SSBGMENU,		//	menu background
	COLOR_SSBGSK,		//	softkeys background
	COLOR_SSINSK,		//	softkeys interior
	COLOR_SSMEM,		//	memory jauge
	COLOR_SSCPU,		//	cpu jauge
	COLOR_SSRED,		//	adornment 1 (red)
	COLOR_SSGREEN,		//	adornment 2 (green)
	COLOR_SSBLUE,		//	adornment 3 (blue)
	COLOR_SSCLTXT,		//	clock text
	COLOR_SSCAPSTXT,	//	"caps" text
	COLOR_SSSKTXT,		//	softkeys text
	COLOR_SSMSGTXT,		//	messages text
	COLOR_SSSKBARS,		//	softkeys bars
	COLOR_SSNET,		//	network arrows
	COLOR_SSBGMEM,		//	memory jauge background
	COLOR_SSBGCPU		//	cpu jauge background
};
/****************************************************************/

/*
 *	This array maps the colors per functionnality to
 *	the numbers 0..15. This determines the roles
 *	of the colors in the color image.
 */

static const SSColor map_hi_func[16] = {
	SSWHITE, SSBGMENU, SSGRAY1, SSINSK,
	SSCAPSTXT, SSNET, SSGRAY2, SSGRAY3,
	SSGRAY4, SSRED, SSGREEN, SSGRAY5,
	SSLGRAY, SSMGRAY, SSDGRAY, SSBLACK
};

static const SSColor map_func[16] = {
	SSWHITE, SSBGMENU, (SSColor)(0), SSINSK,
	SSCAPSTXT, SSNET, (SSColor)(0), (SSColor)(0),
	(SSColor)(0), SSRED, SSGREEN, SSBLUE,
	SSLGRAY, SSMGRAY, SSDGRAY, SSBLACK
};

/****************************************************************/

/*
 *	This is a mapping table for the colors not found in the
 *	CLUT. The first element corresponds to the
 *	SSBGMENU color, and the following elements are the
 *	following relative colors.
 */

static const map_default[SS_COL_NUM - SSBGMENU] = {
	SSLGRAY, SSLGRAY, SSWHITE, SSBLACK,
	SSBLACK, SSWHITE,SSWHITE,SSWHITE,
	SSBLACK, SSBLACK, SSBLACK, SSBLACK,
	SSBLACK, SSBLACK, SSLGRAY, SSLGRAY,
	SSLGRAY, SSLGRAY, SSLGRAY, SSLGRAY, SSLGRAY
};

/****************************************************************/

/*
 *	This table describes the position and size of the icons,
 *	according to their number.
 */

static const Box icons[] = {
	{ { 77, 96 }, { 14, 16 } },		//	0  : rIconCapsOffS
	{ { 77, 80 }, { 14, 16 } },		//	1  : rIconCapsOnS
	
	{ { 60, 320 }, { 14, 16 } },	//	2  : rDISiconeF0
	{ { 60, 0 }, { 14, 16 } },		//	3  : rDISiconeFV5
	{ { 60, 16 }, { 14, 16 } },
	{ { 60, 32 }, { 14, 16 } },
	{ { 60, 48 }, { 14, 16 } },
	{ { 60, 64 }, { 14, 16 } },
	{ { 60, 80 }, { 14, 16 } },
	{ { 60, 96 }, { 14, 16 } },
	{ { 60, 112 }, { 14, 16 } },
	{ { 60, 128 }, { 14, 16 } },
	{ { 60, 144 }, { 14, 16 } },
	{ { 60, 160 }, { 14, 16 } },
	{ { 60, 176 }, { 14, 16 } },
	{ { 60, 192 }, { 14, 16 } },
	{ { 60, 208 }, { 14, 16 } },
	{ { 60, 224 }, { 14, 16 } },
	{ { 60, 240 }, { 14, 16 } },
	{ { 60, 256 }, { 14, 16 } },
	{ { 60, 272 }, { 14, 16 } },
	{ { 60, 288 }, { 14, 16 } },
	{ { 60, 304 }, { 14, 16 } },	//	22 : rDISiconFEX5
	
	{ { 102, 240 }, { 14, 20 } },	//	23 : rIconPixmapOff ??
	
	{ { 0, 0 }, { 10, 10 } },
	
	{ { 102, 0 }, { 14, 20 } },		//	25 : rIconPixV1
	{ { 116, 0 }, { 14, 20 } },		//	26 : rIconPixH1
	{ { 102, 20 }, { 14, 20 } },
	{ { 116, 20 }, { 14, 20 } },
	{ { 102, 40 }, { 14, 20 } },
	{ { 116, 40 }, { 14, 20 } },
	{ { 102, 60 }, { 14, 20 } },
	{ { 116, 60 }, { 14, 20 } },
	{ { 102, 80 }, { 14, 20 } },
	{ { 116, 80 }, { 14, 20 } },
	{ { 102, 100 }, { 14, 20 } },
	{ { 116, 100 }, { 14, 20 } },
	{ { 102, 120 }, { 14, 20 } },
	{ { 116, 120 }, { 14, 20 } },
	{ { 102, 140 }, { 14, 20 } },
	{ { 116, 140 }, { 14, 20 } },
	{ { 102, 160 }, { 14, 20 } },
	{ { 116, 160 }, { 14, 20 } },
	{ { 102, 180 }, { 14, 20 } },
	{ { 116, 180 }, { 14, 20 } },
	{ { 102, 200 }, { 14, 20 } },
	{ { 116, 200 }, { 14, 20 } },
	{ { 102, 220 }, { 14, 20 } },	//	47 : rIconPixV12
	{ { 116, 220 }, { 14, 20 } },	//	48 : rIconPixH12
	
	{ { 40, 90 }, { 14, 36 } },		//	49 : rSmakyLogoMenuS
	{ { 40, 0 }, { 18, 45 } },		//	50 : rSmakyLogoMenuM
	
	{ { 77, 0 }, { 16, 40 } },		//	51 : rIconCapsOn
	{ { 77, 40 }, { 16, 40 } },		//	52 : rIconCapsOff
	
	{ { 102, 240 }, { 14, 20 } },	//	53 : rIconPixOff
	{ { 0, 288 }, { 18, 23 } },		//	54 : rIconPixOffM
	{ { 0, 0 }, { 18, 23 } },		//	55 : rIconPixV1M
	{ { 20, 0 }, { 18, 23 } },		//	56 : rIconPixH1M
	{ { 0, 24 }, { 18, 23 } },
	{ { 20, 24 }, { 18, 23 } },
	{ { 0, 48 }, { 18, 23 } },
	{ { 20, 48 }, { 18, 23 } },
	{ { 0, 72 }, { 18, 23 } },
	{ { 20, 72 }, { 18, 23 } },
	{ { 0, 96 }, { 18, 23 } },
	{ { 20, 96 }, { 18, 23 } },
	{ { 0, 120 }, { 18, 23 } },
	{ { 20, 120 }, { 18, 23 } },
	{ { 0, 144 }, { 18, 23 } },
	{ { 20, 144 }, { 18, 23 } },
	{ { 0, 168 }, { 18, 23 } },
	{ { 20, 168 }, { 18, 23 } },
	{ { 0, 192 }, { 18, 23 } },
	{ { 20, 192 }, { 18, 23 } },
	{ { 0, 216 }, { 18, 23 } },
	{ { 20, 216 }, { 18, 23 } },
	{ { 0, 240 }, { 18, 23 } },
	{ { 20, 240 }, { 18, 23 } },
	{ { 0, 264 }, { 18, 23 } },		//	77 : rIconPixV12M
	{ { 20, 264 }, { 18, 23 } },	//	78 : rIconPixH12M
	
	{ { 94, 0 }, { 8, 10 } },		//	79 : rIconNetIn1
	{ { 94, 16 }, { 8, 10 } },		//	80 : rIconNetIn2
	{ { 94, 32 }, { 8, 10 } },		//	81 : rIconNetIn3
	{ { 94, 48 }, { 8, 10 } },		//	82 : rIconNetOut1
	{ { 94, 64 }, { 8, 10 } },		//	83 : rIconNetOut2
	{ { 94, 80 }, { 8, 10 } },		//	84 : rIconNetOut3
	
	{ { 130, 36 }, { 10, 24 } },	//	85 : rIconMsgMailS
	{ { 130, 60 }, { 10, 24 } },
	{ { 130, 84 }, { 10, 24 } },
	{ { 130, 108 }, { 10, 24 } },
	{ { 130, 132 }, { 10, 24 } },
	{ { 130, 156 }, { 10, 24 } },
	{ { 130, 180 }, { 10, 24 } },
	{ { 130, 204 }, { 10, 24 } },
	{ { 130, 228 }, { 10, 24 } },
	{ { 130, 252 }, { 10, 24 } },
	{ { 130, 276 }, { 10, 24 } },
	{ { 140, 36 }, { 10, 24 } },
	{ { 140, 60 }, { 10, 24 } },
	{ { 140, 84 }, { 10, 24 } },
	{ { 140, 108 }, { 10, 24 } },	//	99 : rIconMsgCut
	
	{ { 150, 0 }, { 16, 32 } },		//	100 : rIconMsgMailM
	{ { 150, 32 }, { 16, 32 } },
	{ { 150, 64 }, { 16, 32 } },
	{ { 150, 96 }, { 16, 32 } },
	{ { 150, 128 }, { 16, 32 } },
	{ { 150, 160 }, { 16, 32 } },
	{ { 150, 192 }, { 16, 32 } },
	{ { 150, 224 }, { 16, 32 } },
	{ { 150, 256 }, { 16, 32 } },
	{ { 166, 0 }, { 16, 32 } },
	{ { 166, 32 }, { 16, 32 } },
	{ { 166, 64 }, { 16, 32 } },
	{ { 166, 96 }, { 16, 32 } },
	{ { 166, 128 }, { 16, 32 } },
	{ { 166, 160 }, { 16, 32 } },	//	114 : rIconMsgCutM
	
	{ { 182, 290 }, { 34, 44 } },	//	115 : rIconSK640a
	{ { 182, 0 }, { 34, 116 } },	//	116 : rIconSK640b
	{ { 272, 204 }, { 44, 52 } },	//	117 : rIconSK800a
	{ { 228, 0 }, { 44, 148 } },	//	118 : rIconSK800b 
	{ { 272, 150 }, { 44, 54 } },	//	119 : rIconSK832a
	{ { 228, 148 }, { 44, 150 } },	//	120 : rIconSK832b
	{ { 272, 256 }, { 46, 54 } },	//	121 : rIconSK864a
	{ { 272, 0 }, { 46, 150 } },	//	122 : rIconSK864b
	{ { 76, 260 }, { 46, 62 } },	//	123 : rIconSK1024a
	{ { 182, 116 }, { 46, 174 } },	//	124 : rIconSK1024b
	
	{ { 40, 126 }, { 14, 36 } },	//	125 : rSmakyLogoMenuDS
	{ { 40, 45 }, { 18, 45 } },		//	126 : rSmakyLogoMenuDM
};

#define LAST_ICON_NUM (126)

/****************************************************************/

/*
 *	Static variables
 */

Gra			*LookMan::ggra = NULL;			// any Gra for Init

ImageDescr	LookMan::mimage;				// monochrome image
ImageDescr	LookMan::cimage;				// color image
ImageDescr	LookMan::himage;				// high color image

void*		LookMan::clut = NULL;			// color image clut
void*		LookMan::hclut = NULL;			// high color image clut
Bool		LookMan::force_mono = FALSE;	// force monochrome drawing

//	Static variables for custom icons management

ImageDescr	LookMan::cmimage;		// bitmap for custom monochrome icons
ImageDescr	LookMan::ccimage;		// color pixmap for custom 4 bit-pixel icons

Card8		LookMan::sizesnum;		// number of icons sizes
Card8		*LookMan::sizes;		// icons sizes
Card8		LookMan::maxicons;		// maximal number of square icons

Box			LookMan::coord[MAX_ICONS_SIZES][MAX_ICONS];	// pos. and dim. of icons in pixmap
Bool		LookMan::occ[MAX_ICONS_SIZES][MAX_ICONS];	// occupated zones in the pixmap
CIconInfo	LookMan::imap[MAX_ICONS_SIZES * MAX_ICONS];	// icon number / size vs pixmap zone

/****************************************************************/

LookMan::LookMan (Gra *gra)
{
	this->gra = gra;
	this->cdata = NULL;
	this->ccdata = NULL;
	this->depth = 1;
	this->use_color = FALSE;
	this->clut16 = NULL;
	
	if (LookMan::ggra == NULL)
		LookMan::ggra = gra;
	
	//	Initialize conversion table
	
	for (int i = 0; i < 16; i++)
		this->ctable[i] = i;
	
	//	Allocate 16 colors clut
	
	this->clut16 = (TCClut *)(new char[sizeof (TCClut) + 256*sizeof (TCEntry)]);
	
	if (this->clut16 == NULL) {
		AfText ("SMA_SMAKY : no more memory for clut in LookMan::LookMan");
		AfCR ();
	}
	
	for (i = 0; i < SS_COL_NUM; i++) this->colors[i] = i;
}


/*
 *	Convert a 4-bpp image faster than any TCOLOR function ! Replace data
 *	byte by byte...
 */

void
LookMan::ConvC (const ImageDescr *src, void *dest, const Card8 *tbl)
{
	if (this->depth == 8) {
		Card16 table[256];
		Card16 *t = table;
		
		for (int i = 0; i < 16; i++)
			for (int j = 0; j < 16; j++)
				*t++ = (tbl[i] << 8) + tbl[j];
		
		int dy = src->box.d.dy;
		int dx = src->box.d.dx / 2;
		
		Card16* pd = (Card16*)(dest);
		Card8*  ps = (Card8*)(src->data);
		
		while (dy--) {
			
			int    n = dx;
			Card8* p = ps;
			
			while (n--) {
				*pd++ = table[*p++];
			}
			
			ps += src->bwidth;
		}
		
		return;
	}
	
	Card8 table[256];
	Card8 *t = table;
	
	for (int i = 0; i < 16; i++)
		for (int j = 0; j < 16; j++)
			*t++ = (tbl[i] << 4) + tbl[j];
	
	t = table;
	
	asm volatile (	"	movel	%0,a2			\n\t"
					"	movel	%1,a3			\n\t"
					"	movew	%2,d2			\n\t"
					"	movew	%3,d3			\n\t"
					"	movel	%4,a4			\n\t"
					"	clrl	d4				\n\t"
					"	subqw	#1,d2			\n\t"
					"1:	movel	a2,a1			\n\t"
					"	movel	a3,a5			\n\t"
					"	movew	d3,d5			\n\t"
					"	subqw	#1,d5			\n\t"
					"2:	moveb	a1@+,d4			\n\t"
					"	moveb	a4@(0,d4:l),a5@+\n\t"
					"	dbra	d5,2b			\n\t"
					"	addw	d3,a2			\n\t"
					"	addw	d3,a3			\n\t"
					"	dbra	d2,1b			\n\t"
		:
		: "g" (src->data), "g" (dest), "g" (src->box.d.dy),
		  "g" (src->bwidth), "g" (t)
		: "d2", "d3", "d4", "d5", "a1", "a2", "a3", "a4", "a5" );
}

void
LookMan::LoadMonoImage ()
{
	//	Load monochrome image
	
	/*
	 *	This loads an image from the disk.
	 *	Now we just get the data from the resource file.
	 *
	Card16 channel;
	SmArgs args;
	Card32 len;
	Card16 err;
	
	err = Fos::ArgsOpen ("(,#):SMA_SMAKY.IMAGE", FOS_OPRD | FOS_OPARGS, &args,
						 NULL, NULL, NULL, 0, &channel);
	
	if (err) {
		AfText ("SMA_SMAKY Fatal : SMA_SMAKY.IMAGE : ");
		AfX4 (err);
		AfCR ();
		
		Fos::Close (channel);
		WAIT_FOREVER;
	}
	
	void *mdata = new char[args.lgf];
	
	if (mdata == NULL) {
		AfText ("SMA_SMAKY Fatal : not enough memory for mdata, size ");
		AfX8 (args.lgf);
		AfCR ();
		
		Fos::Close (channel);
		WAIT_FOREVER;
	}
	
	len = args.lgf;
	Fos::RdByte (channel, &len, mdata);
	Fos::Close (channel);
	 *
	 */
	
	Res mres (rtyp_image, rIconsImageM);
	void *mdata = mres.GetData ();
	
	//	Decode monochrome image
	
	Image header = *((Image *)(mdata));
	
	LookMan::mimage.box.d.dx	= header.dx;
	LookMan::mimage.box.d.dy	= header.dy;
	LookMan::mimage.box.o.x	= 0;
	LookMan::mimage.box.o.y	= 0;
	
	Card16 rbytes	= (header.dx + 7) / 8;
	
	LookMan::mimage.bwidth = rbytes;
	
	LookMan::mimage.data = new char[rbytes * LookMan::mimage.box.d.dy];
	
	void *encoded = mdata + LG_IMAGE;
	
	if (header.coding != 0)
		LookMan::ggra->DecodeImage (LookMan::mimage.data, encoded, & header);
	else
		memcpy (LookMan::mimage.data, encoded, header.length);
	
	/*
	 *	If we loaded the image from the disk...
	 *
	delete mdata;
	 */
}

void
LookMan::LoadColorImage ()
{
	//	Load color image
	
	/*
	 *	This loads an image from the disk.
	 *	Now we just get the data from the resource file.
	 *
	Card16 channel;
	SmArgs args;
	Card32 len;
	Card16 err;
	
	err = Fos::ArgsOpen ("(,#):SMA_SMAKY.COLOR", FOS_OPRD | FOS_OPARGS, &args,
						 NULL, NULL, NULL, 0, &channel);
	
	if (err) {
		AfText ("SMA_SMAKY Fatal : SMA_SMAKY.COLOR : ");
		AfX4 (err);
		AfCR ();
		
		Fos::Close (channel);
		WAIT_FOREVER;
	}
	
	void *gcdata = new char[args.lgf];
	
	if (gcdata == NULL) {
		AfText ("SMA_SMAKY Fatal : not enough memory for gcdata, size ");
		AfX8 (args.lgf);
		AfCR ();
		
		Fos::Close (channel);
		WAIT_FOREVER;
	}
	
	len = args.lgf;
	Fos::RdByte (channel, &len, gcdata);
	Fos::Close (channel);
	 *
	 */
	
	Res cres (rtyp_image, rIconsImageC);
	void *gcdata = cres.GetData ();
	
	//	Decode color image
	//	Does it work ... always ?
	
	Image header = *((Image *)(gcdata));
	
	LookMan::cimage.box.d.dx	= header.dx;
	LookMan::cimage.box.d.dy	= header.dy;
	LookMan::cimage.box.o.x	= 0;
	LookMan::cimage.box.o.y	= 0;
	
	if (header.clutlen) {
		LookMan::clut = new char[header.clutlen];
		memcpy (LookMan::clut, gcdata + LG_IMAGE, header.clutlen);
	}
	
	Card16 rbytes	= ((header.dx + 7) >> 3) * 4;
	
	LookMan::cimage.bwidth = rbytes;
	
	LookMan::cimage.datalen = rbytes * LookMan::cimage.box.d.dy;
	LookMan::cimage.data = new char[LookMan::cimage.datalen + 64];
	LookMan::cimage.data += 64;
	
	void *encoded = gcdata + LG_IMAGE + header.clutlen;
	
	if (header.coding != 0) {
		Card16 old_dx = header.dx;
		Card8 old_depth = header.depth;
		header.dx *= 4;
		header.depth = 1;
		LookMan::ggra->DecodeImage (LookMan::cimage.data, encoded, & header);
		header.dx = old_dx;
		header.depth = old_depth;
	} else
		memcpy (LookMan::cimage.data, encoded, header.length);
	
	/*
	 *	If we loaded the image from the disk...
	 *
	delete gcdata;
	 */
}

void
LookMan::LoadHiColorImage ()
{
	//	Load high color image
	
	/*
	 *	This loads an image from the disk.
	 *	Now we just get the data from the resource file.
	 *
	Card16 channel;
	SmArgs args;
	Card32 len;
	Card16 err;
	
	err = Fos::ArgsOpen ("(,#):SMA_SMAKY.COLOR", FOS_OPRD | FOS_OPARGS, &args,
						 NULL, NULL, NULL, 0, &channel);
	
	if (err) {
		AfText ("SMA_SMAKY Fatal : SMA_SMAKY.COLOR : ");
		AfX4 (err);
		AfCR ();
		
		Fos::Close (channel);
		WAIT_FOREVER;
	}
	
	void *gcdata = new char[args.lgf];
	
	if (gcdata == NULL) {
		AfText ("SMA_SMAKY Fatal : not enough memory for gcdata, size ");
		AfX8 (args.lgf);
		AfCR ();
		
		Fos::Close (channel);
		WAIT_FOREVER;
	}
	
	len = args.lgf;
	Fos::RdByte (channel, &len, gcdata);
	Fos::Close (channel);
	 *
	 */
	
	Res cres (rtyp_image, rIconsImageH);
	void *gcdata = cres.GetData ();
	
	//	Decode color image
	//	Does it work ... always ?
	
	Image header = *((Image *)(gcdata));
	
	LookMan::himage.box.d.dx	= header.dx;
	LookMan::himage.box.d.dy	= header.dy;
	LookMan::himage.box.o.x	= 0;
	LookMan::himage.box.o.y	= 0;
	
	if (header.clutlen) {
		LookMan::hclut = new char[header.clutlen];
		memcpy (LookMan::hclut, gcdata + LG_IMAGE, header.clutlen);
	}
	
	Card16 rbytes	= ((header.dx + 7) >> 3) * 4;
	
	LookMan::himage.bwidth = rbytes;
	
	LookMan::himage.datalen = rbytes * LookMan::himage.box.d.dy;
	LookMan::himage.data = new char[LookMan::himage.datalen + 64];
	LookMan::himage.data += 64;
	
	void *encoded = gcdata + LG_IMAGE + header.clutlen;
	
	if (header.coding != 0) {
		Card16 old_dx = header.dx;
		Card8 old_depth = header.depth;
		header.dx *= 4;
		header.depth = 1;
		LookMan::ggra->DecodeImage (LookMan::himage.data, encoded, & header);
		header.dx = old_dx;
		header.depth = old_depth;
	} else
		memcpy (LookMan::himage.data, encoded, header.length);
	
	/*
	 *	If we loaded the image from the disk...
	 *
	delete gcdata;
	 */
}

void
LookMan::OpenIColor ()
{
	Card16 err;
	
	static char icolor_name[] = "ICOLOR";
	char *name = icolor_name;
	
	asm (	"movel	%1,a3		\n\t"
			"movew	#0xaad0,d3	\n\t"
			".short	0xa102		\n\t"
			"movew	d7,%0"
		: "=g" (err)
		: "g" (name)
		: "d1", "d3", "d7", "a3" );
	
	if (err) {
		AfText ("SMA_SMAKY : OpenIColor error : ");
		AfX4 (err);
		AfCR ();
	}
}

void
LookMan::SetIconsSizes (Card8 number, const Card8 *list)
{
	LookMan::sizesnum = number;
	LookMan::sizes = new Card8[number];
	
	if (LookMan::sizes)
		for (int i = 0; i < number; i++)
			LookMan::sizes[i] = list[i];
}

void
LookMan::SetMaxIconsNum (Card8 num)
{
	LookMan::maxicons = num;
}

void
LookMan::Init ()
{
	if (LookMan::ggra == NULL) {
		AfText ("SMA_SMAKY Fatal : Init and no ggra\r");
		WAIT_FOREVER;
	}
	
	//	Load monochrome and color images
	
	LookMan::LoadMonoImage ();
	LookMan::LoadColorImage ();
	LookMan::LoadHiColorImage ();
	
	//	Open ICOLOR module.
	//	We are probably the first program of the system to do it.
	//	WARNING : if we are on an exclusively monochrome machine, we
	//	shouldn't do it (I believe, or what ?).
	
	LookMan::OpenIColor ();
	
	//	Allocate resources for custom icons
	
	LookMan::InitCustomIcons ();
}

void
LookMan::InitCustomIcons ()
{
	if (LookMan::maxicons && LookMan::sizesnum) {
		
		//	Allocate pixmap
		
		ImageDescr cim;
		ImageDescr mim;
		
		Card8 maxsize = 0;
		Card16 height = 0;
		
		for (int i = 0; i < LookMan::sizesnum; i++) {
			if (LookMan::sizes[i] > maxsize)
				maxsize = LookMan::sizes[i];
			height += LookMan::sizes[i];
		}
		
		cim.bwidth = (maxsize * LookMan::maxicons * 4 + 7) / 8;
		cim.datalen = cim.bwidth * height;
		
		mim.bwidth = (maxsize * LookMan::maxicons + 7) / 8;
		mim.datalen = mim.bwidth * height;
		
		Box b = { { 0, 0 }, { height, cim.bwidth * 8 } };
		cim.box = b;
		mim.box = b;
		
		cim.data = new char[cim.bwidth * height];
		mim.data = new char[mim.bwidth * height];
		
		LookMan::ccimage = cim;
		LookMan::cmimage = mim;
		
		//	Allocate and initialize arrays
		/*
		LookMan::coord = new (Box[maxsize][LookMan::maxicons]);
		LookMan::occ = new (Bool[maxsize][LookMan::maxicons]);
		LookMan::imap = new (Card8[maxsize][LookMan::maxicons][2]);
		*/
		
		Int16 h = 0;
		
		for (int j = 0; j < maxsize; j++) {
			for (int k = 0; k < LookMan::maxicons; k++) {
				occ[j][k] = FALSE;
				
				Box *b = &(coord[j][k]);
				b->o.x = LookMan::sizes[j] * k;
				b->o.y = h;
				b->d.dx = 0;
				b->d.dy = 0;
			}
			h += LookMan::sizes[j];
		}
		
		for (int l = 0; l < (MAX_ICONS_SIZES * MAX_ICONS); l++)
			imap[l].is_free = TRUE;
	}
}


LookMan::~LookMan ()
{
	//	We never quit...
}


Bool
LookMan::FindColor (const GColor& color, Card32& index)
{
	Card32 r = color.r >> 8;
	Card32 g = color.g >> 8;
	Card32 b = color.b >> 8;
	
	Card32 limit = 0xffffffff;
	
	int num = (this->depth == 8) ? 256 : 16;
	
	for (int i = 0; i < num; i++) {
		Int32 dr = r - (this->clut16->entry[i].color.r >> 8);
		Int32 dg = g - (this->clut16->entry[i].color.g >> 8);
		Int32 db = b - (this->clut16->entry[i].color.b >> 8);
		
		dr *= dr;
		dg *= dg;
		db *= db;
		
		Card32 value = (Card32)(dr + dg + db);
		
		if (value < limit) {
			limit = value;
			index = i;
		}
		
		if (value == 0) return TRUE;
	}
	
	if (limit < 2000)	// about 10 % global tolerance
		return TRUE;
	else
		return FALSE;
}


void
LookMan::Update ()
{
	//	Look if we have colors (for the moment, only 16 colors supported)
	
	this->depth = this->gra->GetDepth ();
	
	if ( (this->depth < 4)
	  || (this->depth > 8)
	  || (LookMan::force_mono)
	  || (this->clut16 == NULL) ) {
		this->use_color = FALSE;
		return;
	}
	
	//	From here we assume that we have exacly 16 colors...
	
	this->use_color = TRUE;
	
	//	Read 16 (or even 256) colors clut...
	
	this->tcolor.Update (this->gra->GetTChannel ());
	this->tcolor.SaveClut (TC_TABLE, this->clut16);
	
	//	Find relative colors from the system CRET.
	//	A test is made in order to see if the colors are
	//	defined in the system CRET. If not the case, a
	//	default color is selected.
	
	GColor ncolors[SS_COL_NUM];
	GColor ecolor;
	
	Card32 index;
	
	for (int i = 0; i < SS_COL_AUTO_NUM; i++) {
		if (this->tcolor.SrcCret (map_relcol[i], ncolors[i], ecolor, index)) {
			ncolors[i] = default_colors[i];
		}
	}
	
	for (i = 0; i < 5; i++) {
		ncolors[SSGRAY1+i].r = (Card32)(ncolors[SSLGRAY].r - ncolors[SSMGRAY].r) * (7-i) / 7
							   + ncolors[SSMGRAY].r;
		ncolors[SSGRAY1+i].g = (Card32)(ncolors[SSLGRAY].g - ncolors[SSMGRAY].g) * (7-i) / 7
							   + ncolors[SSMGRAY].g;
		ncolors[SSGRAY1+i].b = (Card32)(ncolors[SSLGRAY].b - ncolors[SSMGRAY].b) * (7-i) / 7
							   + ncolors[SSMGRAY].b;
	}
	
	//	Find these colors in the CLUT
	
	Card8 found[SS_COL_NUM];
	
	for (i = 0; i < SS_COL_NUM; i++) {
		found[i] = this->FindColor (ncolors[i], this->colors[i]) ? 1 : 0;
	}
	
	//	Check if we have at least two gray levels.
	//	If not the case, don't use color.
	
	if ( (found[SSLGRAY] + found[SSMGRAY] + found[SSDGRAY]) < 2) {
		this->use_color = FALSE;
		return;
	}
	
	//	Map gray indexes.
	//	Assume that black and white exist.
	
	if (found[SSLGRAY] == 0) this->colors[SSLGRAY] = this->colors[SSMGRAY];
	else if (found[SSMGRAY] == 0) this->colors[SSMGRAY] = this->colors[SSDGRAY];
	else if (found[SSDGRAY] == 0) this->colors[SSDGRAY] = this->colors[SSMGRAY];
	
	//	Map other indexes
	
	for (i = 0; i < (SS_COL_NUM - SSBGMENU); i++) {
		if (found[i + SSBGMENU] == 0) {
			this->colors[i + SSBGMENU] = this->colors[map_default[i]];
		}
	}
	
	//	Calculate conversion table using
	//	fixed-color functions source image.
	
	const SSColor* map = (this->depth < 8) ? map_func : map_hi_func;
	
	for (i = 0; i < 16; i++) {
		this->ctable[i] = this->colors[map[i]];
	}

	//	If color pixmap, convert color image
	
	if (this->cdata == NULL) {
		this->cdata = new char[LookMan::cimage.datalen*8];	//	worst case 32-bit !
		if (this->cdata == NULL) {
			AfText ("SMA_SMAKY : no more memory for allocating cdata in LookMan\r");
			this->use_color = FALSE;
			return;
		}
	}
	
	ImageDescr* color_image = (this->depth < 8) ? (&LookMan::cimage) : (&LookMan::himage);
	this->ConvC (color_image, this->cdata, this->ctable);
}


void
LookMan::DrawIcon (Card16 num, Pos pos)
{
	if (num > LAST_ICON_NUM) return;
	
	this->gra->SetMode (LOADDOT);
	
	if (this->UseColor ()) {
		int n = (this->depth == 4) ? LookMan::cimage.bwidth : (LookMan::cimage.bwidth * 2);
		this->gra->RasterColor (icons[num], pos, this->cdata, n);
	} else {
		this->gra->RasterMono (icons[num], pos, LookMan::mimage.data, LookMan::mimage.bwidth);
	}
}


void
LookMan::GetIconSize (Card16 num, Dim& dim) const
{
	if (num > LAST_ICON_NUM) {
		dim.dx = 0;
		dim.dy = 0;
	} else {
		dim = icons[num].d;
	}
}


void
LookMan::DrawRectDown (Box rect)
{
	Box b = rect;
	b.d.dy = 0;
	
	this->gra->SetMode (SETDOT);
	this->gra->SetFgPixel (this->colors[SSDGRAY]);
	this->gra->Line (b);
	
	b.o.y++;
	b.d.dy = rect.d.dy - 1;
	b.d.dx = 0;
	
	this->gra->Line (b);
	
	b.o.x += rect.d.dx - 1;
	
	this->gra->SetFgPixel (this->colors[SSWHITE]);
	this->gra->Line (b);
	
	b.o.x = rect.o.x + 1;
	b.o.y = rect.o.y + rect.d.dy - 1;
	b.d.dx = rect.d.dx - 2;
	b.d.dy = 0;
	
	this->gra->Line (b);
}


void
LookMan::DrawRectUp (Box rect)
{
	Box b = rect;
	b.d.dy = 0;
	
	this->gra->SetMode (SETDOT);
	this->gra->SetFgPixel (this->colors[SSWHITE]);
	this->gra->Line (b);
	
	b.o.y++;
	b.d.dy = rect.d.dy - 1;
	b.d.dx = 0;
	
	this->gra->Line (b);
	
	b.o.x += rect.d.dx - 1;
	
	this->gra->SetFgPixel (this->colors[SSDGRAY]);
	this->gra->Line (b);
	
	b.o.x = rect.o.x + 1;
	b.o.y = rect.o.y + rect.d.dy - 1;
	b.d.dx = rect.d.dx - 2;
	b.d.dy = 0;
	
	this->gra->Line (b);
}


void
LookMan::SetColorBack (SSColor color)
{
	this->gra->SetBgPixel (this->colors[color]);
}


void
LookMan::SetColorFore (SSColor color)
{
	 this->gra->SetFgPixel (this->colors[color]);
}


void
LookMan::SetColorsMenuText ()
{
	this->gra->SetBgPixel (this->colors[SSBGMENU]);
	this->gra->SetFgPixel (this->colors[SSBLACK]);
}


void
LookMan::SetColorSKText ()
{
	this->gra->SetBgPixel (this->colors[SSBGSK]);
	this->gra->SetFgPixel (this->colors[SSBLACK]);
}


void
LookMan::RestoreColors ()
{
	this->gra->SetFgPixel (this->colors[SSBLACK]);
	this->gra->SetBgPixel (this->colors[SSWHITE]);
}

Card8
LookMan::AllocateCIcon (Card8 size_num, Dim dim)
{
	return 0;
}

void
LookMan::SetIcon (Card8 num, Card8 depth, Dim dim, const void *data)
{
	if ( (depth != 1) && (depth != 4) ) return;
	
}

void
LookMan::KillCIcon (Card8 num)
{
	
}

void
LookMan::DrawCIcon (Card8 num, Pos pos)
{
	
}

