/*
 *	menu.cxx
 *
 *	Menu bar class for SMA_SMAKY
 *
 *	There are as many instances of menu bars
 *	as the number of DMs (Display Managers)
 *	installed.
 *
 *	(C)	1994 Daniel MARMIER, Pierre ARNAUD
 *			 for the original version
 *
 *			 Erik BRUCHEZ
 *			 for improvements and cleaning
 */

#include "lib.h"
#include "fos.h"
#include "res.h"
#include "menu.h"
#include "resources.h"
#include "icons.h"
#include "waitkeys.h"
#include "activityspy.h"
#include "smevents.h"
#include "str.h"

#include "mon.h"

extern ResType rtyp_fontrec;

extern Clock		clock;
extern ActivitySpy	activity_spy;
extern EventBox		*global_event_box;

/****************************************************************/

#define MAX_MSG_SCR_WIDTH (322 - 14)

struct MenuData
{
	Card16	logo_icon_num;			// number of logo resource
	Card16	logo_down_icon_num;
	Card16	logo_x_pos;				// logo x position
	
	Card16	disp_icon1_x_pos;		// first display icon x position
	Card16	disp_space;				// space between display icons
	
	Card16	disp_v1_icon_num;		// display icons
	Card16	disp_h1_icon_num;
	Card16	disp_off_icon_num;
	
	Card16	caps_icon_x_pos;		// caps icon x position
	
	Card16	caps_on_icon_num;		// caps icons numbers
	Card16	caps_off_icon_num;
	
	Card16	unit_icon1_x_pos;		// first unit icon x position
	Card16	unit_space;				// space between unit icons
	
	Card16	css_box_width;			// clock subsystem width
	Card16	clock_width;			// clock itself width
	Card16	mem_width;				// mem. stats width
	Card16	cpu_width;				// CPU stats width
	Card16	net_width;				// network zone width
	
	Card16	msg_box_width;			// complete message zone box width
	Card16	msg_scroll_width;		// message zone width (without icon)
	Card16	msg_icon_x_pos;			// msg icon x pos. (relative to msg box)
	
	Card16	msg_icon_space;			// width of message icon
	Card16	msg_icon_num;			// number of first msg icon resource
	
	Pos		net_in_pos;				// relative to css box
	Pos		net_out_pos;			// idem
	
	Card16	font_res_num;			// standard font
	Card16	dir_font_res_num;		// directory font
};


static const MenuData data_640 = {
	rSmakyLogoMenuS, 				// number of logo resource
	rSmakyLogoMenuDS,
	4,								// logo x position
	
	42,								// first display icon position //{ {2, 42}, {14, 24} },
	22,								// space between display icons
	
	rIconPixV1,						// display icons
	rIconPixH1,
	rIconPixOff,

	272,							// caps icon position //{ {2, 272}, {14, 16} }
	
	rIconCapsOnS,					// caps icons numbers
	rIconCapsOffS,
	
	296,							// first unit icon position //{ {2, 296}, {14, 14} }
	16,								// space between unit icons
	
	158,							// clock subsystem width
	100,							// clock itself width
	21,								// mem. stats width
	21,								// CPU stats width
	14,								// network zone width
	
	240 - 14,						// complete message zone box width
	208 - 14,						// message zone width (without icon)
	0,								// msg icon x pos. (relative to msg box)
	
	24,								// width of message icon
	rIconMsgMailS,					// number of first msg icon resource
	
	{ 0, 144 },						// net in icon pos. (relative to css box)
	{ 8, 144 },						// net out icon pos. (relative to css box)
	
	rClockFontS,					// standard font
	rClockFontS						// directory font
};


static const MenuData data_800 = {
	rSmakyLogoMenuM,				// number of logo resource
	rSmakyLogoMenuDM,
	4 ,								// logo x position
	
	52,								// first display icon position //{ {2, 52}, {18, 24} }
	26,								// space between display icons
	
	rIconPixV1M,					// display icons
	rIconPixH1M,
	rIconPixOffM,
	
	310,							// caps icon position //{ {2, 310}, {18, 40} }
	
	rIconCapsOnM,					// caps icons numbers
	rIconCapsOffM,
	
	350,							// first unit icon position //{ {4, 350}, {14, 14} }
	16,								// space between unit icons
	
	234,							// clock subsystem width
	176,							// clock itself width
	21,								// mem. stats width
	21,								// CPU stats width
	14,								// network zone width
	
	362 - 14,						// complete message zone box width
	322 - 14,						// message zone width (without icon)
	0,								// msg icon x pos. (relative to msg box)
	
	32,								// width of message icon
	rIconMsgMailM,					// number of first msg icon resource
	
	{ 2, 220 },						// net in icon pos. (relative to css box)
	{ 10, 220 },					// net out icon pos. (relative to css box)
	
	rClockFontL,					// standard font
	rClockFontS						// directory font
};


/****************************************************************/

Menu::Menu (Gra *gra, LookMan	*lookman)
{
	//	Fields initialization
	
	this->gra = gra;
	this->lookman = lookman;
	
	this->fontres  = NULL;
	this->dirfontres = NULL;
	
	this->bigfontres = new Res (rtyp_fontrec, rClockFontL);
	this->bigfont = (Gencar *)(this->bigfontres->GetData ());
	
	this->tbuffer = NULL;
	this->tbuffer_width = 0;
	
	this->caps_icon = 0;
	this->caps_on = FALSE;
	
	for (int i = 0; i < 10; i++)
		this->disp_icon[i] = 0;
	
	for (i = 0; i < 6; i++)
		this->unit_icon[i] = 0;
	
	for (i = 0; i < 3; i++) {
		this->net_in_icon[i] = 0;
		this->net_out_icon[i] = 0;
	}
		
	this->clock_text[0] = '\0';
	this->clock_mode = CLOCK_NORM;
	
	this->disp_installed = 0;
	this->disp_visible = 0;
	
	this->unit_installed = 0;
	this->unit_entered = 0;
	
	this->net_state_in = 0;
	this->net_state_out = 0;
	
	//	Messages fields initialization
	
	this->message = FALSE;
	this->msg_text[0] = '\0';
	this->msg_len = 0;
	this->msg_icon = 0;
	this->msg_pos = 0;
	this->msg_icon_num = 0;
	
	this->msg_buf = NULL;
	this->msg_buf_width = 0;
	
	//	Dummy, in order to be able do use Setxxx before
	//	having used SizeChanged
	
	this->data = & data_640;
}

Menu::~Menu ()
{
/*	Assuming we never quit ...
 *	This is old, old...
 *
 *	delete gra;
 *
 *	delete caps_res;
 *	delete logo_res;
 *
 *	for (int i = 0; i < 10; i++)
 *		delete disp_res [i];
 *
 *	for (int i = 0; i < 6; i++)
 *		delete unit_res [i];
 */
}

void
Menu::SetBox (Box box)
{
	//	Global menu bar box
	
	this->box = box;
	
	//	Clock subsystem region
	
	this->css_box.d.dx = this->data->css_box_width;
	
	this->css_box.o.x = this->box.o.x + this->box.d.dx - this->css_box.d.dx;
	this->css_box.d.dy = this->box.d.dy - 1 - 2;
	this->css_box.o.y = this->box.o.y + 1;
	
	this->clock_box = this->css_box;
	this->clock_box.d.dx = this->data->clock_width;
	
	this->mem_box = this->clock_box;
	this->mem_box.o.x += this->data->clock_width;
	this->mem_box.d.dx = this->data->mem_width;
	
	this->cpu_box = this->mem_box;
	this->cpu_box.o.x += this->data->mem_width;
	this->cpu_box.d.dx = this->data->cpu_width;
	
	this->msg_box.d.dx = this->data->msg_box_width;
	
	this->msg_box.o.x = this->box.o.x + this->box.d.dx
						- this->msg_box.d.dx - this->data->net_width;
	
	this->msg_box.d.dy = this->clock_box.d.dy;
	this->msg_box.o.y = this->clock_box.o.y;
	
	//	Logo box
	
	Dim dim;
	this->lookman->GetIconSize (this->data->logo_icon_num, dim);
	
	Pos p = { this->box.o.y + (this->box.d.dy - dim.dy) / 2,
			  this->box.o.x + data->logo_x_pos };
	
	this->logo_box.o = p;
	this->logo_box.d = dim;
	
	//	First display box
	
	this->lookman->GetIconSize (this->data->disp_v1_icon_num, dim);
	
	p.x = this->box.o.x + this->data->disp_icon1_x_pos;
	p.y = this->box.o.y + (this->box.d.dy - dim.dy) / 2;
	
	this->disp_box.o = p;
	this->disp_box.d = dim;
	
	//	Caps box
	
	this->lookman->GetIconSize (this->data->caps_on_icon_num, dim);
	
	p.x = this->box.o.x + this->data->caps_icon_x_pos;
	p.y = this->box.o.y + (this->box.d.dy - dim.dy) / 2;
	
	this->caps_box.o = p;
	this->caps_box.d = dim;
	
	//	First unit box
	
	this->lookman->GetIconSize (rDISiconeF0, dim);
	
	p.x = this->box.o.x + this->data->unit_icon1_x_pos;
	p.y = this->box.o.y + (this->box.d.dy - dim.dy) / 2;
	
	this->unit_box.o = p;
	this->unit_box.d = dim;
	
	//	Network activity region
	
	this->net_in_pos = this->css_box.o;
	this->net_in_pos.x += this->data->net_in_pos.x;
	this->net_in_pos.y += this->data->net_in_pos.y;
	
	this->net_out_pos = this->css_box.o;
	this->net_out_pos.x += this->data->net_out_pos.x;
	this->net_out_pos.y += this->data->net_out_pos.y;
}


void
Menu::SizeChanged (Box box)
{
	//	Size-specific information
	
	if (box.d.dx >= 800)
		this->data = & data_800;
	else
		this->data = & data_640;
	
	this->SetBox (box);
	this->ReadClock ();
	this->SetNet ();
	
	//	Font resource changing
	
	if (this->fontres) delete this->fontres;
	this->fontres = new Res (rtyp_fontrec, this->data->font_res_num);
	
	if (this->dirfontres) delete this->dirfontres;
	this->dirfontres = new Res (rtyp_fontrec, this->data->dir_font_res_num);
	
	this->clockfont = (Gencar *)(this->fontres->GetData ());
	this->dirfont = (Gencar *)(this->dirfontres->GetData ());
	
	//	Other resources changing
	
	this->SetDisplays (this->disp_installed, this->disp_visible);
	this->SetUnits (this->unit_installed, this->unit_entered);
	this->SetCaps (this->caps_on);
	
	//	Messages
	
	if (this->message) {
		this->gra->SetGencar (this->clockfont);
		this->msg_len = (Card16)(this->gra->TextWidth (this->msg_text));
		
		if (this->msg_icon)
			this->msg_icon_num = this->data->msg_icon_num + this->msg_icon - 1;
		else
			this->msg_icon_num = 0;
		
		this->SetMsgText (this->msg_text, FALSE);
	}
	
	//	Text graphics buffer
	
	if (this->tbuffer) delete this->tbuffer;
	this->tbuffer_width = (this->clock_box.d.dx + 7) / 8;
	this->tbuffer = new char[this->tbuffer_width * this->clock_box.d.dy];
	
	//	Color management
	
	if (this->gra->GetDepth () > 8) {
		
		//	"True color" modes
		
		this->gra->SetFgPixel (0);
		this->gra->SetBgPixel (0xFFFFFFFF);
		
	} else {
		
		//	Monochrome or clut modes
		
		this->gra->SetFgPixel (0xF);
		this->gra->SetBgPixel (0);
	}
	
	this->UpdateLookMan ();
}


void
Menu::SetCaps (Bool caps_on)
{
	Card16 num = (caps_on) ? this->data->caps_on_icon_num
						   : this->data->caps_off_icon_num;
	
	this->caps_icon = num;
	this->caps_on = caps_on;
}

void
Menu::SetDisplays (Card16 inst, Card16 visible)
{
	for (int i = 0; i < 10; i++) {
		
		Card32 num;
		
		if (inst & (1 << i)) {
			
			if (visible & (1 << i))
				num = this->data->disp_v1_icon_num + 2 * i;
			else
				num = this->data->disp_h1_icon_num + 2 * i;
		} else {
			
			num = this->data->disp_off_icon_num;
		}
		this->disp_icon[i] = num;
	}
	this->disp_installed = inst;
	this->disp_visible = visible;
}

void
Menu::SetUnits (Card16 inst, Card16 enter)
{
	for (int i = 0; i < 6; i++) {
		Card32 num = rDISiconeF0;
		Card16 mm_type;
		const char* physname;
		if ( (inst & (1<<i))
		  && (Fos::MmVAssign (i, & physname, & mm_type) == 0) )
		{
			num = rDISiconeFV5 + mm_type * 2;
			if (enter & (1<<i)) num++;
		}
		unit_icon[i] = num;
	}
	this->unit_installed = inst;
	this->unit_entered = enter;
}


void
Menu::SetNet ()
{
	Card32 num_in = 0, num_out = 0;
	
	for (int i = 0; i < 3; i++) {
		switch (i) {
			
			case 0:
				num_in = rIconNetIn1;
				num_out = rIconNetOut1;
				break;
			
			case 1:
				num_in = rIconNetIn2;
				num_out = rIconNetOut2;
				break;
			
			case 2:
				num_in = rIconNetIn3;
				num_out = rIconNetOut3;
				break;
		}
		
		this->net_in_icon[i] = num_in;
		this->net_out_icon[i] = num_out;
	}
}


void
Menu::SetNetIn (Bool active)
{
	if (active)
		this->net_state_in = (this->net_state_in == 1) ? 2 : 1;
	else
		this->net_state_in = 0;
}


void
Menu::SetNetOut (Bool active)
{
	if (active)
		this->net_state_out = (this->net_state_out == 1) ? 2 : 1;
	else
		this->net_state_out = 0;
}


void
Menu::DrawNet ()
{
	//if (clock.IsMuted () || this->message) return;
	if (clock.IsMuted ()) return;
	
	this->gra->SetMode (LOADDOT);
	
	this->lookman->DrawIcon (this->net_in_icon[net_state_in], this->net_in_pos);
	this->lookman->DrawIcon (this->net_out_icon[net_state_out], this->net_out_pos);
}


void
Menu::UpdateLookMan ()
{
	this->lookman->Update ();
}


void
Menu::DrawAll ()
{
	this->DrawAdornments ();
	this->DrawDisplays ();
	this->DrawClock ();
	this->DrawUnits ();
	this->DrawCaps ();
	this->DrawMemProc (TRUE, TRUE);
	this->DrawNet ();
	this->DrawMsgIcon ();
	this->DrawMsgText ();
}


void
Menu::DrawCaps ()
{
	if (clock.IsMuted ()) return;
	
	gra->SetMode (LOADDOT);
	
	this->lookman->DrawIcon (this->caps_icon, this->caps_box.o);
}

void
Menu::DrawDisplays ()
{
	if (clock.IsMuted ()) return;
	
	gra->SetMode (LOADDOT);
	
	Pos p = this->disp_box.o;
	
	for (int i = 0; i < 10; i++) {
		this->lookman->DrawIcon (disp_icon[i], p);
		p.x += data->disp_space;
	}
}

void
Menu::DrawUnits ()
{
	if (clock.IsMuted ()) return;
	
	gra->SetMode (LOADDOT);
	
	Pos p = this->unit_box.o;
	
	for (int i = 0; i < 6; i++) {
		this->lookman->DrawIcon (unit_icon[i], p);
		p.x += data->unit_space;
	}
}

void
Menu::DrawLogo (Bool down = FALSE)
{
	if (clock.IsMuted ()) return;
	
	this->gra->SetMode (LOADDOT);
	
	if (down) {
		this->lookman->DrawIcon (this->data->logo_down_icon_num, this->logo_box.o);
	} else {
		this->lookman->DrawIcon (this->data->logo_icon_num, this->logo_box.o);
	}
}

void
Menu::DrawAdornments ()
{
	if (clock.IsMuted ()) return;
	
	Box box = this->box;
	
	box.o.y = box.d.dy - 1;
	box.d.dy = 0;
	
	if (this->lookman->UseColor ()) {
		this->lookman->SetColorBack (SSBGMENU);
		this->gra->Clear (this->box);
		
		this->gra->SetMode (SETDOT);
		this->lookman->SetColorFore (SSBLACK);
		this->gra->Line (box);
		
		Box b = this->box;
		b.d.dy--;
		
		this->lookman->DrawRectUp (b);
		
		this->lookman->RestoreColors ();
	} else {
		this->gra->Clear (this->box);
		
		this->gra->SetMode (SETDOT);
		this->gra->Line (box);
	}
	
	this->DrawLogo ();
}

void
Menu::DrawClock (Bool clean_up = FALSE)
{
	if (clock.IsMuted () || this->message) return;
	
	if (this->lookman->UseColor ()) {
		this->lookman->SetColorFore (SSCLTXT);
		this->lookman->SetColorBack (SSBGMENU);
	}
	
	if (clean_up)
		this->gra->Clear (this->clock_box);
	
	this->gra->SetMode (LOADDOT);
	this->gra->SetGencar (this->clockfont);
	
	//	Ugly trick in order to clip the clock box
	
	Box b = this->gra->GetClipBox ();
	Box c = this->clock_box;
	c.o.x = 0;
	c.o.y = 0;
	
	this->gra->SetClipBox (this->clock_box);
	
	const char *p;
	
	if (this->clock_mode == CLOCK_DIR) {
		this->gra->SetGencar (this->dirfont);
		char t[DIR_BUFFER_SIZE];
		this->ComputeTruncatedString (this->clock_text, t, this->clock_box.d.dx);
		p = t;
	} else {
		p = this->clock_text;
	}
	
	if (this->tbuffer)
		this->gra->DrawCText (c, p, this->tbuffer, this->tbuffer_width);
	else
		this->gra->DrawCText (c, p);
	
	this->gra->SetClipBox (b);
	
	if (this->lookman->UseColor ())
		this->lookman->RestoreColors ();
}


Bool
Menu::ReadClock ()
{
	Bool large = (this->clock_box.d.dx < data_800.clock_width) ? FALSE : TRUE;
	
	this->clock_mode = clock.GetMode ();
	return clock.Read (this->clock_text, large);
}


void
Menu::DrawBar (Box box, Card32 max, Card32 value, SSColor fgcol, SSColor bgcol)
{
	if (value > max) value = max;
	
	box.d.dx = 9;
	
	this->gra->SetMode (SETDOT);
	
	Box b = box;
	
	if (this->lookman->UseColor ()) {
		this->lookman->DrawRectDown (box);
		
		Card16 h = box.d.dy - 2;
		Card16 r = max ? (value * h / max) : 0;
		
		b.o.x++;
		b.o.y++;
		b.d.dx -= 2;
		
		if (r) {
			this->lookman->SetColorFore (bgcol);
			b.d.dy = r;
			this->gra->Set (b);
		}
		
		if (h - r) {
			this->lookman->SetColorFore (fgcol);
			b.o.y += r;
			b.d.dy = h - r;
			this->gra->Set (b);
		}
	} else {
		
		this->gra->Rect (box);
		
		Card16 h = box.d.dy;
		Card16 r = max ? (value * h / max) : 0;
		b.d.dy = r + 1;
		
		if ((b.d.dy > 1) && (b.d.dy < h)) this->gra->Rect (b);
		
		if (b.d.dy >= 3) {
			b.o.x++;
			b.d.dx -= 2;
			b.o.y++;
			b.d.dy -= 2;
			
			this->gra->Clear (b);
		}
		
		b = box;
		b.d.dy = h - r;
		b.o.y += r;
		
		if (b.d.dy > 0) this->gra->Rect (b);
		
		this->gra->SetMode (LOADDOT);
		
		if (b.d.dy >= 3) {
			b.o.x++;
			b.d.dx -= 2;
			b.o.y++;
			b.d.dy -= 2;
			
			Card32 trame[] = { 0xaa55aa55, 0xaa55aa55 };
			
			this->gra->Trame (b, trame);
		}
	}
}


void
Menu::DrawHistograms (Box box, Card16 max, Card16 *values, SSColor fgcol, SSColor bgcol)
{
	box.d.dx = 19;
	
	if (this->lookman->UseColor ()) {
		this->gra->SetMode (SETDOT);
		this->lookman->DrawRectDown (box);
		
		this->lookman->SetColorFore (fgcol);
		this->lookman->SetColorBack (bgcol);
	} else {
		this->gra->SetMode (SETDOT);
		this->gra->Rect (box);
	}
	
	Box b = box;
	b.d.dx = 0;
	Card16 h = box.d.dy - 3;
	
	//	Clear the holes if needed (because
	//	the menu background color can be
	//	still here).
	
	if (this->lookman->UseColor ()) {
		Box c = box;
		c.o.x++;
		c.o.y += 2;
		c.d.dx = 0;
		c.d.dy -= 3;
		
		this->gra->SetMode (CLRDOT);
		
		for (int i = 0; i < 9; i++) {
			this->gra->Line (c);
			c.o.x += 2;
		}
		
		Box d = box;
		d.o.x++;
		d.o.y++;
		d.d.dx -= 2;
		d.d.dy = 0;
		this->gra->Line (d);
	}
	
	for (int i = 0 ; i < 8; i++) {
		
		b.o.x += 2;
		b.o.y = box.o.y + 2;
		
		Card16 r = values[i] * h / max;
		
		b.d.dy = r;
		
		this->gra->SetMode (CLRDOT);
		if (b.d.dy) this->gra->Line (b);
		
		b.o.y += r;
		b.d.dy = h - r;
		
		this->gra->SetMode (SETDOT);
		if (b.d.dy) this->gra->Line (b);
	}
}


void
Menu::DrawMemProc (Bool mem_changed, Bool cpu_changed,
				   Bool clean_up_mem = FALSE, Bool clean_up_cpu = FALSE)
{
	if (clock.IsMuted () || this->message) return;
	
	if (this->lookman->UseColor ()) {
		this->lookman->SetColorFore (SSBLACK);
		this->lookman->SetColorBack (SSBGMENU);
	}
	
	if (mem_changed) {
		
		Card32 total, hole, free;
		
		activity_spy.GetMemStats (total, hole, free);
		
		Box box = this->mem_box;
		
		if (clean_up_mem) this->gra->Clear (box);
		
		if (activity_spy.GetMemGraphMode () == GRAPH_MODE_HISTOGRAM) {
			
			//	We must draw histograms
			
			Card16 values[8];
			
			for (int i = 0; i < 8; i++)
				values[i] = activity_spy.GetMemStats (i);
			
			this->DrawHistograms (this->mem_box, 100, values, SSMEM, SSBGMEM);
			
		} else {
			
			//	We must draw a simple bar
			
			Card32 total, hole, free;
			activity_spy.GetMemStats (total, hole, free);
			
			Box b = this->mem_box;
			b.o.x += 5;
			
			this->DrawBar (b, total, free, SSMEM, SSBGMEM);
		}
	}
	
	if (cpu_changed) {
		
		if (clean_up_cpu) this->gra->Clear (this->cpu_box);
		
		if (activity_spy.GetCPUGraphMode () == GRAPH_MODE_HISTOGRAM) {
			
			//	We must draw histograms
			
			Card16 values[8];
			
			for (int i = 0; i < 8; i++)
				values[i] = activity_spy.GetCPUStats (i);
			
			this->DrawHistograms (this->cpu_box, 100, values, SSCPU, SSBGCPU);
			
		} else {
			
			//	We must draw a simple bar
			
			Card16 percent;
			activity_spy.GetCPUStats (percent);
			
			Box b = this->cpu_box;
			b.o.x += 5;
			
			this->DrawBar (b, 100, percent, SSCPU, SSBGCPU);
		}
	}
	
	if (this->lookman->UseColor ())
		this->lookman->RestoreColors ();
}


Bool
Menu::PosInBox (Pos pos, Box box) const
{
	if ( (pos.y < box.o.y)
	  || (pos.y >= box.o.y + box.d.dy)
	  || (pos.x < box.o.x)
	  || (pos.x >= box.o.x + box.d.dx) )
	
		return FALSE;
	else
		return TRUE;
}


Bool
Menu::Click (Pos pos, Card8 clicked)
{
	//	Check if the click is in our box
	
	if ( !this->PosInBox (pos, this->box) ) return FALSE;
	
	//	TEST : click on the logo
		
	if (this->PosInBox (pos, this->logo_box)) {
		if (!(clicked & BUTTON_LEFT)) return TRUE;
		EvClickInMenu ev;
		
		ev.click_position = CLICK_IN_LOGO;
		ev.click_button = CLICK_LEFT;
		global_event_box->PostEvent (ev);
		
		return TRUE;
	}
		
	//	Click in the clock subsystem or message zone
	
	if (this->message && this->PosInBox (pos, this->msg_box)) {
		
		clock.StopMessageDisplaying ();
		
		return TRUE;
		
	} else if (!this->message && this->PosInBox (pos, this->css_box)) {
		
		EvClickInMenu ev;
		
		if (clicked & BUTTON_LEFT)
			ev.click_button = CLICK_LEFT;
		else if (clicked & BUTTON_RIGHT)
			ev.click_button = CLICK_RIGHT;
		else
			return TRUE;
		
		if (this->PosInBox (pos, this->clock_box)) {
			ev.click_position = CLICK_IN_CLOCK;
			global_event_box->PostEvent (ev);
		} else if (this->PosInBox (pos, this->mem_box)) {
			ev.click_position = CLICK_IN_MEM;
			global_event_box->PostEvent (ev);
		} else if (this->PosInBox (pos, this->cpu_box)) {
			ev.click_position = CLICK_IN_CPU;
			global_event_box->PostEvent (ev);
		}
		
		return TRUE;
	}
	
	//	Now we want a click with left button only
	
	if (!(clicked & BUTTON_LEFT)) return TRUE;
	
	//	Click position relative to the menu box origin
	
	pos.x -= this->box.o.x;
	pos.y -= this->box.o.y;
	
	//	Click on a pixmap icon
	
	const Pos pd = this->disp_box.o;
	
	if ( (pos.x > pd.x)
	  && (pos.x < pd.x + 10 * data->disp_space) ) {
		
		Card16 num = (pos.x - pd.x) / data->disp_space;
		
		if (disp_installed & (1<<num)) {
			EvClickInMenu ev;
			
			Dim dim;
			this->lookman->GetIconSize (this->data->disp_v1_icon_num, dim);
			
			ev.click_position = CLICK_IN_DISPLAY;
			ev.click_button = CLICK_LEFT;
			
			ev.display_number = num;
			ev.relative_pos.x = pos.x - pd.x - num * data->disp_space;
			ev.relative_pos.y = pos.y - pd.y;
			ev.start_pos = pd;
			ev.space = data->disp_space;
			ev.icon_dim = dim;
			
			if (ShowPixmap (num)) {
				
				//	Send event, in order to move the mouse
				
				global_event_box->PostEvent (ev);
			}
		} else
			Lib::Beep (BIPLIT);
		
		return TRUE;
	}
	
	const Pos pu = this->unit_box.o;
	
	//	Click on an unit icon
	
	if ( (pos.x > pu.x)
	  && (pos.x < pu.x + 6 * data->unit_space) ) {
		
		Card16 num = (pos.x - pu.x) / data->unit_space;
		
		if (unit_installed & (1<<num))
			UnitOp (UNIT_TOGGLE, num);
		else
			Lib::Beep (BIPLIT);
		
		return TRUE;
	}
	
	return TRUE;
}


Bool
Menu::UpdateLogo (Pos pos)
{
	Bool inside;
	
	if (inside = this->PosInBox (pos, this->logo_box)) {
		this->DrawLogo (TRUE);
		inside = TRUE;
	} else {
		this->DrawLogo (FALSE);
		inside = FALSE;
	}
	
	return inside;
}


Bool
Menu::LogoAction ()
{
	Pos current_pos;
	Card8 current_buttons;
	
	Ntr::SetTim (0);
	Lib::SpyMouse ((Card32*)(&current_pos), &current_buttons);
	Ntr::SetTim (0xffff);
	
	Bool inside = this->UpdateLogo (current_pos);
	
	for (;;) {
		Card8 new_buttons;
		Pos new_pos;
		if (current_buttons == 0) break;
		
		Lib::SpyMouse ((Card32*)(&new_pos), &new_buttons);
		current_buttons = new_buttons;
		
		if ( (new_pos.x != current_pos.x) || (new_pos.y != current_pos.y) ) {
			current_pos = new_pos;
			
			Bool new_inside = this->PosInBox (current_pos, this->logo_box);
			
			if (new_inside != inside) {
				inside = new_inside;
				this->UpdateLogo (current_pos);
			}
		}
	}
	
	this->DrawLogo (FALSE);
	
	return inside ? TRUE : FALSE;
}


void
Menu::CenterMouse (Card16 dis_num, Pos rel_pos, Pos start_pos, Card16 space, Dim icon_dim)
{
	Pos pd = this->disp_box.o;
	
	//	Move mouse only if necessary
	
	if ( (this->data->disp_space != space)
	  || (pd.x != start_pos.x)
	  || (pd.y != start_pos.y) ) {
	  	
	  	Dim dim;
		this->lookman->GetIconSize (this->data->disp_v1_icon_num, dim);
		
	  	if ( (dim.dx) && (dim.dy) && (icon_dim.dx) && (icon_dim.dy)
	  	  && ( (dim.dx != icon_dim.dx) || (dim.dy != icon_dim.dy) )) {
	  		
	  		//	Try to keep proportions
	  		
	  		rel_pos.x = rel_pos.x * dim.dx / icon_dim.dx;
	  		rel_pos.y = rel_pos.y * dim.dy / icon_dim.dy;
	  		
	  		//	Control right position. There wouldn't be any
	  		//	problem if we were sure that the original
	  		//	click was exactly inside an icon, but it's
	  		//	not (yet) the case.
	  		
	  		if (((int)(rel_pos.x)) > (int)((Int16)(this->data->disp_space)))
	  			rel_pos.x = this->data->disp_space;
	  	}
		
		const Int16 x = pd.x + this->data->disp_space * dis_num + rel_pos.x;
		const Int16 y = pd.y + rel_pos.y;
		
		Card16 channel;
		const char command[] = { 'P', y >> 8, y & 0xff, x >> 8, x & 0xff, 0};
		Card16 error;
		
		if (error = Fos::Open ("#MOUSE:", FOS_OPRD, &channel)) {
			/*
			AfText ("Error at open ");
			AfX4 (error);
			AfCR ();
			*/
			return;
		}
		if (Fos::Command (channel, command)) {
			/*
			AfText ("Error at command ");
			AfX4 (error);
			AfCR ();
			*/
			return;
		}
		if (Fos::Close (channel)) {
			/*
			AfText ("Error at close ");
			AfX4 (error);
			AfCR ();
			*/
			return;
		}
	}
}


void
Menu::BeginMessage ()
{
	this->message = TRUE;
}


void
Menu::SetMsgText (const char *text, Bool copy = TRUE)
{
	if (!this->message) return;
	
	if (copy) Str::Copy (this->msg_text, text);
	
	this->gra->SetGencar (this->clockfont);
	this->msg_len = (Card16)(this->gra->TextWidth (this->msg_text));
	
	this->msg_buf_width = (this->msg_len + 7) / 8;
	
	if (this->msg_buf) delete this->msg_buf;
	this->msg_buf = new Card8[this->msg_buf_width * this->gra->GetGencarHeight ()];
	
	Box b = { { 0, 0 },
			  { this->gra->GetGencarHeight (), this->msg_len} };
	
	this->msg_buf_box = b;
	
	if (this->msg_buf)
		this->gra->DrawBText (this->msg_text, this->msg_buf, this->msg_buf_width);
}


void
Menu::SetMsgIcon (Card8 icon_number)
{
	this->msg_icon = icon_number;
	
	if (this->msg_icon)
		this->msg_icon_num = this->data->msg_icon_num + this->msg_icon - 1;
	else
		this->msg_icon_num = 0;
}


Card16
Menu::GetMsgMaxLen (const char *text)
{
	this->gra->SetGencar (this->bigfont);
	return (Card16)(this->gra->TextWidth (text));
}


Card16
Menu::GetMsgMaxScr () const
{
	return MAX_MSG_SCR_WIDTH;
}


void
Menu::DrawMsgIcon ()
{
	if ( (clock.IsMuted ()) || (!this->message) ) return;
	
	if (this->lookman->UseColor ())
		this->lookman->SetColorBack (SSBGMENU);
	
	this->gra->Clear (this->msg_box);
	
	if (this->msg_icon) {
		
		this->gra->SetMode (LOADDOT);
		
		Dim dim;
		this->lookman->GetIconSize (this->msg_icon_num, dim);
		
		Pos p = { this->msg_box.o.y + (this->msg_box.d.dy - dim.dy) / 2,
				  this->msg_box.o.x + this->data->msg_icon_x_pos };
		
		this->lookman->DrawIcon (this->msg_icon_num, p);
	}
	
	if (this->lookman->UseColor ())
		this->lookman->RestoreColors ();
}


void
Menu::SetMsgPos (Int16 position)
{
	this->msg_pos = position;
}


void
Menu::DrawMsgText ()
{
	if ( (clock.IsMuted ()) || (!this->message) ) return;
	
	Box b = this->msg_box;
	b.o.x += this->data->msg_icon_space;
	b.d.dx = this->data->msg_scroll_width;
	
	Box old = this->gra->GetClipBox ();
	
	this->gra->SetClipBox (b);
	this->gra->SetMode (LOADDOT);
	this->gra->SetGencar (this->clockfont);
	
	if (this->lookman->UseColor ()) {
		this->lookman->SetColorFore (SSMSGTXT);
		this->lookman->SetColorBack (SSBGMENU);
	}
	
	if (msg_pos > 0) {
		Box clearbox = { { 0, 0}, { b.d.dy, msg_pos } };
		this->gra->Clear (clearbox);
	}
	
	Pos tp = { (this->msg_box.d.dy - this->gra->GetGencarHeight ()) / 2,
			   this->msg_pos };
	
	if (this->msg_buf)
		this->gra->RasterMono (this->msg_buf_box, tp, this->msg_buf, this->msg_buf_width);
	
	Int32 p;
	
	if ( (p = this->msg_pos + this->msg_len) < (Int32)(b.d.dx) ) {
		Box clearbox = { { 0, p }, { b.d.dy, b.d.dx - p } };
		this->gra->Clear (clearbox);
	}
	
	this->gra->SetClipBox (old);
	
	if (this->lookman->UseColor ())
		this->lookman->RestoreColors ();
}


void
Menu::SetEndMessage ()
{
	if (!this->message) return;
	
	if (this->msg_icon_num)
		this->msg_icon_num = 0;
	
	if (this->msg_buf) {
		delete this->msg_buf;
		this->msg_buf = NULL;
	}
	
	this->message = FALSE;
}


void
Menu::DrawEndMessage ()
{
	if (clock.IsMuted ()) return;
	
	if (this->lookman->UseColor ())
		this->lookman->SetColorBack (SSBGMENU);
	
	this->gra->Clear (this->msg_box);
	
	this->DrawClock ();
	this->DrawMemProc (TRUE, TRUE);
	this->DrawNet ();
}


Bool
Menu::ComputeTruncatedString (const char *text, char *dest, Card32 wmax)
{
	dest[0] = '\0';
	
	//	Text fits
	
	Card32 w = this->gra->TextWidth (text);
	if (w <= wmax) {
		Str::Copy (dest, text);
		return TRUE;
	}
	
	//	Text won't fit at all, even with dots
	//	WARNING : '\2' is a special character !
	
	Card32 dotsw = this->gra->CharWidth ('\2');
	if (dotsw > wmax) return FALSE;
	
	//	Text is too short for us now...
	
	Card32 l = Str::Len (text);
	if (l <= 3) return FALSE;
	
	//	Find middle character (graphically)
	
	const Card32 diff = w + dotsw - wmax;
	Card32 tmpw = 0;
	Card32 ldiff = 0, rdiff = 0;
	const Card32 mpos = w / 2;
	const char *p = text;
	
	for (int i = 0; i < l; i++) {
		const Card32 cw = this->gra->CharWidth (*p++);
		const Card32 neww = tmpw + cw;
		if (neww >= mpos) {
			ldiff = mpos - tmpw;
			rdiff = cw - ldiff;
			break;
		}
		tmpw = neww;
	}
	
	//	Now *(p - 1) is the char. crossed by the center of the zone.
	//	Kill characters around the center of the text until it fits.
	
	const char *rp = p;
	const char *lp = p - 2;
	
	Card32 d = rdiff + ldiff;
	
	while (d < diff) {
		Card32 rw = this->gra->CharWidth (*rp);
		Card32 lw = this->gra->CharWidth (*lp);
		Bool br = FALSE;
		
		while ( (lw + ldiff) < (rw + rdiff) ) {
			
			//	Truncate left
			
			ldiff += lw;
			d += lw;
			lp--;
			
			if (d >= diff) {
				br = TRUE;
				break;
			}
			
			lw = this->gra->CharWidth (*lp);
		}
		
		if (br) break;
		
		while ( (lw + ldiff) >= (rw + rdiff) ) {
			
			//	Truncate right
			
			rdiff += rw;
			d += rw;
			rp++;
			
			if (d >= diff) break;
			
			rw = this->gra->CharWidth (*rp);
		}
		
		/*
		if ( (lw + ldiff) < (rw + rdiff) ) {
//		if ( ldiff < rdiff ) {
			
			//	Truncate left before
			
			ldiff += lw;
			d += lw;
			lp--;
			
			if (d >= diff) break;
			
			//	Truncate right
			
			rdiff += rw;
			d += rw;
			rp++;
		} else {
			
			//	Truncate right before
			
			rdiff += rw;
			d += rw;
			rp++;
			
			if (d >= diff) break;
			
			//	Truncate left
			
			ldiff += lw;
			d += lw;
			lp--;
		}
		*/
	}
	
	//	Build resulting string
	
	while (text <= lp)
		*dest++ = *text++;
	
	while (text < rp) text++;
	
	*dest++ = '\2';
	
	while (*dest++ = *text++)
		/* nothing */ ;
	
	return TRUE;
}

