/*
 *	pixmapman.cxx
 *
 *	Little pixmap manager for SMA_SMAKY.
 *
 *	It provides the user one valid drawing
 *	interface for each visible pixmap.
 *
 *	(C)	1994 Erik BRUCHEZ
 *  (C) 1997 Pierre ARNAUD
 */

#include "pixmapman.h"
#include "softkeys.h"

#include "lib.h"
#include "fos.h"
#include "mon.h"


/*
 *	The constructor reads HWDIS status in order to 
 *	obtain information on the installed display
 *	managers, creates copies of display descrip-
 *	tors and new `Gra' instances ("drawing inter-
 *	faces").
 *
 *	WARNING : assumes that the display managers
 *	are returned in a "logic" order : first DM
 *	in the first record, second in the second,
 *	etc., so there are no "holes" between them !
 */

PixmapMan::PixmapMan ()
{
	int i;
	
	//	Get $DIS_0: window descriptor
	
	Card16 channel;
	
	Fos::Open ("$DIS_0:", FOS_OPRD, &channel);
	Fos::RStatus (channel, 4, &dd0);
	Fos::Close (channel);
	
	//	Read HWDisRec
	
	Fos::RStatus (dd0->chhwd, sizeof (HWDisRec) * 4, this->disrec);
	
	for (i = 0; i < 4; i++) {
		this->is_valid[i] = FALSE;
		this->gra[i] = NULL;
		this->pixmap_id[i] = 0;
	}
	
	//	Create descriptors & drawing interfaces
	
	this->dm_num = 0;
	
	for (i = 0; i < 4; i++) {
		if (this->disrec[i].channel) {
			
			memcpy (&this->desc[i], dd0, sizeof (DisDesc));
			this->desc[i].check = &this->desc[i];
			
			//	Clean-up known fields...
			
			this->desc[i].csf = 15;
			this->desc[i].ccf = 0;
			
			this->desc[i].grabcnt = -1;
			//void	*grabsem; WARNING !
			
			this->gra[i] = new Gra (& this->desc[i]);
			
			this->dm_num++;
		}
	}
	
	this->redraw_needed = FALSE;
	this->active_dm = 0;
	this->saved_dm = 0xFF;
	this->popup_softkeys = FALSE;
}


PixmapMan::~PixmapMan ()
{
	//	Assumes we never quit
}


/*
 *	Set the function that will be called in order
 *	to obtain the size of the menu and of the softkeys
 *	depending of the size of the screen.
 */

void
PixmapMan::SetGeometryFunction (GetGeomFunc func)
{
	this->get_geometry = func;
}


/*
 *	This must be called each time a pixmap is shown.
 *	The `descr' parameter is the display descriptor of
 *	the pixmap newly shown.
 *
 *	WARNING: No drawing is allowed in this method !
 */

void
PixmapMan::PixmapShown (DisDesc *descr)
{
	for (int i = 0; i < this->dm_num; i++) {
		
		if (descr->chdm == this->disrec[i].channel) {
			
			//	Take essential infos from $DIS_0:
			
			this->desc[i].abs = this->dd0->abs;		// pixmap adress
			this->desc[i].cab = this->dd0->cab;		// idem
			this->desc[i].iix = this->dd0->iix;		// bah... (historical)
			this->desc[i].iiy = this->dd0->iiy;		// y line increment
			this->desc[i].cnb = this->dd0->cnb;		// color number
			this->desc[i].cnp = this->dd0->cnp;		// depth
			this->desc[i].cmd = this->dd0->cmd;		// TRUE if depth > 1
			this->desc[i].cpf = this->dd0->cpf;		// TCOLOR channel
			
			if (descr->iiy == 0) {
				this->desc[i].iiy = 0;				// special case: use accelerated graphics
			}
			
			this->desc[i].now = descr->now;			// window number
			
			this->desc[i].cnr = descr->cnr;			// relative colors ; ness. ?
			this->desc[i].cmr = descr->cmr;
			this->desc[i].cpr = descr->cpr;
			
			this->desc[i].cns = descr->cns;			// stack ; ness. ?
			this->desc[i].cms = descr->cms;
			this->desc[i].cps = descr->cps;
			
			//	Private variables these are OK since not modified
			//	by owner program.
			
			this->desc[i].chhwd = descr->chhwd;			// HWDIS channel
			this->desc[i].chdm = descr->chdm;			// DM channel
			this->desc[i].pflags = descr->pflags;		// bit 0 : no title; bit 1 : no soft-keys
			this->desc[i].pmid = descr->pmid;			// pixmap id
			this->desc[i].pdim = descr->pdim;			// pixmap dim
			this->desc[i].bdraw = descr->bdraw;			// begin draw
			this->desc[i].edraw = descr->edraw;			// end draw
			this->desc[i].drawfuncs = descr->drawfuncs;	// drawing functions
			this->desc[i].var = descr->var;				// misc. stuff (GRA2 extensions ?)
			
//			AfText ("DrawFuncs 0:"); AfX8 ((Card32)(this->dd0->drawfuncs));
//			AfText (" i:"); AfX8 ((Card32)(descr->drawfuncs));
//			AfText (" IIY="); AfX4 (this->desc[i].iiy);
			
			//	The window is the whole pixmap
			
			Box b = { { 0, 0 }, { descr->pdim.dy, descr->pdim.dx } };
			Seg s = { { 0, 0 }, { b.d.dy - 1, b.d.dx - 1 } };
			
			this->desc[i].wind_box = b;
			this->desc[i].clip_box = b;
			this->desc[i].w = s;
			
			//	Other fields
			
			//	Modify $DIS_0: for mouse if there are softkeys
			
			if ( !(descr->pflags & 0x02) )
				this->dd0->i_wind_box.d.dy -= 16;
			
			//	Set pixmapman information
			
			this->is_valid[i] = (descr->abs) ? TRUE : FALSE;
			
//			AfText ("is_valid["); AfX2 (i); AfText ("] = "); AfX2 (this->is_valid[i]); AfCR ();
			
			if (this->desc[i].pmid != this->pixmap_id[i]) {
				
				//	We show another pixmap on this DM
				
				this->pixmap_id[i] = this->desc[i].pmid;
				this->redraw_needed = TRUE;
				
//				AfText ("/other");
				
			} else
				this->redraw_needed = FALSE;
			
			this->active_dm = i;
			
//			AfCR ();
			return;
		}
	}
	
	AfText ("SMA_SMAKY : PixmapMan::PixmapShown, channel not found");
	AfCR ();
}


void
PixmapMan::NotifyPixmapSwap (Card32 pm_id, Card32 address, Card32 iiy)
{
	for (int i = 0; i < 4; i++) {
		if (pm_id == this->desc[i].pmid) {
			this->desc[i].abs = (void*)address;
			this->desc[i].cab = (void*)address;
			this->desc[i].iiy = iiy;
		}
	}
}

void
PixmapMan::PixmapInstalled (DisDesc *descr)
{
}


void
PixmapMan::PixmapDeInstalled (DisDesc *descr)
{
}


/*
 *	Return the drawing interface associated with the DM `number'.
 *	If the DM is not installed, return NULL.
 */

Gra *
PixmapMan::GetGra (Card8 number)
{
	return this->gra[number % 4];
}


/*
 *	Return information on the visible pixmap associated with
 *	DM `number'. If the DM is not installed, or if there is no
 *	pixmap shown by this DM, return FALSE, else TRUE.
 *
 *	If the returned value is TRUE :
 *	- `is_active' is set to TRUE if the pixmap shown with this DM
 *	  is the active one, else to FALSE ;
 *	- `has_menu' and `has_softkeys' are updated ;
 *	- `menu_box' and `softkeys_box' are updated if the associated
 *	  boolean value is set to TRUE.
 */

Bool
PixmapMan::GetInfo (Card8 number, Bool& is_active,
					Bool& has_menu, Bool& has_softkeys,
					Box& menu_box, Box& softkeys_box)
{
	Card8 n = number % 4;
	
	is_active    = FALSE;
	has_menu     = FALSE;
	has_softkeys = FALSE;
	
	if (!this->is_valid[n]) return FALSE;
	
	//	Now we assume that everything is valid for this number
	
	DisDesc *descr = &this->desc[n];
	
	is_active = (n == this->active_dm) ? TRUE : FALSE;
	
	has_menu = !(descr->pflags & 0x01);
	has_softkeys = !(descr->pflags & 0x02) || this->popup_softkeys;
	
	Card32 dim = this->get_geometry ((descr->pdim.dy << 16) + descr->pdim.dx);
	
	if (has_menu) {
		Box b = { { 0, 0 }, { dim >> 16, descr->pdim.dx } };
		menu_box = b;
	}
	
	{
		Box b = { { descr->pdim.dy - (dim & 0xffff), 0 },
				  { dim & 0xffff, descr->pdim.dx } };
		softkeys_box = b;
	}
	
	return TRUE;
}

Bool
PixmapMan::HasSoftkeys (Card8 number)
{
	Card8 n = number % 4;
	if (!this->is_valid[n]) return FALSE;
	return (!(this->desc[n].pflags & 0x02)) || this->popup_softkeys;
}

Bool
PixmapMan::HasMenu (Card8 number)
{
	Card8 n = number % 4;
	if (!this->is_valid[n]) return FALSE;
	return !(this->desc[n].pflags & 0x01);
}


Box
PixmapMan::GetScreenBox ()
{
	if (this->is_valid[this->active_dm]) {
		DisDesc *descr = &this->desc[this->active_dm];
		Box b = { { 0, 0}, { descr->pdim.dy, descr->pdim.dx } };
		return b;
	} else {
//		AfText ("SMA_SMAKY : invalid DM, PixmapMan::GetScreenBox -> ");
//		AfX4 (this->active_dm);
//		AfCR ();
		Box b = { { 0, 0}, { 100, 100} };
		return b;
	}
}

/******************************************************************************/

void
PixmapMan::SaveDesc (DescSave& save, DisDesc* dd)
{
	save.abs      = dd->abs;
	save.iix      = dd->iix;
	save.iiy      = dd->iiy;
	save.wind_box = dd->wind_box;
	save.clip_box = dd->clip_box;
	save.w        = dd->w;
}

void
PixmapMan::RestoreDesc (DescSave& save, DisDesc* dd)
{	
	dd->abs      = save.abs;
	dd->iix      = save.iix;
	dd->iiy      = save.iiy;
	dd->wind_box = save.wind_box;
	dd->clip_box = save.clip_box;
	dd->w        = save.w;
}

/******************************************************************************/

Bool
PixmapMan::SetBufferDrawing (Card8 number)
{
	if (this->saved_dm != 0xFF) {
		AfText ("SMA_SMAKY : PixmapMan::SetBufferDrawing called twice");
		AfCR ();
		return FALSE;
	}
	
	//	Check DM number validity
	
	Card8 n = number % 4;
	if (!this->is_valid[n]) return FALSE;
	
	//	Check depth
	
	DisDesc *dd = &this->desc[n];

//PA//begin
//	Card8 depth = dd->cnp;
//	if ( (depth != 4) && (depth != 1) ) return FALSE;
//PA//end

	//	Computes requested memory
	
	Card32 dim = this->get_geometry ((dd->pdim.dy << 16) + dd->pdim.dx);
	Bool has_menu = !(dd->pflags & 0x01);
	Bool has_softkeys = !(dd->pflags & 0x02);
	Bool pop_softkeys = this->popup_softkeys;
	
	if (!(has_menu || has_softkeys || pop_softkeys)) {
		AfText ("SMA_SMAKY : PixmapMan::SetBufferDrawing, illegal call");
		AfCR ();
		return FALSE;
	}
	
	const Card8 pow2table[33] = { 0, 1, 2, 4, 4,
								  8, 8, 8, 8,
								  16, 16, 16, 16, 16, 16, 16, 16,
								  32, 32, 32, 32, 32, 32, 32, 32,
								  32, 32, 32, 32, 32, 32, 32, 32 };
	
	Card32 pow2depth  = pow2table[dd->cnp];
	Card16 h1 = 0, h2 = 0;
	Card32 buffer_iiy = pow2depth * dd->wind_box.d.dx / 8;
	
	if (has_menu) h1 = dim >> 16;
	if (has_softkeys || pop_softkeys) h2 = dim & 0xffff;
	
	if (h2 > h1) h1 = h2;
	
	Card32 size = buffer_iiy * h1;
	
	//	In case we don't use the softkeys, but we still want to display
	//	them, allocate twice as much memory as needed, so we can save
	//	the background of the screen before displaying the softkeys.
	
	if (pop_softkeys) {
		size *= 2;
	}
	
	//	Try memory allocation
	
	void *mem = (void *)(new char[size]);
	if (mem == NULL) return FALSE;
	
	//	Everything is ok, it is possible to save and update the
	//	descriptor.
	
	PixmapMan::SaveDesc (this->save, dd);
	
	memcpy (&this->saved_desc, dd, sizeof (DisDesc));
	this->saved_desc.check = &this->saved_desc;
	
	dd->iiy = buffer_iiy;
	dd->abs = mem;
	
	Box b = { { 0, 0 }, { h1, dd->pdim.dx } };
	Seg s = { { 0, 0 }, { b.d.dy - 1, b.d.dx - 1 } };
	
	dd->wind_box = b;
	dd->clip_box = b;
	dd->w = s;
	
	this->saved_dm = n;
	
	return TRUE;
}



static void
GraQuickSemLock (DisDesc* dd)
{
	asm volatile ( "movel %0,a0\n\t"
				   "movel %1,a2\n\t"
				   "jsr a0@\n\t"
				 : : "g" (dd->pgrbef), "g" (dd) : "a0", "a2", "d7" );
}


static void
GraQuickSemUnlock (DisDesc* dd)
{
	asm volatile ( "movel %0,a0\n\t"
				   "movel %1,a2\n\t"
				   "jsr a0@\n\t"
				 : : "g" (dd->pgraft), "g" (dd) : "a0", "a2", "d7" );
}


void
PixmapMan::PopupSaveBuffer (Box box, Pos pos)
{
	Card8    n     = this->saved_dm;
	DisDesc* dd    = &this->desc[n];
	Card8    depth = dd->cnp;
	
	if (this->popup_softkeys) {
		
		GraQuickSemLock (dd->GetPDis (dd->now));
		
		DescSave saved;
		void*  abs = (Card8*)dd->abs + dd->wind_box.d.dy*dd->iiy;
		Card16 iiy = dd->iiy;
		
		PixmapMan::SaveDesc (saved, dd);
		PixmapMan::RestoreDesc (this->save, dd);
		
//		AfText ("Save abs="); AfX8 (abs); AfText ("/"); AfX4 (iiy); AfCR();
//		AfText ("  ...abs="); AfX8 (this->save.abs); AfText ("/"); AfX4 (this->save.iiy); AfCR ();
		
		dd->abs = abs;
		dd->iiy = iiy;
		
		Box source = box;				//	dimensions in the window
		Pos dest   = { 0, 0 };			//	destination in memory
		source.o   = pos;				//	origin in the window
		
		//	`dd' describes the second buffer allocated to save the window's
		//	contents.
		
		Gra gra (dd);
		gra.SetMode (LOADDOT);
		
		if (depth > 1) {
			gra.RasterColor (source, dest, this->save.abs, this->save.iiy);
		} else {
			gra.RasterMono (source, dest, this->save.abs, this->save.iiy);
		}
		
		PixmapMan::RestoreDesc (saved, dd);
	}
}

static void
SetSilentMouse (DisDesc* dd, Bool mode)
{
	DisDesc* d = dd->pde[0];
	d->i_silent_mouse = mode;
}

void
PixmapMan::PopupActivate (Box box, Pos pos)
{
	if (g_popup_mode & POPUP_SLOW_ON) {
		Card16 max_dy = box.d.dy;
		Card32 steps  = max_dy / 4;
		
		box.d.dy -= steps * 4;
		pos.y    += steps * 4;
		
		while (steps--) {
			if (box.d.dy) {
				this->DisplayBuffer (box, pos);
			}
			box.d.dy += 4;
			pos.y    -= 4;
			Ntr::Delms (1);
		}
	}
	
	if (box.d.dy) {
		this->DisplayBuffer (box, pos);
	}
	
	DisDesc* dd = &this->desc[this->saved_dm];
	
	PixmapMan::SaveDesc (this->save_popup, dd);
	PixmapMan::RestoreDesc (this->save, dd);
	
	SetSilentMouse (dd, TRUE);
}

static void
fade_away (Box box, Pos pos, Card8 depth, Gra& gra, void* key_abs, void* saved_abs,
		   Card16 iiy, PixmapMan* that)
{
	Box o_box = box;
	Pos o_pos = pos;
	
	if (g_popup_mode & POPUP_SLOW_OFF) {
		Card16 max_dy = box.d.dy;
		Card32 steps  = max_dy / 4;
		
		while (steps--) {
			if (box.d.dy) {
				if (depth > 1) {
					gra.RasterColor (box, pos, key_abs, iiy);
				} else {
					gra.RasterMono (box, pos, key_abs, iiy);
				}
				
				o_box.d.dy = max_dy - box.d.dy;
				
				if (o_box.d.dy) {
					if (depth > 1) {
						gra.RasterColor (o_box, o_pos, saved_abs, iiy);
					} else {
						gra.RasterMono (o_box, o_pos, saved_abs, iiy);
					}
				}
			}
			box.d.dy -= 4;
			pos.y    += 4;
			Ntr::Delms (1);
		}
		
		o_box.d.dy = max_dy;
	}
	
//	AfText ("Fade abs="); AfX8 (saved_abs); AfText ("/"); AfX4 (iiy); AfCR();
		
	if (depth > 1) {
		gra.RasterColor (o_box, o_pos, saved_abs, iiy);
	} else {
		gra.RasterMono (o_box, o_pos, saved_abs, iiy);
	}
}

void
PixmapMan::PopupRestoreBuffer (Box box, Pos pos)
{
	Card8    n     = this->saved_dm;
	DisDesc* dd    = &this->desc[n];
	Card8    depth = dd->cnp;
	
	if (this->popup_softkeys) {
		
		Card16 iiy = this->save_popup.iiy;
		void*  abs = (Card8*)this->save_popup.abs
				   + this->save_popup.wind_box.d.dy * iiy;
		
		Gra gra (&this->saved_desc);
		gra.SetMode (LOADDOT);
		
		fade_away (box, pos, depth, gra, this->save_popup.abs, abs, iiy, this);
		
		//	Restore the state which had been saved in `PopupActivate'. This
		//	is not really useful yet, since only `dd->abs' is used afterwards
		//	in `RestoreBufferDrawing'; nevertheless, do it !
		
		dd->abs = this->save_popup.abs;
		dd->iix = this->save_popup.iix;
		dd->iiy = this->save_popup.iiy;
		dd->wind_box = this->save_popup.wind_box;
		dd->clip_box = this->save_popup.clip_box;
		dd->w = this->save_popup.w;
	}
	
	SetSilentMouse (dd, FALSE);
	GraQuickSemUnlock (dd->GetPDis (dd->now));
}


void
PixmapMan::DisplayBuffer (Box box, Pos pos)
{
	Card8 n = this->saved_dm;
	DisDesc *dd = &this->desc[n];
	Card8 depth = dd->cnp;
	
	Gra gra (&this->saved_desc);
	gra.SetMode (LOADDOT);
	
	if (depth > 1) {
		gra.RasterColor (box, pos, dd->abs, dd->iiy);
	} else {
		gra.RasterMono (box, pos, dd->abs, dd->iiy);
	}
}


void
PixmapMan::RestoreBufferDrawing ()
{
	DisDesc *dd = &this->desc[this->saved_dm];
	
	if (dd->abs) delete dd->abs;
	
	dd->abs = this->save.abs;
	dd->iix = this->save.iix;
	dd->iiy = this->save.iiy;
	dd->wind_box = this->save.wind_box;
	dd->clip_box = this->save.clip_box;
	dd->w = this->save.w;
	
	this->saved_dm = 0xFF;
}


Bool
PixmapMan::ClutChanged (Card8 number)
{
	//	Check DM number validity
	
	Card8 n = number % 4;
	if (!this->is_valid[n]) return FALSE;
	
	//	Test DM dynamic flag
	
	return (this->disrec[n].dyn->clut_chg & 0x01) ? TRUE : FALSE;
}

void
PixmapMan::ClutChangedAck (Card8 number)
{
	//	Check DM number validity
	
	Card8 n = number % 4;
	if (!this->is_valid[n]) return;
	
	//	Reset DM dynamic flag
	
	this->disrec[n].dyn->clut_chg &= ~0x01;
}

