/*
 *	pixmapman.h
 *
 *	Little pixmap manager for SMA_SMAKY.
 *
 *	It provides the user one valid drawing
 *	interface for each visible pixmap.
 *
 *	(C)	1994, 1995 Erik BRUCHEZ
 */


#ifndef _PIXMAP_MAN_H_
#define _PIXMAP_MAN_H_

#include "gra.h"
#include "hwdm.h"

typedef Card32 (*GetGeomFunc)(Card32);

class PixmapMan {

private:
	
	struct DescSave
	{
		void	*abs;
		Card16	iix;
		Card16	iiy;
		Box		wind_box;
		Box		clip_box;
		Seg		w;
	};
	
	DisDesc		*dd0;			// $DIS_0: window descriptor
	
	Card8		dm_num;			// number of DM
	
	DisDesc		desc[4];		// up to 4 display descriptors
	HWDisRec	disrec[4];		// DM info
	Gra			*gra[4];		// drawing interfaces for each visible pixmap
	Card32		pixmap_id[4];	// ids of visible pixmaps
	Bool		is_valid[4];	// information validity
	
	Bool		popup_softkeys;	// do pop-up the softkeys
	
	Bool		redraw_needed;
	Card8		active_dm;		// number of the active DM
	
	DisDesc		saved_desc;		// for buffer drawing
	Card8		saved_dm;
	DescSave	save;
	DescSave	save_popup;		// copy of "save" context for pop-up only
	
	GetGeomFunc	get_geometry;

protected:
	static void SaveDesc (DescSave& save, DisDesc* dd);
	static void RestoreDesc (DescSave& save, DisDesc* dd);
	
public:
	
	PixmapMan ();
	~ PixmapMan ();
	
	//	This MUST be called before using the class
	
	void SetGeometryFunction (GetGeomFunc func);
	
	//	This MUST be called each time a pixmap is shown
	//	in order to obtain valid `GetGra' and `GetInfo'
	//	functions.
	
	void PixmapShown (DisDesc *descr);
	void PixmapInstalled (DisDesc *descr);
	void PixmapDeInstalled (DisDesc *descr);
	
	void NotifyPixmapSwap (Card32 pm_id, Card32 address, Card32 iiy);
	
	//	Inquiry methods
	
	Card8 GetDMNumber () { return this->dm_num; }
	
	Gra *GetGra (Card8 number);
	
	Bool GetInfo (Card8 number, Bool& is_active,
				  Bool& has_menu, Bool& has_softkeys,
				  Box& menu_box, Box& softkeys_box);
	

	Bool HasSoftkeys (Card8 n);
	Bool HasMenu (Card8 n);
	
	Bool RedrawNeeded () { return this->redraw_needed; }
	Card8 GetActiveDM () { return this->active_dm; }
	Box GetScreenBox ();
	
	//	Bufferized drawing
	
	Bool SetBufferDrawing (Card8 number);
	void DisplayBuffer (Box box, Pos pos);
	void PopupSaveBuffer (Box box, Pos pos);
	void PopupActivate (Box box, Pos pos);
	void PopupRestoreBuffer (Box box, Pos pos);
	void RestoreBufferDrawing ();
	
	//	Color management
	
	Bool ClutChanged (Card8 number);
	void ClutChangedAck (Card8 number);
	//16-bit, 24-bit... info
	
	//	Softkeys pop-up management
	
	void SetPopupSoftkeys (Bool x) { this->popup_softkeys = x; }
};

#endif /* _PIXMAP_MAN_H_ */

