/*
 *	smevents.cxx
 *
 *	Event management for SMA_SMAKY
 *
 *	(C)	1994
 *				Pierre ARNAUD, Daniel MARMIER
 *				for the original version
 *
 *		1994, 1995
 *				Erik BRUCHEZ
 *				for the BARs system, code cleaning,
 *				and general maintenance
 *
 *		1997
 *				Pierre ARNAUD
 *				for updates
 */

#include "smevents.h"
#include "gra.h"
#include "res.h"
#include "softkeys.h"
#include "menu.h"
#include "lib.h"
#include "pixmapman.h"
#include "activityspy.h"
#include "clock.h"
#include "fos.h"

#include "mon.h"

/********************************************************************/

extern PixmapMan	pman;
extern SoftKeys*	softkeys[4];
extern Menu*		menu[4];
extern ActivitySpy	activity_spy;
extern Clock		clock;
extern EventBox*	global_event_box;
extern BarEvent*	global_event_pixswap;
extern BarEvent*	global_event_displaysync;
extern Configurator	configurator;

/********************************************************************/

static Box  popup_box;
static Pos  popup_pos;

void
PopupSoftKeysBegin (Card8 dm)
{
	if ( (g_popup_active)
	  || (g_popup_mode == POPUP_NO) ) {
		return;
	}
	
	dm %= 4;
	
	Bool is_active, has_menu, has_softkeys, ok;
	Box  menu_box, softkeys_box;
	
	ok = pman.GetInfo (dm, is_active, has_menu, has_softkeys, menu_box, softkeys_box);
	
	if ( (!ok)
	  || (clock.IsMuted ())
	  || (softkeys[dm]->IsEmpty ())
	  || (has_softkeys)				//	has soft-keys => don't need to pop up !
	  || (!has_menu)) {				//	has no menu => sMACk does not want the pop up !
		return;
	}
	
	pman.SetPopupSoftkeys (TRUE);
	ok = pman.SetBufferDrawing (dm);
	
	if (ok) {
		
		//	Buffer drawing accepted

		Box box  = softkeys_box; //softkeys[dm]->GetBox ();
		Pos spos = softkeys_box.o;
		
		softkeys_box.o.x = 0;
		softkeys_box.o.y = 0;
		
		softkeys[dm]->SizeChanged (softkeys_box);
		softkeys[dm]->SetBox (softkeys_box);
		softkeys[dm]->DrawAll ();
		softkeys[dm]->SetBox (box);
		
		pman.PopupSaveBuffer (softkeys_box, spos);
		pman.PopupActivate (softkeys_box, spos);
		
		popup_box = softkeys_box;
		popup_pos = spos;
		g_popup_active = TRUE;
		
	} else {
		pman.SetPopupSoftkeys (FALSE);
	}
}

void
PopupSoftKeysEnd ()
{
	if (g_popup_active) {
		pman.PopupRestoreBuffer (popup_box, popup_pos);
		pman.RestoreBufferDrawing ();
		pman.SetPopupSoftkeys (FALSE);
		g_popup_active = FALSE;
	}
}

/********************************************************************/

Event::Event (EventType type)
{
	Ntr::GetTimeMS (this->time_stamp);
	this->type = type;
}

Bool
Event::AcceptReplacing () const
{
	return TRUE;
}

/********************************************************************/

void
EvDisplaySync::HandleEvent ()
{
	//	Warning ! In this method, we are synchronized with DIS.
	//	We musn't forget to make a "this->bar.EndAccept (0)" before
	//	returning.
	//	It is also FORBIDDEN TO USE DIS (OR EVEN LIB) ANYWHERE IN THE
	//	EVENT HANDLING FUNCTIONS, EVEN THE ONES THAT ARE NOT SYNCHRO-
	//	NIZED WITH DIS !
	
	int   j;
	Card8 mode;
	
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	this->inst |= 1 << (9 - 1);	//	because DIS forgets it... TEMPORARY
	
	switch (this->type) {
		
		/******************************/
		/*       Pixmap Install       */
		/******************************/
		
		case 1:
			
			pman.PixmapInstalled (this->desc);
			
			for (j = 0; j < pman.GetDMNumber (); j++) {
				is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
										 menu_box, softkeys_box);
				
				menu[j]->SetDisplays ((Card16)(this->inst), 1 << this->active);
				
				if (is_valid && has_menu) menu[j]->DrawDisplays ();
			}
			break;
		
		/********************************/
		/*       Pixmap Deinstall       */
		/********************************/
		
		case 2:
			
			pman.PixmapDeInstalled (this->desc);
			
			for (j = 0; j < pman.GetDMNumber (); j++) {
				is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
										 menu_box, softkeys_box);
				
				menu[j]->SetDisplays ((Card16)(this->inst), 1 << this->active);
				
				if (is_valid && has_menu) menu[j]->DrawDisplays ();
			}
			break;
		
		/*******************************/
		/*       Pixmap Swapping       */
		/*******************************/
		
		case 3:				//	Pixmap Show Phase #1 - drawing allowed
			
			global_event_box->Freeze (TRUE);
			
			mode = g_popup_mode & POPUP_SLOW_OFF;
			g_popup_mode &= ~ POPUP_SLOW_OFF;
			PopupSoftKeysEnd ();				//	turn off pop-up soft keys
			g_popup_mode |= mode;
			break;
		
		case 4:				//	Pixmap Show Phase #2 - update only - no drawing
			
			pman.PixmapShown (this->desc);
			activity_spy.SetBox (pman.GetScreenBox ());
			break;
		
		case 5:				//	Pixmap Show Phase #3 - redraw after switching pixmaps
			
			global_event_box->Freeze (FALSE);
			
			Card8 active = pman.GetActiveDM ();
			Bool  active_drawn = FALSE;
			
			is_valid = pman.GetInfo (active, is_active, has_menu, has_softkeys,
									 menu_box, softkeys_box);
			
			menu[active]->SetDisplays ((Card16)(this->inst), 1 << this->active);
			
			if (is_valid && has_menu) {
				menu[active]->SizeChanged (menu_box);
				menu[active]->DrawAll ();
				active_drawn = TRUE;
			}
			
			if (is_valid && has_softkeys) {
				softkeys[active]->SizeChanged (softkeys_box);
				softkeys[active]->DrawFrame ();
			}
			
			//	Update displays icons on each visible pixmap
			
			for (j = 0; j < pman.GetDMNumber (); j++) {
				
				if ( (j != active) || !(active_drawn) ) {
					is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
											 menu_box, softkeys_box);
					
					menu[j]->SetDisplays ((Card16)(this->inst), 1 << this->active);
					
					if (is_valid && has_menu) menu[j]->DrawDisplays ();
				}
			}
			
			break;
		
		/*******************************/
		/*             ???             */
		/*******************************/
		
		default:
			break;
	}
	
	this->bar->EndAccept (0);
	
	Card16 error;
	
	if (error = this->bar->GetError ()) {
		AfText ("SMA_SMAKY : EndAccept display error ");
		AfX4 (error);
		AfCR ();
	}
}

void
EvDisplaySync::ReadYourself (void *message)
{
	EvDisplaySyncRecord *rec = (EvDisplaySyncRecord *)(message);
	
	this->type = rec->type;
	this->length = rec->length;
	this->inst = rec->inst;
	this->visible = rec->visible;
	this->active = rec->active;
	this->desc = rec->desc;
}

/********************************************************************/

void
EvPixSwapSync::HandleEvent ()
{
	//	Warning ! In this method, we are synchronized with the Display
	//	Manager. We musn't forget to make a "this->bar.EndAccept (0)" before
	//	returning.
	//	It is also FORBIDDEN TO USE DIS, GRA OR EVEN LIB ANYWHERE IN THIS
	//	EVENT HANDLING FUNCTION, since this would lead to a deadlock.
	
	//	This is called whenever a pixmap's configuration (base address and/or
	//	byte width) changes. This allows SMA_SMAKY to update its descriptors.
	
	pman.NotifyPixmapSwap (this->msg->pm_id, this->msg->address, this->msg->iiy);
	
	this->bar->EndAccept (this->msg);
}

void
EvPixSwapSync::ReadYourself (void *message)
{
	this->msg = (EvPixSwapSyncRecord*) message;
}

/********************************************************************/

static void
UpdateColors (Card8 dm)
{
	dm %= 4;
	
	Bool is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	Bool is_valid = pman.GetInfo (dm, is_active, has_menu, has_softkeys,
								  menu_box, softkeys_box);
	
	if (!(has_menu || has_softkeys) || !(is_valid) || (clock.IsMuted ())) return;
	
	//	Redrawing needed
	
	menu[dm]->UpdateLookMan ();
	Bool ok = pman.SetBufferDrawing (dm);
	
	if (ok) {
		
		//	Buffer drawing accepted
		
		if (has_menu) {
			Box box = menu[dm]->GetBox ();
			Pos mpos = menu_box.o;
			
			menu_box.o.x = 0;
			menu_box.o.y = 0;
			menu[dm]->SetBox (menu_box);
			
			menu[dm]->DrawAll ();
			
			menu[dm]->SetBox (box);
			
			pman.DisplayBuffer (menu_box, mpos);
		}
		
		if (has_softkeys) {
			Box box = softkeys[dm]->GetBox ();
			Pos spos = softkeys_box.o;
			
			softkeys_box.o.x = 0;
			softkeys_box.o.y = 0;
			softkeys[dm]->SetBox (softkeys_box);
			
			softkeys[dm]->DrawAll ();
			
			softkeys[dm]->SetBox (box);
			
			pman.DisplayBuffer (softkeys_box, spos);
		}
		
		pman.RestoreBufferDrawing ();
		
	} else {
		
		//	Just redraw on screen
		
		if (has_menu)
			menu[dm]->DrawAll ();
		
		if (has_softkeys)
			softkeys[dm]->DrawAll ();
	}
	
}

/********************************************************************/

void
EvNetworkSync::HandleEvent ()
{
	if (!activity_spy.StillRegistered ())
		activity_spy.NetDriverRegister (this->driver_name, this->rec);
	
	this->bar->EndAccept (0);
	
	Card16 error;
	
	if (error = this->bar->GetError ()) {
		AfText ("SMA_SMAKY : EndAccept network error ");
		AfX4 (error);
		AfCR ();
	}
}

void
EvNetworkSync::ReadYourself (void *message)
{
	//	The message is a pointer to two pointers
	
	Card32	*msg = (Card32 *)(message);
	
	//	The first is a pointer to the complete driver name
	//	without any '$' or ':' (ex. : "Z_0")
	
	this->driver_name = (char *)(msg[0]);
	
	//	The second is a pointer to the counters structure
	
	this->rec = (NetCountRec *)(msg[1]);
}

/********************************************************************/

void
EvServicesSync::HandleEvent ()
{
	//...
	
	this->bar->EndAccept (0);
	Card16 error;
	
	if (error = this->bar->GetError ()) {
		AfText ("SMA_SMAKY : EndAccept user icons error ");
		AfX4 (error);
		AfCR ();
	}
}

void
EvServicesSync::ReadYourself (void *message)
{
	//...
}

/********************************************************************/

static Bool
UpdateClockTextes (ClockMode cm, ClockMode old_cm)
{
	Bool draw_clock = FALSE;
	
	if (cm != old_cm){
		draw_clock = TRUE;
		if (cm == CLOCK_MEM) {
			char t[MEM_BUFFER_SIZE];
			activity_spy.GetMemStats (t);
			clock.SetMemText (t);
		} else if (cm == CLOCK_CPU) {
			char t[CPU_BUFFER_SIZE];
			activity_spy.GetCPUStats (t);
			clock.SetCPUText (t);
		}
	}
	return draw_clock;
}

/********************************************************************/

void
EvSmConfigSync::HandleEvent ()
{
	//	Set configuration and manage particular cases
	
	const ClockMode old_cm = clock.GetMode ();
	const GraphMode old_mem_grmode = activity_spy.GetMemGraphMode ();
	const GraphMode old_cpu_grmode = activity_spy.GetCPUGraphMode ();
	
	Bool ok;
	
	if (ok = configurator.SetConfiguration (&this->config)) {
		const Bool draw_clock = UpdateClockTextes (clock.GetMode (), old_cm);
		const Bool draw_mem = (activity_spy.GetMemGraphMode () != old_mem_grmode);
		const Bool draw_cpu = (activity_spy.GetCPUGraphMode () != old_cpu_grmode);
		
		Bool is_valid, is_active, has_menu, has_softkeys;
		Box menu_box, softkeys_box;
		
		//	Clock / mem / cpu redrawing if modes changed
		
		for (int j = 0; j < pman.GetDMNumber (); j++) {
			is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
									 menu_box, softkeys_box);
			
			if (is_valid && has_menu) {
				if (draw_clock) {
					if (is_valid && has_menu) {
						menu[j]->ReadClock ();
						menu[j]->DrawClock (TRUE);
					}
				}
				
				if (draw_mem)
					menu[j]->DrawMemProc (TRUE, FALSE, TRUE, FALSE);
				
				if (draw_cpu)
					menu[j]->DrawMemProc (FALSE, TRUE, FALSE, TRUE);
			}
		}
	}
	
	this->bar->EndAccept ((void *)(ok));
	Card16 error;
	
	if (error = this->bar->GetError ()) {
		AfText ("SMA_SMAKY : EndAccept config error ");
		AfX4 (error);
		AfCR ();
	}
}

void
EvSmConfigSync::ReadYourself (void *message)
{
	ConfigSet *config = (ConfigSet *)(message);
	
	this->config = *config;
}

/********************************************************************/

void
EvClickInMenu::Replacing (Event* event)
{
	EvClickInMenu *that = (EvClickInMenu *)(event);
	
	this->click_position = that->click_position;
	this->click_button = that->click_button;
	
	this->display_number = that->display_number;
	this->relative_pos = that->relative_pos;
	this->start_pos = that->start_pos;
	this->space = that->space;
	this->icon_dim = that->icon_dim;
}

void
EvClickInMenu::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	Bool draw_clock = FALSE;
	
	//	Changing mode
	
	switch (this->click_position) {
		
		case CLICK_DISMISS_POPUP:
			PopupSoftKeysEnd ();
			break;
		
		case CLICK_SHOW_POPUP:
			PopupSoftKeysBegin (pman.GetActiveDM ());
			break;
		
		case CLICK_IN_LOGO:
			break;
		
		case CLICK_IN_DISPLAY:
			break;
		
		case CLICK_IN_CLOCK:
			
			//	Changing clock mode
			
			const ClockMode old_cm = clock.GetMode ();
			
			if (this->click_button == CLICK_LEFT)
				clock.NextMode ();	//	Click with left button
			else
				clock.PrevMode ();	//	Click with right button
			
			draw_clock = UpdateClockTextes (clock.GetMode (), old_cm);
			
			break;
		
		case CLICK_IN_MEM:
			
			//	Changing memory mode
			
			if (activity_spy.GetMemGraphMode () == GRAPH_MODE_BAR)
				activity_spy.SetMemGraphMode (GRAPH_MODE_HISTOGRAM);
			else
				activity_spy.SetMemGraphMode (GRAPH_MODE_BAR);
			
			break;
		
		case CLICK_IN_CPU:
			
			//	Changing processor time mode
			
			if (activity_spy.GetCPUGraphMode () == GRAPH_MODE_BAR)
				activity_spy.SetCPUGraphMode (GRAPH_MODE_HISTOGRAM);
			else
				activity_spy.SetCPUGraphMode (GRAPH_MODE_BAR);
			
			break;
		
		default:
			
			break;
	}
	
	//	Redrawing
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		if (is_valid && has_menu) {
			
			switch (this->click_position) {
				
				case CLICK_IN_LOGO:
					
					if (is_active) {
						Bool do_it = menu[j]->LogoAction ();
						
						//	Testing color update
						
						if (do_it) {
							UpdateColors (pman.GetActiveDM ());
							Fos::NExecute ("%M_SMALOG.CODE -logo");
						}
					}
					break;
				
				case CLICK_IN_DISPLAY:
					
					if (is_active) {
						menu[j]->CenterMouse (this->display_number, this->relative_pos,
											  this->start_pos, this->space, this->icon_dim);
					}
					break;
					
				case CLICK_IN_CLOCK:
					
					if (draw_clock) {
						menu[j]->ReadClock ();
						menu[j]->DrawClock (TRUE);
					}
					break;
				
				case CLICK_IN_MEM:
					
					menu[j]->DrawMemProc (TRUE, FALSE, TRUE, FALSE);
					break;
				
				case CLICK_IN_CPU:
					
					menu[j]->DrawMemProc (FALSE, TRUE, FALSE, TRUE);
					break;
				
				default:
					
					break;
			}
		}
	}
}

/********************************************************************/

Bool
EvMessageBegin::AcceptReplacing () const
{
	return FALSE;
}

void
EvMessageBegin::Replacing (Event* event)
{
}

void
EvMessageBegin::HandleEvent ()
{
	Card16 maxlen = 0;
	Card16 maxscr = 0;
	
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		if (is_valid) {
			
			if (j == 0) {
				maxlen = menu[j]->GetMsgMaxLen (this->text);
				maxscr = menu[j]->GetMsgMaxScr ();
			}
			
			menu[j]->BeginMessage ();
			menu[j]->SetMsgText (this->text);
			menu[j]->SetMsgIcon (this->icon_number);
			
			if (has_menu) menu[j]->DrawMsgIcon ();
		}
	}
	
	clock.SignalMsgGeom (maxlen, maxscr);
}

void
EvMessageScroll::Replacing (Event* event)
{
	EvMessageScroll* that = (EvMessageScroll *)(event);
	
	if (this->message_id == that->message_id) {
		this->position = that->position;
	} else {
		AfText ("SMA_SMAKY : MessageScroll evt Replacing with different ids !");
		AfCR ();
	}
}

void
EvMessageScroll::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		if (is_valid) {
			menu[j]->SetMsgPos (this->position);
			if (has_menu) menu[j]->DrawMsgText ();
		}
	}
}

Bool
EvMessageEnd::AcceptReplacing () const
{
	return FALSE;
}

void
EvMessageEnd::Replacing (Event* event)
{
}

void
EvMessageEnd::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		if (is_valid) {
			menu[j]->SetEndMessage ();
			if (has_menu) menu[j]->DrawEndMessage ();
		}
	}
}

/********************************************************************/

void
EvSoftKeyChanged::Replacing (Event* event)
{
	EvSoftKeyChanged* that = (EvSoftKeyChanged*)(event);
	
	if (that->bars_changed /*&& (this->bars_changed == FALSE)*/) {	// @PA
		bars_changed = TRUE;
		bars = that->bars;
	}
	
	if (that->text_changed /*&& (this->text_changed == FALSE)*/) {	// @PA
		text_changed = TRUE;
		char *p = & text[0];
		char *q = & that->text[0];
		for (int i = 0; i < 12; i++)
			while (*p++ = *q++);
	}
}

void
EvSoftKeyChanged::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		//	Until now we change the textes & bars only in the
		//	active pixmap...
		
		if (is_valid && is_active) {
			
			softkeys[j]->SetText (this->text);
			softkeys[j]->SetBars (this->bars);
			
			if (has_softkeys) {
				if (text_changed) softkeys[j]->DrawText ();
				if (bars_changed) softkeys[j]->DrawBars ();
			}
		}
	}
}

/********************************************************************/

void
EvSecondElapsed::Replacing (Event* event)
{
	EvSecondElapsed* that = (EvSecondElapsed*)(event);
	
	this->seconds += that->seconds;
}

void
EvSecondElapsed::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	clock.TimeElapsed (this->seconds);
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		if (is_valid && has_menu) {
			Bool clean_up = menu[j]->ReadClock ();
			menu[j]->DrawClock (clean_up);
		}
	}
}

/********************************************************************/

void
EvActivityChanged::Replacing (Event* event)
{
	EvActivityChanged *that = (EvActivityChanged *)(event);
	
	this->mem_changed |= that->mem_changed;
	this->mem_histo_changed |= that->mem_histo_changed;
	this->cpu_changed |= that->cpu_changed;
	this->cpu_histo_changed |= that->cpu_histo_changed;
	this->clut_changed |= that->clut_changed;
	
	for (int i = 0; i < 4; i++)
		this->dm_clut_chg[i] |= that->dm_clut_chg[i];
	
	if (this->net_changed && that->net_changed) {
		
		//	Both `this' and `that' have net_changed. Merge.
		
		for (int i = 0; i < 4; i++) {
			if (this->changed[i] && that->changed[i]) {
				
				//	Both `this' and `that' have changed[i]
				
				this->activity[i] = that->activity[i];
				
			} else if (that->changed[i]) {
				
				//	Only `that' has changed[i]
				
				this->changed[i] = TRUE;
				this->activity[i] = that->activity[i];
			}
		}
	} else if (that->net_changed) {
		
		//	Only `that' has net_changed. Copy.
		
		this->net_changed = TRUE;
		
		for (int i = 0; i < 4; i++) {
			this->changed[i] = that->changed[i];
			this->activity[i] = that->activity[i];
		}
	}
}

void
EvActivityChanged::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	Bool draw_text = FALSE;
	
	//	This is a special case : the memory (or cpu) is displayed as
	//	text too, so the "clock" zone will have to be redrawn. Clock
	//	is given the message to display.
	
	if ( (this->mem_changed) && (clock.GetMode () == CLOCK_MEM) ) {
		char t[MEM_BUFFER_SIZE];
		activity_spy.GetMemStats (t);
		clock.SetMemText (t);
		draw_text = TRUE;
	}
	
	if ( (this->cpu_changed) && (clock.GetMode () == CLOCK_CPU) ) {
		char t[CPU_BUFFER_SIZE];
		activity_spy.GetCPUStats (t);
		clock.SetCPUText (t);
		draw_text = TRUE;
	}
	
	//	Clut changed test
	
	if (this->clut_changed) {
		for (int i = 0; i < 4; i++)
			if (this->dm_clut_chg[i]) UpdateColors (i);
	}
	
	//	Redraw what needs to be redrawn...
	
	const Bool memhisto = (activity_spy.GetMemGraphMode () == GRAPH_MODE_HISTOGRAM);
	const Bool drawmem = (this->mem_histo_changed && memhisto)
					  || (this->mem_changed && !memhisto);
	
	const Bool cpuhisto = (activity_spy.GetCPUGraphMode () == GRAPH_MODE_HISTOGRAM);
	const Bool drawcpu = (this->cpu_histo_changed && cpuhisto)
					  || (this->cpu_changed && !cpuhisto);
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		if (is_valid && has_menu) {
			menu[j]->DrawMemProc (drawmem, drawcpu);
			
			if (draw_text) {
				menu[j]->ReadClock ();
				menu[j]->DrawClock ();
			}
			
			if (this->net_changed) {
				
				if (this->changed[1]) {
					
					//	Sent messages number changed
					
					Bool active = (this->activity[1]) ? TRUE : FALSE;
					
					menu[j]->SetNetOut (active);
				}
				
				if (this->changed[3]) {
					
					//	Received messages received changed
					
					Bool active = (this->activity[3]) ? TRUE : FALSE;
					
					menu[j]->SetNetIn (active);
				}
				
				menu[j]->DrawNet ();
			}
		}
	}
}

/********************************************************************/

void
EvCapsChanged::Replacing (Event* event)
{
	EvCapsChanged* that = (EvCapsChanged*)(event);
	
	this->status = that->status;
}

void
EvCapsChanged::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		menu[j]->SetCaps (status);
		
		if (is_valid && has_menu) menu[j]->DrawCaps ();
	}
}

/********************************************************************/

void
EvUnitChanged::Replacing (Event* event)
{
	EvUnitChanged* that = (EvUnitChanged*)(event);
	
	this->inst		= that->inst;
	this->entered	= that->entered;
}

void
EvUnitChanged::HandleEvent ()
{
	Bool is_valid, is_active, has_menu, has_softkeys;
	Box menu_box, softkeys_box;
	
	for (int j = 0; j < pman.GetDMNumber (); j++) {
		is_valid = pman.GetInfo (j, is_active, has_menu, has_softkeys,
								 menu_box, softkeys_box);
		
		menu[j]->SetUnits (this->inst, this->entered);
		
		if (is_valid && has_menu) menu[j]->DrawUnits ();
	}
}

/********************************************************************/

/*
 *	Implementation of the EventBox class.
 */

EventBox::EventBox ()
{
	this->head    = 0;
	this->tail    = 0;
	this->current = 0;
	this->bar_event_num = 0;
	this->is_frozen = FALSE;
	
	//	Memory zone initialization
	
	Card32 size = EVENT_BOX_DIFF_EVENTS * EVENT_BOX_BIGGEST_EVENT;
	this->mem_zone = this->mem_account.Alloc (size);
	
	for (int i = 0; i < EVENT_BOX_DIFF_EVENTS; i++) {
		this->occupated[i] = FALSE;
	}
	
	//	Sync. selection initialization
	
	this->sel.AddSync (& this->sync, EVENT_BOX_MAX_BAR_EVENTS);
}


EventBox::~EventBox ()
{
	this->lock.Enter ();
	
	Event* temp = this->head;
	Event* next = 0;
	
	while (temp) {
		next = temp->next;
		this->Free (temp);
		temp = next;
	}
	
	this->Free (this->current);
	this->lock.Release ();
}


Bool
EventBox::RegisterBarEvent (const char *bar_name, BarEvent *event, Bool frozen)
{
	if (this->bar_event_num >= EVENT_BOX_MAX_BAR_EVENTS) return FALSE;
	
	this->bars[this->bar_event_num] = new NtrBar (bar_name);
	if (!this->bars[this->bar_event_num]) return FALSE;
	
	this->bar_events[this->bar_event_num] = event;
	event->SetBar (this->bars[this->bar_event_num]);
	
	this->sel.AddAcceptBar (this->bars[this->bar_event_num], this->bar_event_num);
	
	if (frozen) {
		this->frozen_sel.AddAcceptBar (this->bars[this->bar_event_num], this->bar_event_num);
	}
	
	this->bar_event_num++;
	
	return TRUE;
}


Event*
EventBox::Copy (const Event* event)
{
	int    size = (event->Size () + 3) & (~3);
	Event* copy = (Event*)(this->Alloc (size));
	
	if (copy) {
		
		Card32*       p = (Card32*)(copy);
		const Card32* q = (const Card32*)(event);
		
		size = size / 4;
		
		while (size--)
			*p++ = *q++;
	}
	
	return copy;
}


void
EventBox::Free (Event* event)
{
	/*
	if (event)
		this->mem_account.Free (event);
	*/
	
	Card32 p = (Card32)(event) - (Card32)(this->mem_zone);
	Bool error = FALSE;
	
	if ((p % EVENT_BOX_BIGGEST_EVENT) == 0) {
		
		//	Aligned pointer
		
		Card16 i = p / EVENT_BOX_BIGGEST_EVENT;
		
		if (i < EVENT_BOX_DIFF_EVENTS) {
			
			this->occupated[i] = FALSE;
			
		} else {
			
			//	Bad zone
			
			error = TRUE;
		}
	} else {
		
		//	Misaligned pointer
		
		error = TRUE;
	}
	
	
	if (error) {
		
		AfText ("SMA_SMAKY : error in EventBox::Free : ");
		AfX4 (error);
		AfCR ();
	}
}


void *
EventBox::Alloc (Card32 size)
{
	//	Look for an empty zone
	
	for (Card16 i = 0; i < EVENT_BOX_DIFF_EVENTS; i++)
		if (!this->occupated[i]) break;
	
	if (i < EVENT_BOX_DIFF_EVENTS) {
		
		//	Found one...
		
		this->occupated[i] = TRUE;
		
		return this->mem_zone + i * EVENT_BOX_BIGGEST_EVENT;
		
	} else {
		
		//	What ???
		
		return NULL;
	}
}


void
EventBox::PostEvent (const Event& event)
{
	//this->lock.Enter ();
	Event* copy = this->Copy (& event);
	//this->lock.Release ();
	if (copy) {
		
		this->lock.Enter ();
		
		Event* temp   = this->head;
		Bool   is_new = FALSE;
		
		if (copy->AcceptReplacing ()) {
			
			while (temp) {
				if (temp->type == copy->type)
					break;
				
				temp = temp->next;
			}
			
		} else {
			temp = 0;
		}
		
		if (temp == 0) {
			
			//	Insert element into queue, at the very end.
			
			copy->next = 0;
			copy->prev = this->tail;
			is_new     = TRUE;
			
		} else {

			//	Replace an existing event in the queue.
			
			copy->Replacing (temp);
			copy->next = temp->next;
			copy->prev = temp->prev;
		}
		
		(copy->next ? copy->next->prev : this->tail) = copy;
		(copy->prev ? copy->prev->next : this->head) = copy;
		
		this->lock.Release ();
		
		if (temp) {
			this->Free (temp);
		}
		
		if (is_new) {
			this->sync.Signal ();
		}
	}
}


Event*
EventBox::GetEvent ()
{
	Event*  event  = 0;
	Card32  ident  = 0;
	void*   accept = 0;
	NtrSel* select = (this->is_frozen) ? (&this->frozen_sel) : (&this->sel);
	
	Card16 error = select->Sync (ident, accept);
	
	if (error) {
		AfText ("SMA_SMAKY : Sync. error ");
		AfX4 (error);
		AfCR ();
	}
	
	if (ident == EVENT_BOX_MAX_BAR_EVENTS) {
		
		//	Got a signal on the sync. semaphore
		
		this->lock.Enter ();
		
		if (this->current) this->Free (this->current);
		
		this->current = event = this->head;
		
		(event->next ? event->next->prev : this->tail) = event->prev;
		(event->prev ? event->prev->next : this->head) = event->next;
		
		event->next = 0;
		event->prev = 0;
		
		this->lock.Release ();
		
		return event;
		
	} else {
		
		//	Got an accept on bar number `ident'
		
//		if (this->is_frozen) {
//			AfText ("Got frozen event ");
//			AfX2 (ident);
//			AfCR ();
//		}
		
		this->bar_events[ident]->ReadYourself (accept);
		this->bar_events[ident]->bar = this->bars[ident];
		
		event = (Event *)(this->bar_events[ident]);
		
		Ntr::GetTimeMS (event->time_stamp);
		
		return event;
	}
}

