/*
 *	smevents.h
 *
 *	Event management for SMA_SMAKY
 *
 *	(C)	1994 Pierre ARNAUD, Daniel MARMIER
 *			 for the original version
 *
 *			 Erik BRUCHEZ
 *			 for the BARs system and code cleaning
 */

#ifndef	_SMEVENTS_H_
#define	_SMEVENTS_H_

#include "system.h"
#include "ntrel.h"
#include "gesmem.h"
#include "fos.h"
#include "gra.h"
#include "clock.h"
#include "config.h"

/********************************************************************/

enum EventType
{
	EV_DISPLAY_SELECTED		= 1,		//	a display has been selected
	EV_SOFTKEY_CHANGED		= 2,		//	the soft-key menu has changed
	EV_SECOND_ELAPSED		= 3,		//	one second has elapsed
	
	EV_REDRAW_NEEDED		= 4,		//	redrawing is needed

	EV_CAPS_CHANGED			= 5,		//  CAPS key status has changed
	EV_DISP_CHANGED			= 6,		//  display icons have changed
	EV_UNIT_CHANGED			= 7, 		//  unit icons have changed
	
	EV_ACTIVITY_CHANGED		= 8,		//	memory / proc. / net changed
	
	EV_DISPLAY_SYNC			= 9,		//	display pixmap sync. events
	EV_NETWORK_SYNC			= 10,		//	net driver sync. events
	EV_SERVICES_SYNC		= 11,		//	user services communication
	EV_SM_CONFIG_SYNC		= 12,		//	SMA_SMAKY configuration event
	EV_PIXSWAP_SYNC			= 13,		//	display manager "pixmap swapped" event
	
	EV_CLICK_IN_MENU		= 15,		//	mouse click in the menu bar
	
	EV_MESSAGE_BEGIN		= 16,		//	begin message displaying
	EV_MESSAGE_SCROLL		= 17,		//	scroll message
	EV_MESSAGE_END			= 18		//	end of message displaying
};

/********************************************************************/

/*
 *	"Local" event sent in the programm itself.
 */

struct Event
{
	Event*			next;			//	pointer to next event
	Event*			prev;			//	pointer to previous event
	
	EventType		type;			//	type of event
	Card32			time_stamp;		//	time stamp of event
	
public:
	Event (EventType type);
	virtual void Replacing (Event*) = 0;
	virtual Bool AcceptReplacing () const;
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent () = 0;
};

/*
 *	"Distant" event received on a BAR from another programm.
 */

struct BarEvent : Event
{
	NtrBar	*bar;
public:
	BarEvent (EventType type) : Event (type) {}
	NtrBar* GetBar () { return this->bar; }
	void SetBar (NtrBar* bar) { this->bar = bar; }
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent () = 0;
	virtual void ReadYourself (void *message) = 0;
};

/********************************************************************/

struct EvDisplaySyncRecord
{
	Card32		type;
	Card32		length;
	Card32		inst;
	Card32		visible;		// not used
	Card16		active;
	Card16		reserve;
	DisDesc		*desc;
};

struct EvDisplaySync : BarEvent
{
	Card32		type;
	Card32		length;
	Card32		inst;
	Card32		visible;
	Card16		active;
	DisDesc		*desc;
	
public:
	EvDisplaySync () : BarEvent (EV_DISPLAY_SYNC) {}
	
	virtual void Replacing (Event*) {}
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
	virtual void ReadYourself (void *message);
};

/********************************************************************/

struct EvPixSwapSyncRecord
{
	Card32		dm_id;
	Card32		pm_id;
	Card32		address;
	Card32		iiy;
};

struct EvPixSwapSync : BarEvent
{
	EvPixSwapSyncRecord* msg;
	
public:
	EvPixSwapSync () : BarEvent (EV_PIXSWAP_SYNC) {}
	
	virtual void Replacing (Event*) {}
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
	virtual void ReadYourself (void *message);
};

/********************************************************************/

struct NetCountRec
{
	Card32	sent_blocks;	//	number of sent blocks
	Card32	sent_msgs;		//	number of sent messages
	Card32	rcvd_blocks;	//	number of received blocks
	Card32	rcvd_msgs;		//	number of received messages
	
	Card32	res1[6];		//	reserve
	
	Card8	fifo_empty;		//	flag indicating the FIFO is empty
	Card8	res2[3];		//	reserve
};

struct EvNetworkSync : BarEvent
{
	char		*driver_name;
	NetCountRec	*rec;
	
public:
	EvNetworkSync () : BarEvent (EV_NETWORK_SYNC) {}
	
	virtual void Replacing (Event*) {}
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
	virtual void ReadYourself (void *message);
};

/********************************************************************/

struct EvServicesSync : BarEvent
{
	//...
public:
	EvServicesSync () : BarEvent (EV_SERVICES_SYNC) {}
	
	virtual void Replacing (Event*) {}
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
	virtual void ReadYourself (void *message);
};

/********************************************************************/

struct EvSmConfigSync : BarEvent
{
	ConfigSet	config;
public:
	EvSmConfigSync () : BarEvent (EV_SM_CONFIG_SYNC) {}
	
	virtual void Replacing (Event*) {}
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
	virtual void ReadYourself (void *message);
};

/********************************************************************/

enum ClickPosition
{
	CLICK_IN_LOGO,
	CLICK_IN_DISPLAY,
	CLICK_IN_CLOCK,
	CLICK_IN_MEM,
	CLICK_IN_CPU,
	CLICK_SHOW_POPUP,
	CLICK_DISMISS_POPUP
};

enum ClickButton
{
	CLICK_LEFT,
	CLICK_MIDDLE,
	CLICK_RIGHT
};

struct EvClickInMenu : Event
{
	ClickPosition	click_position;
	ClickButton		click_button;
	
	Card16			display_number;	// if CLICK_IN_DISPLAY
	Pos				relative_pos;	// idem
	Pos				start_pos;		// idem
	Card16			space;			// idem
	Dim				icon_dim;		// idem
	
public:
	EvClickInMenu () : Event (EV_CLICK_IN_MENU) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/
/*
enum MessageOp
{
	MESSAGE_BEGIN,
	MESSAGE_SCROLL,
	MESSAGE_END
};
*/
struct EvMessageBegin : Event
{
	Card8	icon_number;
	Card16	message_id;
	char	text[MSG_BUFFER_SIZE];
public:
	EvMessageBegin () : Event (EV_MESSAGE_BEGIN) {}
	
	virtual void Replacing (Event*);
	virtual Bool AcceptReplacing () const;
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

struct EvMessageScroll : Event
{
	Card16	message_id;
	Int16	position;
public:
	EvMessageScroll () : Event (EV_MESSAGE_SCROLL) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

struct EvMessageEnd : Event
{
	Card16	message_id;
public:
	EvMessageEnd () : Event (EV_MESSAGE_END) {}
	
	virtual void Replacing (Event*);
	virtual Bool AcceptReplacing () const;
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvSoftKeyChanged : Event
{
	Bool			text_changed;	//	TRUE => text changed
	Bool			bars_changed;	//	TRUE => bars changed
	
	int				bars;			//	bitset of current bars
	char			text[500];		//	current text
	
public:
	EvSoftKeyChanged () : Event (EV_SOFTKEY_CHANGED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvSecondElapsed : Event
{
	Card16			seconds;		//	1..n seconds elapsed
	
public:
	EvSecondElapsed () : Event (EV_SECOND_ELAPSED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvActivityChanged : Event
{
	Bool			mem_changed;
	Bool			mem_histo_changed;
	Bool			cpu_changed;
	Bool			cpu_histo_changed;
	Bool			net_changed;
	Card32			changed[4];
	Card32			activity[4];
	Bool			clut_changed;
	Bool			dm_clut_chg[4];
public:
	EvActivityChanged () : Event (EV_ACTIVITY_CHANGED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvCapsChanged : Event
{
	Bool	status;

public:
	EvCapsChanged () : Event (EV_CAPS_CHANGED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvUnitChanged : Event
{
	Card16	inst;		// bitset of installed units
	Card16	entered;	// bitset of entered units

public:
	EvUnitChanged () : Event (EV_UNIT_CHANGED) {}

	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/
/*
 *	The event box class copies "local" events and inserts them
 *	into a queue at one end and extracts them at the other end.
 *
 *	It is also able to receive "distant" events on BARs. These
 *	events don't need a special handling (but an "EndAccept" in
 *	the handling method of the event).
 */

#define EVENT_BOX_MAX_BAR_EVENTS (6)

#define EVENT_BOX_DIFF_EVENTS (10)
#define EVENT_BOX_BIGGEST_EVENT (32 * 4 + 500)

class EventBox
{
	NtrSync		sync;									//	synchronisation semaphore
	NtrLock		lock;									//	exclusive lock
	MemAccount	mem_account;							//	memory account
	
	Event*		head;									//	first event in queue
	Event*		tail;									//	last event in queue
	
	Event*		current;								//	current event, will be freed
	
	NtrSel		sel;									//	ntr multi-sync.
	NtrSel		frozen_sel;								//	ntr multi-sync. when frozen
	Bool		is_frozen;								//	TRUE => only receive on "frozen_sel"
	
	Card8		bar_event_num;							//	number of bar events reg.
	BarEvent*	bar_events[EVENT_BOX_MAX_BAR_EVENTS];	//	BAR events register
	NtrBar*		bars[EVENT_BOX_MAX_BAR_EVENTS];			//	associated BARs
	
	//	Events memory zone management
	
	void*		mem_zone;								//	global memory zone
	Bool		occupated[EVENT_BOX_DIFF_EVENTS];		//	memory zone occupation
	
protected:
	Event *Copy (const Event*);
	void Free (Event*);
	void *Alloc (Card32 size);
	
public:
	EventBox ();
	~EventBox ();
	
	void PostEvent (const Event& event);
	Event* GetEvent ();
	
	//	Two kind of event sources exist : the normal and the unfreezable. When the
	//	display swaps pixmaps, all normal event sources are frozen in order to avoid
	//	deadlocks (if a normal event source starts drawing at that time, it will lock
	//	up).
	
	Bool IsFrozen () const { return this->is_frozen; }
	void Freeze (Bool freeze) { this->is_frozen = freeze; }
	
	Bool RegisterBarEvent (const char *bar_name, BarEvent *event, Bool frozen = FALSE);
};

extern EventBox*	global_event_box;

extern void LibEventLoop (Card32 param);

#endif /* _SMEVENTS_H_ */

