/*
 *	smservices.h
 *
 *	Event management for SMA_SMAKY
 *
 *	(C)	1994 Pierre ARNAUD, Daniel MARMIER
 *			 for the original version
 *
 *			 Erik BRUCHEZ
 *			 for the BARs system and code cleaning
 */

#ifndef	_SMSERVICES_H_
#define	_SMSERVICES_H_

#include "system.h"
#include "ntrel.h"
#include "gesmem.h"
#include "gra.h"

/********************************************************************/

enum EventType
{
	EV_DISPLAY_SELECTED	= 1,		//	a display has been selected
	EV_SOFTKEY_CHANGED	= 2,		//	the soft-key menu has changed
	EV_SECOND_ELAPSED	= 3,		//	one second has elapsed
	
	EV_REDRAW_NEEDED	= 4,		//	redrawing is needed

	EV_CAPS_CHANGED		= 5,		//  CAPS key status has changed
	EV_DISP_CHANGED		= 6,		//  display icons have changed
	EV_UNIT_CHANGED		= 7, 		//  unit icons have changed
	
	EV_MEMPROC_CHANGED	= 8,		//	memory occupation has changed
	
	EV_DISPLAY_SYNC		= 9,		//	display pixmap sync. events
	EV_CLICK_IN_MENU	= 10		//	mouse click in the menu bar
};

/********************************************************************/

/*
 *	"Local" event sent in the programm itself.
 */

struct Event
{
	Event*			next;			//	pointer to next event
	Event*			prev;			//	pointer to previous event
	
	EventType		type;			//	type of event
	Card32			time_stamp;		//	time stamp of event
	
public:
	Event (EventType type);
	virtual void Replacing (Event*) = 0;
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent () = 0;
};

/*
 *	"Distant" event received on a BAR from another programm.
 */

struct BarEvent : Event
{
	NtrBar	*bar;
public:
	BarEvent (EventType type) : Event (type) {}
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent () = 0;
	virtual void ReadYourself (void *message) = 0;
};

/********************************************************************/

struct EvDisplaySyncRecord
{
	Card32		type;
	Card32		length;
	Card32		inst;
	Card32		visible;
	Card16		active;
	Card16		reserve;
	DisDesc		*desc;
};

struct EvDisplaySync : BarEvent
{
	Card32		type;
	Card32		length;
	Card32		inst;
	Card32		visible;
	Card16		active;
	DisDesc		*desc;
	
public:
	EvDisplaySync () : BarEvent (EV_DISPLAY_SYNC) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
	virtual void ReadYourself (void *message);
};

/********************************************************************/

struct EvClickInMenu : Event
{
	Bool			click_in_logo;
	Bool			click_in_clock;
	Bool			click_in_mem;
	Bool			click_in_time;

public:
	EvClickInMenu ();
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvSoftKeyChanged : Event
{
	Bool			text_changed;	//	TRUE => text changed
	Bool			bars_changed;	//	TRUE => bars changed
	
	int				bars;			//	bitset of current bars
	char			text[500];		//	current text
	
public:
	EvSoftKeyChanged () : Event (EV_SOFTKEY_CHANGED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvSecondElapsed : Event
{
	int				seconds;		//	1..n seconds elapsed
	
public:
	EvSecondElapsed () : Event (EV_SECOND_ELAPSED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvMemProcChanged : Event
{
	Bool			mem_changed;
	Bool			time_changed;
public:
	EvMemProcChanged () : Event (EV_MEMPROC_CHANGED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvCapsChanged : Event
{
	Bool	status;

public:
	EvCapsChanged () : Event (EV_CAPS_CHANGED) {}
	
	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/

struct EvUnitChanged : Event
{
	Card16	inst;		// bitset of installed units
	Card16	entered;	// bitset of entered units

public:
	EvUnitChanged () : Event (EV_UNIT_CHANGED) {}

	virtual void Replacing (Event*);
	virtual int Size () const { return sizeof (*this); }
	virtual void HandleEvent ();
};

/********************************************************************/
/*
 *	The event box class copies "local" events and inserts them
 *	into a queue at one end and extracts them at the other end.
 *
 *	It is also able to receive "distant" events on BARs. These
 *	events don't need a special handling (but an "EndAccept" in
 *	the handling method of the event).
 */

#define EVENT_BOX_MAX_BAR_EVENTS (4)

#define EVENT_BOX_DIFF_EVENTS (10)
#define EVENT_BOX_BIGGEST_EVENT (32 * 4)

class EventBox
{
	NtrSync		sync;			//	synchronisation semaphore
	NtrLock		lock;			//	exclusive lock
	MemAccount	mem_account;	//	memory account
	
	Event*		head;			//	first event in queue
	Event*		tail;			//	last event in queue
	
	Event*		current;		//	current event, will be freed
	
	NtrSel		sel;			//	ntr multi-sync.
	
	Card8		bar_event_num;	//	number of bar events reg.
	BarEvent	*bar_events[EVENT_BOX_MAX_BAR_EVENTS];	//	BAR events register
	NtrBar		*bars[EVENT_BOX_MAX_BAR_EVENTS];		//	associated BARs
	
	//	Events memory zone management
	
	void		*mem_zone;		//	global memory zone
	Bool		occupated[EVENT_BOX_DIFF_EVENTS];	//	memory zone occupation
	
protected:
	Event *Copy (const Event*);
	void Free (Event*);
	void *Alloc (Card32 size);
	
public:
	EventBox ();
	~EventBox ();
	
	void PostEvent (const Event& event);
	Event* GetEvent ();
	
	Bool RegisterBarEvent (const char *bar_name, BarEvent *event);
};

extern EventBox*	global_event_box;

extern void LibEventLoop (Card32 param);

#endif /* _SMSERVICES_H_ */

