/*
 *	softkeys.cxx
 *
 *	Softkeys class for SMA_SMAKY
 *
 *	There are as many instances of softkeys
 *	as the number of DMs (Display Managers)
 *	installed.
 *
 *	(C)	1994	Daniel MARMIER
 *				for the original version
 *
 *		1994, 1995
 *				Erik BRUCHEZ
 *				for improvements, cleaning and
 *				maintainance
 */

#include "lib.h"
#include "fos.h"
#include "gra.h"
#include "icons.h"
#include "mon.h"
#include "resources.h"
#include "softkeys.h"
#include "lookman.h"
#include "str.h"
#include "smevents.h"
#include "pixmapman.h"

/****************************************************************/

extern PixmapMan pman;

struct SKData
{
	Card16	first_icon_num;		// icon number for F0 frame
	Card16	second_icon_num;	// icon number for others frames
	Card16	margin;				// frame border
	
	Card8	char_height;		// .GENC char height
	Dim		char_dim;			// real char dim
	Card16	char_gap;			// additional gap between two chars (0 or 1)
	
	Card16	bar_height;			// height of a bar
	
	Int16	frame_y_pos;		// y position of the frame (rel. to subwindow)
	Card16	max_space;			// maximal space between two frames
};

static const SKData data_640 = {
	rIconSK640a,				// icon number for F0 frame
	rIconSK640b,				// icon number for others frames
	5,							// frame border
	
	7,							// .GENC char height
	{ 10, 6 },					// real char dim
	0,							// additional gap between two chars (0 or 1)
	
	1,							// height of a bar
	
	3,							// y position of the frame (rel. to subwindow)
	1000						// maximal space between two frames
};

static const SKData data_800 = {
	rIconSK800a,				// icon number for F0 frame
	rIconSK800b,				// icon number for others frames
	6,							// frame border
	
	11,							// .GENC char height
	{ 16, 8 },					// real char dim
	0,							// additional gap between two chars (0 or 1)
	
	1,							// height of a bar
	
	3,							// y position of the frame (rel. to subwindow)
	1000						// maximal space between two frames
};

static const SKData data_832 = {
	rIconSK832a,				// icon number for F0 frame
	rIconSK832b,				// icon number for others frames
	6,							// frame border
	
	11,							// .GENC char height
	{ 16, 8 },					// real char dim
	0,							// additional gap between two chars (0 or 1)
	
	2,							// height of a bar
	
	4,							// y position of the frame (rel. to subwindow)
	1000						// maximal space between two frames
};

static const SKData data_864 = {
	rIconSK864a,				// icon number for F0 frame
	rIconSK864b,				// icon number for others frames
	6,							// frame border
	
	11,							// .GENC char height
	{ 16, 8 },					// real char dim
	0,							// additional gap between two chars (0 or 1)
	
	2,							// height of a bar
	
	4,							// y position of the frame (rel. to subwindow)
	1000						// maximal space between two frames
};

static const SKData data_1024 = {
	rIconSK1024a,				// icon number for F0 frame
	rIconSK1024b,				// icon number for others frames
	6,							// frame border
	
	11,							// .GENC char height
	{ 16, 8 },					// real char dim
	1,							// additional gap between two chars (0 or 1)
	
	3,							// height of a bar
	
	6,							// y position of the frame (rel. to subwindow)
	1000						// maximal space between two frames
};

/****************************************************************/

SoftKeys::SoftKeys (Gra *gra, LookMan	*lookman)
{
	//	Init graphics
	
	this->gra = gra;
	this->lookman = lookman;
	
	this->tbuffer = NULL;
	this->tbuffer_width = 0;
	
	//	Init characters (with help from $DIS_0: !)
	
	Card16 channel;
	DisDesc *dd;
	
	Fos::Open ("$DIS_0:", FOS_OPRD | FOS_OPWR, & channel);
	Fos::RStatus (channel, 4, & dd);
	
	char msg[] = { 0xa1, 'N', 'E' + 128, 7, 0xa0, 'N', 'E' + 128, 7};
	Card32 len = 8;
	
	Fos::WrByte (channel, & len, msg);
	
	this->nef07 = dd->pgc;
	
	msg[3] = 11;
	msg[7] = 11;
	len = 8;
	
	Fos::WrByte (channel, & len, msg);
	
	this->nef11 = dd->pgc;
	
	Fos::Close (channel);
	
	//	Init keyboard
	
	this->kbd_chan = 0;
	
	for (Card16 kbd_num = 1; kbd_num <= 32; kbd_num++) {
		Card16 proc_num;
		const char *proc_name;
		
		Lib::SrcKNP (kbd_num, proc_num, proc_name);
		
		if (Str::Comp ( "FIRST", proc_name)) {
			Lib::SrcCHK (kbd_num, this->kbd_chan);
			break;
		}
	}
	
	if (this->kbd_chan == 0) {
		AfText ("SMA_SMAKY : FIRST keyboard channel not found in SK...");
		AfCR ();
	}
	
	//	Clean-up fields
	
	this->data = NULL;
	this->bars_bitset = 0;
	this->text[0] = '\0';
}


SoftKeys::~SoftKeys ()
{
/*	Assuming we never quit ...
 *
 *	if (this->frame_image) delete this->frame_image;
 *	if (this->gra) delete this->gra;
 */
}


void
SoftKeys::SetBox (Box box)
{
	this->box = box;
	
	//	Set icons boxes
	
	Dim icon_dim1, icon_dim2;
	this->lookman->GetIconSize (this->data->first_icon_num, icon_dim1);
	this->lookman->GetIconSize (this->data->second_icon_num, icon_dim2);
	
	const Card16 space = box.d.dx - icon_dim1.dx - 5 * icon_dim2.dx;
	const Card16 nspace = space / 7;
	const Card16 rspace = space % 7;
	const Card16 espace = rspace ? (nspace + (rspace + 1) / 2) : (nspace);
	
	const Box box0 = { { box.o.y + this->data->frame_y_pos, box.o.x + espace }, icon_dim1 };
	this->icons_boxes[0] = box0;
	
	for (int i = 0; i < 5; ) {
		const Card16 px = box.o.x + espace
						  + icon_dim1.dx + nspace
						  + i * (nspace + icon_dim2.dx);
		const Box box1 = { { box.o.y + this->data->frame_y_pos, px }, icon_dim2 };
		
		this->icons_boxes[++i] = box1;
	}
	
	//	Set textes boxes
	
	const Card16 charwidth = this->data->char_dim.dx + this->data->char_gap;
	const Card16 twidth1 = 5 * charwidth;
	const Card16 twidth2 = 17 * charwidth;
	
	const Card16 margin = this->data->margin;
	const Card16 interior1x = icon_dim1.dx - 2 * this->data->margin;
	const Card16 interior2x = icon_dim2.dx - 2 * this->data->margin;
	const Card16 trpos1x = this->data->margin + (interior1x - twidth1) / 2;
	const Card16 trpos2x = margin + (interior2x - twidth2) / 2;
	
	const Card16 interiory = icon_dim1.dy - 2 * margin;
	const Card16 trposy = (interiory - 2 * this->data->char_dim.dy) / 3;
	const Card16 trpos1y = margin+ trposy;
	const Card16 trpos2y = margin + interiory - this->data->char_dim.dy - trposy;
	
	Box tbox0 = { { this->icons_boxes[0].o.y + trpos1y, this->icons_boxes[0].o.x + trpos1x },
				  { this->data->char_dim.dy, twidth1 } };
	this->textes_boxes[0] = this->textes_boxes[1] = tbox0;
	this->textes_boxes[1].o.y = this->icons_boxes[0].o.y + trpos2y;
	
	for (int j = 0; j < 5; j++) {
		Box tbox1 = { { this->icons_boxes[0].o.y + trpos1y, this->icons_boxes[j + 1].o.x + trpos2x },
					  { this->data->char_dim.dy, twidth2 } };
		this->textes_boxes[2 + 2 * j] = this->textes_boxes[3 + 2 * j] = tbox1;
		this->textes_boxes[3 + 2 * j].o.y = this->icons_boxes[0].o.y + trpos2y;
	}
	
	//	Set bars boxes
	
	const Box b = { { this->icons_boxes[0].o.y + this->icons_boxes[0].d.dy + 1,
					  this->textes_boxes[0].o.x },
				    { this->data->bar_height,
					  twidth1 } };
	this->bars_boxes[0] = b;
	
	for (int k = 0; k < 5; k++) {
		const Card16 w1 = 5 * charwidth + charwidth / 2;
		const Card16 w2 = 6 * charwidth;
		Box b = { { this->icons_boxes[0].o.y + this->icons_boxes[0].d.dy + 1,
					this->textes_boxes[2 + 2 * k].o.x },
				  { this->data->bar_height,
					 w1 } };
		this->bars_boxes[1 + 3 * k] = b;
		this->bars_boxes[2 + 3 * k] = b;
		this->bars_boxes[3 + 3 * k] = b;
		
		this->bars_boxes[2 + 3 * k].o.x += w1;
		this->bars_boxes[2 + 3 * k].d.dx = w2;
		this->bars_boxes[3 + 3 * k].o.x += w1 + w2;
	}
}


void
SoftKeys::SizeChanged (Box box)
{
	//	Set appropriate data
	
	if (box.d.dx >= 1024)
		this->data = & data_1024;
	else if (box.d.dx >= 864)
		this->data = & data_864;
	else if (box.d.dx >= 832)
		this->data = & data_832;
	else if (box.d.dx >= 800)
		this->data = & data_800;
	else
		this->data = & data_640;
	
	//	Set buffer for text drawing
	
	Dim dim;
	this->lookman->GetIconSize (this->data->second_icon_num, dim);
	
	if (this->tbuffer) delete this->tbuffer;
	this->tbuffer_width = (dim.dx + 7) / 8;
	this->tbuffer = new char[this->tbuffer_width * this->data->char_dim.dy];
	
	//	Set boxes
	
	this->SetBox (box);
}

void
SoftKeys::SetText (const char* newtext)
{
	Bool  empty = TRUE;
	char  c;
	char* t = this->text;
	
	for (int i = 0; i < 12; i++) {
		while (c = *newtext++) {
			*t++ = c;
			if (c != ' ') empty = FALSE;
		}
		*t++ = c;
	}
	
	this->is_empty = empty;
}

void
SoftKeys::SetBars (Card32 newbars)
{
	this->bars_bitset = newbars;
}


void
SoftKeys::DrawAll ()
{
	this->DrawFrame ();
	this->DrawText ();
	this->DrawBars ();
}


void
SoftKeys::DrawText ()
{
	const int size[12]  = { 5,5,17,17,17,17,17,17,17,17,17,17 };
	
	if (this->data->char_height == 7)
		this->gra->SetGencar (this->nef07);
	else
		this->gra->SetGencar (this->nef11);
	
	const char* text = this->text;
	
	if (this->lookman->UseColor ()) {
		this->lookman->SetColorBack (SSINSK);
		this->lookman->SetColorFore (SSSKTXT);
	}
	
	for (int i = 0; i < 12; i++) {
		
		const char* temp = text;
		int         len  = 0;
		int         filla = 0;
		int			fillb = 0;
		
		char line[100];
		char *ptr = line;
		
		Bool has_spec_cars = FALSE;
		
		while (*temp) {
			if ((*temp & 0x80) == 0)
				len++;
			else
				has_spec_cars = TRUE;
			
			temp++;
		}
		
		if (len <= size[i]) {
			
			//	Text ok
			
			filla = (text[0] != ' ') ? (size[i] - len) / 2 : 0;
			fillb = size[i] - len - filla;
		} else {
			
			//	Text too long
			
			len = size[i];
		}
		
		Pos p = this->textes_boxes[i].o;
		p.y += this->gra->GetGencarAscent ();
		
		if (!has_spec_cars) {
			
			//	The text is raw text without control characters
			
			while (filla--) *ptr++ = ' ';
			
			while (*text) {
				if (len-- > 0) *ptr++ = *text;
				text++;
			}
			
			while (fillb--) *ptr++ = ' ';
			
			*ptr = '\0';
			
			text++;
			
			this->gra->SetMode (LOADDOT);
			
			//	Try optimized text drawing if possible
			
			if (this->tbuffer && !this->data->char_gap)
				this->gra->DrawText (p, line, this->tbuffer, this->tbuffer_width);
			else
				this->gra->DrawText (p, line, this->data->char_gap);
			
		} else {
			
			//	The text contains control characters
			//	The only control characters allowed are
			//	AFDIR, AFINV, AFFULL, AFGREY
			
			this->gra->SetMode (LOADDOT);
			
			while (filla--)
				this->gra->DrawChar (p, ' ', this->data->char_gap);
			
			Card16 mode = LOADDOT;
			
			while (*text) {
				char c = *text++;
				if (c & 0x80) {
					switch ((Card8)(c)) {
						case 0x90 :		// AFDIR
							
							mode &= ~CPLDOT;
							break;
						
						case 0x91 :		// AFINV
							
							mode |= CPLDOT;
							break;
						
						case 0xb6 :		// AFFULL
							
							mode &= ~GREDOT;
							break;
						
						case 0xb7 :		// AFGREY
							
							mode |= GREDOT;
							break;
						
						default :
							
							AfCR ();
							AfText ("SMA_SMAKY : illegal char. in softkeys :");
							AfX2 (c);
							AfCR ();
							return;
							break;
					}
					this->gra->SetMode (mode);
				} else if (len-- > 0) {
					this->gra->DrawChar (p, c, this->data->char_gap);
				}
			}
			
			this->gra->SetMode (LOADDOT);
			
			while (fillb--)
				this->gra->DrawChar (p, ' ', this->data->char_gap);
			
			text++;
		}
	}
	
	if (this->lookman->UseColor ())
		this->lookman->RestoreColors ();
}

void
SoftKeys::DrawBars ()
{
	if (this->lookman->UseColor ()) {
		this->lookman->SetColorBack (SSBGSK);
		this->lookman->SetColorFore (SSSKBARS);
	}
	
	for (int i = 0; i < 16; i++)
		if (bars_bitset & (1 << i))
			gra->Set (this->bars_boxes[i]);
		else
			gra->Clear (this->bars_boxes[i]);
	
	if (this->lookman->UseColor ())
		this->lookman->RestoreColors ();
}

void
SoftKeys::DrawFrame ()
{
	Box line = this->box;
	line.d.dy = 0;
	
	if (this->lookman->UseColor ())
		this->lookman->SetColorBack (SSBGSK);
	
	this->gra->SetMode (LOADDOT);
	this->gra->Clear (this->box);
	
	this->lookman->DrawIcon (this->data->first_icon_num, this->icons_boxes[0].o);
	
	for (int i = 1; i <= 5; i++)
		this->lookman->DrawIcon (this->data->second_icon_num, this->icons_boxes[i].o);
	
	if (this->lookman->UseColor ()) {
		this->gra->SetMode (SETDOT);
		
		this->lookman->SetColorFore (SSBLACK);
		this->gra->Line (line);
		
		Box b = this->box;
		b.o.y++;
		b.d.dy--;
		this->lookman->DrawRectUp (b);
		
		this->lookman->RestoreColors ();
	} else {
		this->gra->SetMode (SETDOT);
		this->gra->Line (line);
	}
}


Bool
SoftKeys::PosInBox (Pos pos, Box box)
{
	if ( (pos.y < box.o.y)
	  || (pos.y >= box.o.y + box.d.dy)
	  || (pos.x < box.o.x)
	  || (pos.x >= box.o.x + box.d.dx) )
	
		return FALSE;
	else
		return TRUE;
}

/******************************************************************************/

Card8 g_popup_mode   = POPUP_NO;
Bool  g_popup_active = FALSE;

static void
dismiss_popup ()
{
	if (g_popup_active) {
		EvClickInMenu ev;
		ev.click_position = CLICK_DISMISS_POPUP;
		ev.click_button = CLICK_LEFT;
		global_event_box->PostEvent (ev);
	}
}

static void
show_popup ()
{
	if (!g_popup_active) {
		EvClickInMenu ev;
		ev.click_position = CLICK_SHOW_POPUP;
		ev.click_button = CLICK_LEFT;
		global_event_box->PostEvent (ev);
	}
}

/******************************************************************************/

Bool
SoftKeys::Click (Pos pos, Card8 clicked, Card8 pressed)
{
	static Bool above_bottom = TRUE;
	Bool has_softkeys = pman.HasSoftkeys (pman.GetActiveDM ());
	
	//	Check if the click is in our box
	
	Card16 dy = pman.GetScreenBox ().d.dy;
	
	if ( (!above_bottom)
	  && (pos.y < (dy - 40)) ) {
		above_bottom = TRUE;
	}
	
	if ( (!pressed)
	  && (pos.y >= (dy - 3))
	  && (above_bottom)
	  && (g_popup_mode & POPUP_YES) ) {
		show_popup ();
		above_bottom = FALSE;
		goto ok;
	}
	
	if ( (has_softkeys)
	  && (!this->PosInBox (pos, this->box))
	  && (pos.y < (dy - 3)) ) {
		dismiss_popup ();
		return FALSE;
	}
	
ok:	if (has_softkeys == FALSE) return FALSE;
	
	//	Accept only click with left button
	
	if (!(clicked & BUTTON_LEFT)) return TRUE;
	
	//	Find the key
	
	Card32 key = F0;
	
	Box b1 = this->textes_boxes[0];
	Box b2 = this->textes_boxes[1];
	b1.o.x = b2.o.x = this->box.o.x;
	b1.d.dx = b2.d.dx = this->box.d.dx;
	
	if (PosInBox (pos, b1))
		key += SHIFT;
	else if (PosInBox (pos, b2))
		/* nothing */;
	else {
		Lib::Beep (BIPLIT);
		return TRUE;
	}
	
	for (int i = 0; i < 16; i++) {
		const Int16 p1 = this->bars_boxes[i].o.x;
		const Int16 p2 = p1 + this->bars_boxes[i].d.dx;
		if ( (pos.x >= p1) && (pos.x < p2) ) {
			key += i;
			
			char buffer[4];
			
			buffer[0] = 0x0a;				// KEYSND command
			buffer[1] = (char)(key & 0xff);	// character
			buffer[2] = 0;					// super-shift
			buffer[3] = '\0';
			
			if (this->kbd_chan) {
				if (g_popup_mode & POPUP_CLICK_OFF) dismiss_popup ();
				Fos::Command (this->kbd_chan, buffer);
			}
			
			return TRUE;
		}
	}
	
	Lib::Beep (BIPLIT);
	return TRUE;
}

