/*
 *	biosdriver.cc
 *
 *	Implmentation d'un pilote BIOS nomm XDRI. Il permet d'implmenter des
 *	external file systems avec la version 8 du FOS sur Smaky.  Il ncessite
 *	le systme 10.
 *
 *	(C) Copyright 1995, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#pragma implementation "biosdriver.h"

#include "portable.h"
#include "biosdriver.h"
#include "gesmem.h"
#include "ntrel.h"
#include "mon.h"
#include "efs.h"

union NetMisc
{
	Card8		name[4];
	Card16		val_16[2];
	Card32		val_32[1];
};

struct NetRequest
{
	Card16		fos;
	Card16		revision;
	Card16		priority;
	Card16		timeout;
	Card16		port;
	NetMisc		param;
};

#define	pretend_to_use(x)	asm volatile ("" : : "g" (&x))


/*
 *	Driver "XDRI", nouvelle mouture crite en C++. Vive l'volution
 *	du systme et du BIOS.
 */

const char*	BIOSDriver::driver_name = "XDRI";
Card8		BIOSDriver::driver_attr = BIOS_ATTR_RDOK+BIOS_ATTR_WROK;
Card8		BIOSDriver::driver_prio = 0;
Card8		BIOSDriver::driver_type = BIOS_TYPE_NETW;
Card16		BIOSDriver::driver_base = 0x0090;


/*
 *	Alloue une nouvelle instance du driver XDRI.
 */

Int16
new_bios_driver (Card32 type, Card32 desc, BIOSDriver* & driver, Card32 num)
{
    Int16 error = 0x8231;
    
    if (num != 0) {
    	return error;
    }
    
    error  = 0;
    driver = new (type, desc) BIOSDriver (error, num);
    
    if (driver == 0) error = 0x8300;					//	erreur GESMEM
    
    return error;
}


/*
 *	Oprateur NEW utilis par la classe BIOSDriver pour allouer
 *	une instance de celle-ci. On utilise btement GESMEM.
 */

static Card32 global_mem_type = 0;
static Card32 global_mem_desc = 0;

void*
BIOSDriver::operator new (long int size, Card32 type, Card32 desc)
{
	global_mem_type = type;
	global_mem_desc = desc;
	
	MemAccount account = MemAccount (desc, type);
	void*      memory  = account.Alloc (size);
	
    return memory;
}

void
BIOSDriver::operator delete (void* ptr)
{
	MemAccount account = MemAccount (global_mem_desc, global_mem_type);
	account.Free (ptr);
}

extern "C" void abort ();
extern "C" void* memcpy (void* dst, const void* src, long unsigned size);

void abort ()
{
	AfText ("ABORT--fatal.\r");
}

void*
memcpy (void* dst, const void* src, long unsigned size)
{
	Card8*       d = (Card8*)(dst);
	const Card8* s = (const Card8*)(src);
	
	while (size--) *d++ = *s++;
	return dst;
}

void*
operator new (long unsigned size)
{
	MemAccount account = MemAccount (global_mem_desc, global_mem_type);
	return account.Alloc (size);
}


void
operator delete (void* ptr)
{
	MemAccount account = MemAccount (global_mem_type, global_mem_desc);
	account.Free (ptr);
}


/****************************************************************************/
/*																			*/
/*						  POINTS D'ENTREE DU DRIVER XDRI					*/
/*																			*/
/****************************************************************************/

BIOSDriver::BIOSDriver (Int16& error, Card32 num)
{
    error = 0;
    
    for (int port = 1; port < MAXPORT; port++) {
	    this->is_open[port] = FALSE;
	    this->command[port].count = 0;
	    this->command[port].write_in_progress = FALSE;
    }
    
    for (int efs = 0; efs < MAXEFS; efs++) {
    	this->efs[efs] = 0;
    }
    
    this->is_open[0] = TRUE;							//	ne jamais utiliser le port 0
}

BIOSDriver::~BIOSDriver ()
{
}

/*
 *	Ouvre un port de communication. On peut en ouvrir au plus
 *	MAXPORT-1 simultanment.
 *
 *	Le numro de port retourn doit avoir le bit 7 mis, de telle
 *	manire que le FOS travaille de manire synchrone pour les
 *	commandes : on se passe du processus de rception du FOS !
 */

Int16
BIOSDriver::Open (Card16& port)
{
	Int16 error = 0x8971;								//	plus assez de ports
	
	if (port < MAXPORT) {
		AutoLock lock = AutoLock (this->port_lock);
		int      iter = 0;
		
		pretend_to_use (lock);
		
		if (port) {
			if (this->is_open[port]) return 0x8972;		//	port dj occup
			this->is_open[port] = TRUE;
			port |= 0x0080;
			return 0x0000;								//	ok, port accept
		}
		
		for (iter = 1; iter < MAXPORT; iter++) {
			if (this->is_open[iter]) continue;
			this->is_open[iter] = TRUE;
			port  = iter | 0x0080;
			return 0x0000;
		}
	}
	
    return error;
}



Int16
BIOSDriver::Command (const char* cmd)
{
    return 1;
}

Int16
BIOSDriver::ReadStatus (void* buffer, Card32 length)
{
    return 1;
}


/*
 *	Le FOS transmet une requte. Celle-ci est simplement prise en
 *	compte si l'on sait l'excuter. C'est au READ que la rponse
 *	et le travail seront rellement excuts.
 *
 *	Il est vital que le WRITE se fasse avant le READ : il faut donc
 *	utiliser un port synchrone.
 */

Int16
BIOSDriver::WriteData (const void* buffer, Card32 pos, Card32 length, Card32 net)
{
	const NetSegment* segment = (const NetSegment*)(buffer);
	const NetRequest* request = (NetRequest*)(segment[0].address);
	
	Card16 port      = pos & 0x007F;
//	Card16 msg_count = (pos >> 16) & 0x00FF;
	Card32 num_seg   = length;
	Card16 dest_port = net & 0x007F;
	Card16 dest_id   = net >> 16;
	
	if (dest_id > MAXEFS) return 0x8912;				//	adresse invalide
	
	if (dest_port == 'W') {
		
		//	Cas particulier : le port vient de recevoir des donnes en
		//	provenance d'un Write en cours.
		
		if (this->command[port].write_in_progress == FALSE) return 0x0003;
		if (this->command[port].write_length != segment[0].length) return 0x0004;
		
		this->command[port].count = 1;
		this->command[port].write_has_occurred = TRUE;
		this->command[port].write_data         = segment[0].address;
		return 0;
	}
	
	if (num_seg != 1) return 0x0001;
	if (dest_port != 1) return 0x0002;
	
	switch (request->fos) {
		case 0x0000:	//	?OPEN
		case 0x0006:	//	?CREATE
		case 0x006F:	//	?nDELETE
		case 0x0008:	//	?RENAME
		case 0x0055:	//	?MOVE
		case 0x000A:	//	?CHATR
		case 0x005F:	//	?IFOPEN
		case 0x003C:	//	?SARGS
		case 0x0009:	//	?CLEAR
		case 0x002F:	//	?DISK
		case 0x0021:	//	?CDIR
		case 0x0001:	//	?RDBYTE
		case 0x8002:	//	?WRBYTE (quick)
		case 0x0002:	//	?WRBYTE (general)
		case 0x0004:	//	?SPOS
		case 0x0003:	//	?GPOS
		case 0x0035:	//	?TRUNC
		case 0x0065:	//	?MAJBD
		case 0x0005:	//	?CLOSE
		case 0x000C:	//	?ENTER
		case 0x000D:	//	?RELEASE
		case 0x0057:	//	?GETHACH
		case 0x004E:	//	?LISTOP
		case 0x0050:	//	?LISTBUF
		case 0x0051:	//	?LISTCLO
		case 0x004D:	//	?REMOTE
			break;
		
		default:		//	pas support !
			return 0x85FE;
	}
	
	this->command[port].count = 1;
	this->command[port].segment[0] = segment[0];
	this->command[port].segment[0].address = this->command[port].buffer;
	
	memcpy (this->command[port].segment[0].address, segment[0].address, segment[0].length);
	
	return 0;
}


/*
 *	Rcupre le rsultat d'une requte transmise par un WRITE
 *	sur le mme port, un peu plus tt.
 */

Int16
BIOSDriver::ReadData (void* buffer, Card32 pos, Card32 length, Card32 net)
{
	Card16 port      = pos & 0x007F;
//	Card16 msg_count = (pos >> 16) & 0x00FF;
	Card32 num_seg   = length;
//	Card16 dest_port = net & 0x007F;
	Card16 dest_id   = net >> 16;
	
	Int16  error     = 0x0000;
	
	NetSegment* segment = (NetSegment*)(buffer);
	NetRequest* request = (NetRequest*)(this->command[port].segment[0].address);
	
	if (this->command[port].count == 0) {
		segment[num_seg-1].length = 0;
		return 1;
	}
	
	Card16* c16 = (Card16*)(segment[0].address);
	Card32* c32 = (Card32*)(segment[0].address);
	Int16*  i16 = (Int16*)(segment[0].address);
//	Int32*  i32 = (Int32*)(segment[0].address);
	
	//	Le REMOTE est utilis pour envoyer des commandes de configuration
	//	au driver XDRI. Il peut tre utilis pour monter des file systems.
	
	if (request->fos == 0x004D)	{					//	?REMOTE
		
		const char* cmd = (const char*)request->param.name;
		const char* arg = (const char*)cmd;
		
		while (*arg++) {
			//	saute la commande, la suite correspond  l'argument
		}
		
		this->ParseCommand (cmd, arg, dest_id, i16[0]);
		
		c16[1] = 0;
		this->command[port].count = 0;
		segment[num_seg-1].length = 0;
		return 0x0000;								//	pas de problme  signaler
	}
	
	if (this->efs[dest_id] == 0) {
		this->command[port].count = 0;
		segment[num_seg-1].length = 0;
		return 0x8952;								//	station inexistante
	}
	
	volatile AutoLock lock = AutoLock (this->efs_lock[dest_id]);
	EFS*     efs  = this->efs[dest_id];
	
	pretend_to_use (lock);
	i16[0] = 0x8888;
	
	if ( this->command[port].write_in_progress
	  && this->command[port].write_has_occurred ) {
		
		//	Le FOS a fait le ?WRBYTE et le premier dialogue s'est bien droul.
		//	Maintenant, nous avons reu un beau segment avec les donnes. Pourvu
		//	que personne ne les dsalloue entre le BIOS?DWRITE et le ?DREAD.
		
		c32    = (Card32*)(c16 + 1);
		c32[0] = this->command[port].write_length;
		this->command[port].write_efs->WrByte ( this->command[port].write_channel, c32[0],
												this->command[port].write_data, i16[0] );
		
		goto end_of_read;
	}
	
	efs->ClearBALMessage ();
	
	switch (request->fos) {
		
		case 0x0000:	//	?OPEN
			efs->Open ( request->param.name + 8,
					    request->param.val_32[0],
					    request->param.val_32[1],
					    c16[1], i16[0] );
			
			c16[2] = 0;
			c16[3] = 0;
			break;
		
		case 0x0006:	//	?CREATE
			efs->Create ( request->param.name + 8,
						  request->param.val_32[0],
						  request->param.val_16[2],
						  request->param.val_16[3],
						  i16[0] );
			
			c16[1] = 0;
			break;
		
		case 0x006F:	//	?nDELETE
			efs->Delete ( request->param.name + 8,
						  request->param.val_32[0],
						  i16[0] );
			
			c16[1] = 0;
			break;
		
		case 0x0008:	//	?RENAME
			efs->Rename ( request->param.name, i16[0] );
			c16[1] = 0;
			break;
		
		case 0x0055:	//	?MOVE
			efs->Move ( request->param.name, i16[0] );
			c16[1] = 0;
			break;
		
		case 0x000A:	//	?CHATR
			efs->Chatr ( request->param.name + 4,
						 request->param.val_32[0],
						 i16[0] );
			
			c16[1] = 0;
			break;
		
		case 0x005F:	//	?IFOPEN
			efs->IfOpen ( request->param.name, i16[0] );
			c16[1] = 0;
			break;
		
		case 0x003C:	//	?SARGS
			efs->Args ( request->param.name + 2,
						request->param.val_16[0],
						segment[1].address,
						c16[1], i16[0] );
			
			c16[2] = 0;
			segment[1].length = 64;
			break;
		
		case 0x0009:	//	?CLEAR
			efs->Clear ( request->param.name, i16[0] );
			c16[1] = 0;
			break;
		
		case 0x002F:	//	?DISK
			c32 = (Card32*)(c16 + 1);
			efs->Disk ( request->param.name, c32[0], c32[1], i16[0] );
			c16[5] = 0;
			break;
		
		case 0x0021:	//	?CDIR
			efs->CDir ( request->param.name + 6,
						request->param.val_32[0],
						i16[0] );
			c16[1] = 0;
			break;
		
		case 0x0001:	//	?RDBYTE
			c32    = (Card32*)(c16 + 1);
			c32[0] = ((Card32*)(request->param.val_16 + 1))[0];		//	longueur  lire
			efs->RdByte ( request->param.val_16[0],					//	canal
						  c32[0],									//	longueur  lire/lue
						  segment[1].address,						//	buffer  remplir
						  i16[0] );
			c16[3] = 0;
			segment[1].length = c32[0];
			break;
		
		case 0x8002:	//	?WRBYTE (quick)
			c32    = (Card32*)(c16 + 1);
			c32[0] = ((Card32*)(request->param.val_16 + 1))[0];		//	longueur  crire
			efs->WrByte ( request->param.val_16[0],					//	canal
						  c32[0],									//	longueur  crire/crite
						  (Card8*)(request->param.val_16 + 3),		//	buffer avec donnes
						  i16[0] );
			c16[1] = 0;
			break;
		
		case 0x0002:	//	?WRBYTE (general)

			//	Le droulement d'une criture se fait comme suit :
			//	
			//	1. Le FOS demande un transfert d'un (gros) bloc de donnes
			//	   pour lequel XDRI devrait allouer de la mmoire. Mais
			//	   comme tout se fait en local, le buffer devient inutile.
			//
			//	2. XDRI ouvre un port spcial 'W', qu'il va communiquer  FOS,
			//	   afin de recevoir les donnes  crire. C'est la seule
			//	   rponse que le FOS reoit en ce moment.
			//
			//	3. Le FOS envoie sur le port distant (mme port local) les
			//	   donnes relles. XDRI fait le WRITE  ce moment l.
			
			this->command[port].write_in_progress  = TRUE;
			this->command[port].write_has_occurred = FALSE;
			this->command[port].write_channel      = request->param.val_16[0];
			this->command[port].write_length       = ((Card32*)(request->param.val_16 + 1))[0];
			this->command[port].write_efs          = efs;
			
			i16[0] = 0;												//	succs garanti
			c16[1] = 'W';											//	port distant bidon
			break;
		
		case 0x0004:	//	?SPOS
			{
				Card32 poslow  = ((Card32*)(request->param.val_16 + 1))[0];
				Card8  poshigh = request->param.val_16[3] >> 8;
				Card16 mode    = request->param.val_16[3] & 0x00FF;
				
				c32 = (Card32*)(c16 + 1);
				
				efs->SetPos ( request->param.val_16[0],
							  poslow, poshigh,
							  mode, i16[0] );
				
				c32[0] = poslow;
				c16[3] = poshigh << 8;
				c16[4] = 0;
			}
			break;
		
		case 0x0003:	//	?GPOS
			{
				Card32 poslow  = 0;
				Card8  poshigh = 0;
				
				c32 = (Card32*)(c16 + 1);
				
				efs->GetPos ( request->param.val_16[0],
							  poslow, poshigh, i16[0] );
				
				c32[0] = poslow;
				c16[3] = poshigh << 8;
				c16[4] = 0;
			}
			break;
		
		case 0x0035:	//	?TRUNC
			efs->Trunc ( request->param.val_16[0], i16[0] );
			c16[1] = 0;
			break;
		
		case 0x0065:	//	?MAJBD
			i16[0] = 0;
			c16[1] = 0;
			break;
		
		case 0x0005:	//	?CLOSE
			efs->Close ( request->param.val_16[0],
						 ((Card32*)(request->param.val_16 + 1))[0],
						 ((Card32*)(request->param.val_16 + 1))[1],
						 i16[0] );
			c16[1] = 0;
			break;
		
		case 0x000C:	//	?ENTER
			efs->Enter ( request->param.name + 2,
						 request->param.val_16[0],
						 i16[0] );
			c16[1] = 0;
			break;
		
		case 0x000D:	//	?RELEASE
			efs->Release (request->param.name, i16[0]);
			c16[1] = 0;
			break;
		
		case 0x0057:	//	?GETHACH
			i16[0] = 0;
			c16[1] = 31;
			c16[2] = 0;
			break;
		
		case 0x004E:	//	?LISTOP
			c32 = (Card32*)(c16 + 1);
			efs->ListOpen ( request->param.name + 6,
						    request->param.val_32[0],
						    c32[0], c16[3], c32[2], i16[0] );
			c16[7] = 0;
			break;
		
		case 0x0050:	//	?LISTBUF
			c32    = (Card32*)(c16 + 1);
			c32[0] = request->param.val_32[1];
			efs->ListBuf ( request->param.val_32[0], c32[0],
						   segment[1].address, i16[0] );
			segment[1].length = c32[0];
			break;
		
		case 0x0051:	//	?LISTCLO
			efs->ListClose ( request->param.val_32[0], i16[0] );
			break;
		
		default:
			break;
	}
	
end_of_read:
	
	if ((Card16)(i16[0]) == (Card16)(0x85F1)) {
		
		//	Le FOS ne gre pas l'erreur ERWAIT de la mme manire quand on est
		//	en mode synchrone sans processus fils, alors on doit tricher en lui
		//	demandant de revenir  cause d'un mauvais paquet rseau...
		
		Ntr::Delms (5);
		
		error = 0x8984;				//	comme si l'on avait reu un mauvais paquet rseau
		
	} else {
		
		//	Pas de messages pour les botes aux lettres du FOS. C'est dans
		//	le segment final qu'ils sont normalement ajouts, mais on ne les
		//	supporte pas (encore).
		
		this->command[port].count = 0;
		
		efs->ReportBALMessage (segment[num_seg-1].address, segment[num_seg-1].length);
	}
	
    return error;
}


Int16
BIOSDriver::Close (Card16 port)
{
	port &= 0x007F;
	
	if (port < MAXPORT) {
		AutoLock lock = AutoLock (this->port_lock);
		pretend_to_use (lock);
		
		this->is_open[port] = FALSE;
		this->command[port].write_in_progress  = FALSE;
		this->command[port].write_has_occurred = FALSE;
	}
	
    return 0x0000;
}

Int16
BIOSDriver::Stop ()
{
    return 1;
}

Int16
BIOSDriver::Start ()
{
    return 1;
}

Int16
BIOSDriver::Abort (Card16 port)
{
    return 1;
}

Int16
BIOSDriver::Aux1 (const char* cmd, Card32& wr, const void* pwr, Card32& rd, void* prd)
{
    return 1;
}

Int16
BIOSDriver::Aux2 (const char* cmd, Card32& wr, const void* pwr, Card32& rd, void* prd)
{
    return 1;
}



static Bool
parse_hex (const char* arg, Card32& address)
{
	Card32 a = 0;
	int    i = 8;
	
	while (i--) {
		char c = *arg++;
		if (c < 'A') return FALSE;
		if (c > 'P') return FALSE;
		a  = (a << 4) | (c - 'A');
	}
	
	address = a;
	return TRUE;
}


/*
 *	Les commandes suivantes sont supportes (pour l'instant) :
 *
 *	- "MOUNT: xxxxxxxx:yyyyyyyy" o "xxxxxxxx" correspond au canal hexa du
 *	  BAR  utiliser pour l'instance de l'EFS  monter. "yyyyyyyy" correspond
 *    au pointeur  l'instance.
 *
 *	- "UNMOUNT: EFS", l'inverse de MOUNT.
 */

void
BIOSDriver::ParseCommand (const char* cmd, const char* arg, Card16 dest_id, Int16& err)
{
	Card32 bar = 0;
	Card32 ins = 0;
	
	err = 0x0010;
	
	AutoLock lock (this->efs_lock[dest_id]);
	pretend_to_use (lock);
	
	if ( (cmd[0] == 'M') && (cmd[1] == 'O') && (cmd[2] == 'U') && (cmd[3] == 'N') ) {
		if (this->efs[dest_id] == 0) {
			if ( (parse_hex (arg+0, bar))
			  && (arg[8] == ':')
			  && (parse_hex (arg+9, ins)) ) {
				EFS* efs = new EFS (bar, ins);
				this->efs[dest_id] = efs;
				err = (efs == 0) ? 0x8300 : 0x0000;
			}
		}
	}
	
	if ( (cmd[0] == 'U') && (cmd[1] == 'N') && (cmd[2] == 'M') && (cmd[3] == 'O')
	  && (this->efs[dest_id]) ) {
		delete this->efs[dest_id];
		this->efs[dest_id] = (EFS*)(0);
		err = 0x0000;
	}
	
	if ( (cmd[0] == 'R') && (cmd[1] == 'E') && (cmd[2] == 'Q') && (cmd[3] == 'U')
	  && (this->efs[dest_id]) ) {
		this->efs[dest_id]->Request (arg, err);
	}
}


