/*
 *	efs.cc
 *
 *	La classe EFS utilise par le serveur de fichiers gre la rception
 *	des paquets en provenance du pilote $XDRI et appelle les mthodes
 *	adquates.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#pragma implementation "efs.h"

#include "portable.h"
#include "efs.h"
#include "ntrel.h"


/*
 *	Convertit une valeur 32-bit en un nombre pseudo-hexa 'A'  'P'.
 *	C'est le format utilis par le pilote $XDRI pour la transmission
 *	messages au moyen du ?REMOTE.
 */

static void
build_hexa (char* buffer, Card32 hexa)
{
	int    i = 8;
	Card32 a = 0;
	
	while (i--) {
		a         = (hexa & 0xF0000000) >> 28;
		hexa      = hexa << 4;
		*buffer++ = (char)(a) + 'A';
	}
	
	buffer[0] = 0;
}


/*
 *	Processus fils excutant la requte FOS ?REMOTE pour que $XDRI
 *	soit mis au courant de l'existance de l'EFS.
 */

void
efs_mount (Card32 param)
{
	const char* dir = (const char*)(param);
	const char* prg = dir; while (*prg++);
	
	asm volatile ( "movel %0,a3\n\t"
				   "movel %1,a4\n\t"
				   ".long 0x4E45004D"
				 :
				 : "g" (dir), "g" (prg)
				 : "d7", "a3", "a4" );
}


/*
 *	Cre le BAR priv utilis pour communiquer avec le pilote
 *	$XDRI. L'initialisation relle se fait par une mthode
 *	spcifique, StartServer...
 */

EFS::EFS () : bar ("EFS-server")
{
}

/*
 *	La destruction du BAR se fait une fois que l'EFS a t
 *	dmont par XDRI, pas avant !
 */

EFS::~EFS ()
{
}



/*
 *	Mthode centrale qui fait tout le travail de dialogue avec
 *	le pilote $XDRI :
 *
 *	1. Connecte au pilote $XDRI et demande de monter l'EFS.
 *	2. Attend des requtes depuis le pilote $XDRI.
 *	3. Le pilote $XDRI excute un MOUNT, ce qui provoque un
 *	   appel  la mthode EFS::Mount.
 *	4. Attend encore, jusqu' ce que quelqu'un demande au
 *	   pilote $XDRI de dmonter l'EFS. Ceci gnre un appel
 *	    la mthode EFS::Unmount.
 *	5. Le serveur retourne la main  l'appelant qui dtruit
 *     l'instance.
 */
 
void
EFS::StartServer (const char* xdri)
{
	char        buffer[200];
	char*       line = buffer;
	const char* src;
	char        hexa_bar[12];
	char        hexa_ins[12];
	
	
	//	Signale au pilote $XDRI que notre EFS est prt  l'emploi.
	
	build_hexa (hexa_bar, (Card32)(this->bar.GetBar ()));
	build_hexa (hexa_ins, (Card32)(this));
	
	src = xdri;     while ((*line++ = *src++)); line[-1] = ':';
	src = "MOUNT:"; while ((*line++ = *src++)); line[-1] = 0;
	src = hexa_bar; while ((*line++ = *src++)); line[-1] = ':';
	src = hexa_ins; while ((*line++ = *src++)); line[-1] = 0;
	
	{
		Card32 pid  = 0;
		Card16 pino = 0;
		Ntr::CreTask (efs_mount, 1000, 8, "EFS-mount", (Card32)(buffer), 0, &pid, &pino);
	}
	
	//	Boucle principale d'excution de commandes.
	
	Bool first_time = TRUE;
	this->running   = TRUE;
	
	while (this->running) {
		
		EFSmsg* msg = 0;
		void*   ptr = 0;
		
		if (first_time) Ntr::SetTim (50);
		
		this->bar.Accept (ptr);
		
		if (first_time) {
			first_time = FALSE;
			Ntr::SetTim (0xFFFF);
		}
		
		if (this->bar.GetError ()) break;
		
		msg = (EFSmsg*)(ptr);
		msg->i16[0] = 0x0000;
		
		switch (msg->op) {
			
			case EFS_MOUNT:
				this->Mount ();
				break;
			
			case EFS_UNMOUNT:
				this->Unmount ();
				this->running = FALSE;
				break;
			
			case EFS_REQUEST:
				this->Request (msg->name[0], msg->i16[0]);
				break;
			
			case EFS_OPEN:
				this->Open (msg->name[0], msg->c32[0], msg->c32[1], msg->c16[0], msg->i16[0]);
				break;
			
			case EFS_CREATE:
				this->Create (msg->name[0], msg->c32[0], msg->i16[0]);
				break;
			
			case EFS_DELETE:
				this->Delete (msg->name[0], msg->c32[0], msg->i16[0]);
				break;
			
			case EFS_RENAME:
				this->Rename (msg->name[0], msg->name[1], msg->i16[0]);
				break;
			
			case EFS_MOVE:
				this->Move (msg->name[0], msg->name[1], msg->i16[0]);
				break;
			
			case EFS_CHATR:
				this->Chatr (msg->name[0], msg->c32[0], msg->i16[0]);
				break;
			
			case EFS_IFOPEN:
				this->IfOpen (msg->name[0], msg->i16[0]);
				break;
			
			case EFS_ARGS:
				this->Args (msg->name[0], msg->c16[0], msg->p[0], msg->c16[1], msg->i16[0]);
				break;
			
			case EFS_CLEAR:
				this->Clear (msg->name[0], msg->i16[0]);
				break;
			
			case EFS_DISK:
				this->Disk (msg->name[0], msg->c32[0], msg->c32[1], msg->i16[0]);
				break;
			
			case EFS_CDIR:
				this->CDir (msg->name[0], msg->i16[0]);
				break;
			
			case EFS_RDBYTE:
				this->RdByte (msg->c16[0], msg->c32[0], msg->p[0], msg->i16[0]);
				break;
			
			case EFS_WRBYTE:
				this->WrByte (msg->c16[0], msg->c32[0], msg->pc[0], msg->i16[0]);
				break;
			
			case EFS_SETPOS:
				this->SetPos (msg->c16[0], msg->c32[0], msg->c32[1], msg->c16[1], msg->i16[0]);
				break;
				
			case EFS_GETPOS:
				this->GetPos (msg->c16[0], msg->c32[0], msg->c32[1], msg->i16[0]);
				break;
			
			case EFS_TRUNC:
				this->Trunc (msg->c16[0], msg->i16[0]);
				break;
			
			case EFS_CLOSE:
				this->Close (msg->c16[0], msg->c32[0], msg->c32[1], msg->i16[0]);
				break;
			
			case EFS_ENTER:
				this->Enter (msg->name[0], msg->c16[0], msg->i16[0]);
				break;
			
			case EFS_RELEASE:
				this->Release (msg->name[0], msg->i16[0]);
				break;
			
			case EFS_LISTOPEN:
				this->ListOpen (msg->name[0], msg->name[1], msg->c32[0],
								msg->c32[1], msg->c32[2], msg->c32[3], msg->i16[0]);
				break;
			
			case EFS_LISTBUF:
				this->ListBuf (msg->c32[0], msg->c32[1], msg->p[0], msg->i16[0]);
				break;
			
			case EFS_LISTCLOSE:
				this->ListClose (msg->c32[0], msg->i16[0]);
				break;
			
			default:
				msg->i16[0] = 0x0001;
				break;
		}
		
		this->bar.EndAccept (0);
	}
}

void
EFS::Open (const char* name, Card32 mode, Card32 len, Card16& ch, Int16& err)
{
}


void
EFS::Create (const char* name, Card32 attr, Int16& err)
{
}


void
EFS::Delete (const char* name, Card32 mode, Int16& err)
{
}


void
EFS::Rename (const char* name_src, const char* name_dst, Int16& err)
{
}


void
EFS::Move (const char* name_src, const char* name_dst, Int16& err)
{
}


void
EFS::Chatr (const char* name, Card32 attr, Int16& err)
{
}


void
EFS::IfOpen (const char* name, Int16& err)
{
}


void
EFS::Args (const char* name, Card16 mode, Card8* args, Card16& count, Int16& err)
{
}


void
EFS::Clear (const char* name, Int16& err)
{
}


void
EFS::Disk (const char* name, Card32& free, Card32& used, Int16& err)
{
}


void
EFS::CDir (const char* name, Int16& err)
{
}


void
EFS::RdByte (Card16 ch, Card32& len, Card8* data, Int16& err)
{
}


void
EFS::WrByte (Card16 ch, Card32& len, const Card8* data, Int16& err)
{
}


void
EFS::SetPos (Card16 ch, Card32& poslow, Card32& poshigh, Card16 mode, Int16& err)
{
}


void
EFS::GetPos (Card16 ch, Card32& poslow, Card32& poshigh, Int16& err)
{
}


void
EFS::Trunc (Card16 ch, Int16& err)
{
}


void
EFS::Close (Card16 ch, Card32 h, Card32 d, Int16& err)
{
}


void
EFS::Enter (const char* name, Card16 mode, Int16& err)
{
}


void
EFS::Release (const char* name, Int16& err)
{
}


void
EFS::ListOpen (const char* dir, const char* crit, Card32 mode, Card32& ch, Card32& num, Card32& id, Int16& err)
{
}


void
EFS::ListBuf (Card32 ch, Card32& len, Card8* buf, Int16& err)
{
}


void
EFS::ListClose (Card32 ch, Int16& err)
{
}


void
EFS::Mount ()
{
}


void
EFS::Unmount ()
{
}


void
EFS::Request (const char* arg, Int16& err)
{
}

