/*
 *	fatbufwrite.cc
 *
 *	Do full cylinder buffered writes to slow devices.  Uses a simple
 *	buffered read/delayed write algorithm.
 *
 *	(C) Copyright 1995, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#include "fatdisk.h"

void
FATDisk::WriteDisk (Card32 start, const Card8* buf, Card32 len)
{
	Card32 i;
	Card32 length;
	
	const Card8* buf_ptr;
	Card8*       disk_ptr;
	
	Card32 where = 0;
	Card32 tail  = 0;
	
	//	Don't use cache ?
	
	if (this->disk_size == 1) {
		
		where = (start * MSECTOR_SIZE) + this->disk_offset;
		
		this->SeekDevice (where);
		this->WriteDevice (buf, len);
		
		return;
	}
	
	tail = start + (len / MSECTOR_SIZE) - 1;
	
	for (i = start; i <= tail; i++) {
		
		//	A cache miss...
		
		if ( (i < this->disk_current)
		  || (i >= this->disk_current + this->disk_size) ) {
			
			if (this->disk_dirty) this->FlushDisk ();
			
			this->disk_current = (i / this->disk_size) * this->disk_size;
			
			//	If there is something on the new cylinder that you want to
			//	keep, you'll have to read it first before writing.
			
			if (this->ReadDiskCyl (this->disk_current)) {
				
				where  = (this->disk_current * MSECTOR_SIZE) + this->disk_offset;
				length = this->disk_size * MSECTOR_SIZE;
				
				//	Move to next location
				
				this->SeekDevice (where);
				this->ReadDevice (this->disk_buf, length);
			}
		}
		
		//	A cache hit ...
		
		buf_ptr  = buf + ((i - start) * MSECTOR_SIZE);
		disk_ptr = this->disk_buf + ((i - this->disk_current) * MSECTOR_SIZE);
		
		memcpy (disk_ptr, buf_ptr, MSECTOR_SIZE);
		this->disk_dirty = TRUE;
	}
}


/*
 *	Flush a dirty buffer to disk. Resets disk_dirty.
 */

void
FATDisk::FlushDisk ()
{
	if ( (this->disk_current < 0)
	  || (this->disk_dirty == FALSE) )
		return;

	Card32 where = (this->disk_current * MSECTOR_SIZE) + this->disk_offset;
	Card32 len   = this->disk_size * MSECTOR_SIZE;
	
	this->SeekDevice (where);
	this->WriteDevice (this->disk_buf, len);
	
	this->disk_dirty = FALSE;
}


/*
 * Determine if the cylinder has some useful information on it.  Returns a 1
 * if it needs to be read.
 */

Bool
FATDisk::ReadDiskCyl (Card32 sector)
{
	Card32 i;
	Card32 start;
	Card32 end;
	
	if (sector == 0) return TRUE;
	
	start = ((sector - this->dir_start - this->dir_len) / this->clus_size) + 2;
	end   = ((sector + this->disk_size - this->dir_start - this->dir_len) / this->clus_size) + 2;
	
	//	Important information if DecodeFAT returns non-zero...
	
	for (i = start; i < end; i++) {
		if (this->DecodeFAT (i)) return TRUE;
	}
	
	return FALSE;
}


