/*
 *	fatefs.cc
 *
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */


#pragma implementation "fatefs.h"

#include <string.h>
#include <ctype.h>

#include "portable.h"
#include "efs.h"
#include "fatefs.h"
#include "fatdisk.h"
#include "ntrel.h"

typedef FATDateTime SmakyDate;

struct SmakyArgs
{
	Card8		type;				//	type de fichier (0)
	Card8		open;				//	compteur d'ouvertures
	Card32		attr;				//	attributs
	Card32		block_num;			//	nombre de blocs
	Card16		block_last;			//	taille dans le dernier bloc
	SmakyDate	d_create;
	SmakyDate	d_modif;
	SmakyDate	d_use;
	Card32		res_1;
	Card32		bd;
	Card32		length;
	Card16		code;
	Card8		res_2[20];
};

struct SmakyNameArgs
{
	char		name[16];			//	nom du fichier
	Card8		type;				//	type de fichier (0)
	Card8		open;				//	compteur d'ouvertures
	Card32		attr;				//	attributs
	Card32		block_num;			//	nombre de blocs
	Card16		block_last;			//	taille dans le dernier bloc
	SmakyDate	d_create;
	SmakyDate	d_modif;
	SmakyDate	d_use;
	Card32		res_1;
	Card32		bd;
	Card32		length;
	Card16		code;
	Card32		rank;
};

static Card8
make_bcd (Card8 bin)
{
	Card8 bcd = (bin / 10) * 16 + (bin % 10);
	return bcd;
}




FatEFS::FatEFS ()
{
	int i;
	
	for (i = 0; i < FATMAXCHAN; i++) {
		this->chan[i].used = FALSE;
	}
	
	this->enter_ok = FALSE;
	this->media    = 0;
	this->open     = 0;
}


FatEFS::~FatEFS ()
{
	int i;
	
	for (i = 0; i < FATMAXCHAN; i++) {
		if (this->chan[i].used) {
			//	...
		}
	}
	
	if (this->media) delete this->media;
}




void
FatEFS::Open (const char* name, Card32 mode, Card32 len, Card16& ch, Int16& err)
{
	if (this->JustEnter (err) == FALSE) return;
	
	if (mode & 0x814) {
		err = 0x8530;			//	"fichier protg"
		return;
	}
	
	ch = 0;
	
	for (ch = 0; ch < FATMAXCHAN; ch++) {
		if (this->chan[ch].used == FALSE) {
			break;
		}
	}
	
	if (ch == FATMAXCHAN) {
		err = 0x852C;			//	"mmoire du serveur pleine"
		return;
	}
	
	FATDirectory dir;
	
	err = this->media->FindFile (name, dir);
	if (err) return;
	
	this->chan[ch].used      = TRUE;
	this->chan[ch].max_pos   = dir.size[0] + dir.size[1]*256 + dir.size[2]*256*256 + dir.size[3]*256*256*256;
	this->chan[ch].cur_pos   = 0;
	this->chan[ch].cluster   = dir.start[0] + dir.start[1]*256;
	
	strncpy (this->chan[ch].name, name, sizeof (this->chan[ch].name) - 1);
	this->open++;
}


void
FatEFS::Create (const char* name, Card32 attr, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::Delete (const char* name, Card32 mode, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::Rename (const char* name_src, const char* name_dst, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::Move (const char* name_src, const char* name_dst, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::Chatr (const char* name, Card32 attr, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::IfOpen (const char* name, Int16& err)
{
	err = 0x0000;
}


void
FatEFS::Args (const char* name, Card16 mode, Card8* buf, Card16& count, Int16& err)
{
	if (this->JustEnter (err) == FALSE) return;
	
	FATDirectory dir;
	
	err = this->media->FindFile (name, dir);
	if (err) return;
	
	SmakyArgs*  args = (SmakyArgs*)(buf);
	FATDateTime time;
	
	this->media->ConvDate (dir.date[1], dir.date[0], time);
	this->media->ConvTime (dir.time[1], dir.time[0], time);
	
	count   = 0;
	
	args->d_create.day    = make_bcd (time.day);
	args->d_create.month  = make_bcd (time.month);
	args->d_create.year   = make_bcd (time.year);
	args->d_create.hour   = make_bcd (time.hour);
	args->d_create.minute = make_bcd (time.minute);
	args->d_create.second = make_bcd (time.second);
	
	args->d_modif    = args->d_create;
	args->d_use      = args->d_create;
	args->type       = 0;
	args->open       = 0;
	args->attr       = 0x000007FF;
	args->length     = dir.size[0]
					 + dir.size[1] * 256
					 + dir.size[2] * 256 * 256
					 + dir.size[3] * 256 * 256 * 256;
	
	args->block_num  = (args->length+255) / 256;
	args->block_last = args->length - args->block_num * 256;
	args->code       = 0;

	if ((dir.attr & 0x02) == 0) {
		args->attr |= 0x00002000;		//	list
	}
	
	if (dir.attr & 0x10) {
		args->attr |= 0x00001000;		//	dir
	}
}


void
FatEFS::Clear (const char* name, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::Disk (const char* name, Card32& free, Card32& used, Int16& err)
{
	if (this->JustEnter (err) == FALSE) return;
	
	free = 0;
	used = 1;
	err  = 0;
}


void
FatEFS::CDir (const char* name, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::RdByte (Card16 ch, Card32& len, Card8* data, Int16& err)
{
	if ( (ch >= FATMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	Card32 cur = this->chan[ch].cur_pos;
	Card32 pos = this->chan[ch].cur_pos + len;
	
	if (pos > this->chan[ch].max_pos) {
		len = this->chan[ch].max_pos - this->chan[ch].cur_pos;
		pos = this->chan[ch].max_pos;
		err = 0x8511;		//	"hors du fichier"
	}
	
	this->chan[ch].cur_pos = pos;
	
	this->media->ReadFile (this->chan[ch].cluster, this->chan[ch].max_pos, data, cur, len);
}


void
FatEFS::WrByte (Card16 ch, Card32& len, const Card8* data, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::SetPos (Card16 ch, Card32& poslow, Card32& poshigh, Card16 mode, Int16& err)
{
	if ( (ch >= FATMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	/*
	AfText ("SetPos mode=");
	AfX4 (mode);
	AfText (", ch=");
	AfX4 (ch);
	AfCR ();
	if (mode) {
		err = 0x8534;			//	"criture impossible"
		return;
	}
	*/
	
	if ( (poslow > this->chan[ch].max_pos)
	  || (poshigh) ) {
		poslow  = this->chan[ch].max_pos;
		poshigh = 0;
	}
	
	this->chan[ch].cur_pos = poslow;
}


void
FatEFS::GetPos (Card16 ch, Card32& poslow, Card32& poshigh, Int16& err)
{
	if ( (ch >= FATMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	poslow  = this->chan[ch].cur_pos;
	poshigh = 0;
}


void
FatEFS::Trunc (Card16 ch, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
FatEFS::Close (Card16 ch, Card32 h, Card32 d, Int16& err)
{
	if ( (ch >= FATMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	this->chan[ch].used      = FALSE;
	this->open--;
}



Bool
FatEFS::JustEnter (Int16& err)
{
	err = 0x0000;
	if (this->enter_ok) return TRUE;
	
	this->media = new FATDisk;
	if (this->media == 0) {
		err = 0x8341;				//	"plus assez de mmoire"
		return FALSE;
	}
	
	err = this->media->Init ('C');
	
	if (err) {
		delete this->media;
		this->media = 0;
		return FALSE;
	}
	
	this->enter_ok = TRUE;
	return TRUE;
}


void
FatEFS::Enter (const char* name, Card16 mode, Int16& err)
{
	err = 0x0000;
	if (this->JustEnter (err) == FALSE) return;
	
	err = this->media->SubDir (name);
}


void
FatEFS::Release (const char* name, Int16& err)
{
	if (this->open) {
		err = 0x8515;				//	"fichier utilis"
		return;
	}
	
	if (this->media) {
		delete this->media;
		this->media    = 0;
		this->enter_ok = FALSE;
	}
}


void
FatEFS::ListOpen (const char* dir, const char* crit, Card32 mode, Card32& ch,
				  Card32& num, Card32& id, Int16& err)
{
	static Card32 idx = 0;
	
	if ((mode & 0x00020008) == 0x00020008) {
		//	Ne supporte pas les arbres !
		err = 0x0001;
		return;
	}
	
	if (this->JustEnter (err) == FALSE) return;
	
	err = this->media->SubDir (dir);
	if (err) return;
	
	num = 0;
	id  = idx++;
	err = 0;
	ch  = 0;
	
	FatList* list = new FatList;
	
	if (list == 0) {
		err = 0x8341;		//	"plus assez de mmoire"
		return;
	}
	
	list->b_dossier = ((mode) & (1 << 23)) ? TRUE : FALSE;
	
	for (Card32 entry = 0; entry < this->media->dir_entries; entry++) {
		
		char        name[16];
		FATDateTime time = { 0 };
		
		name[0] = 0;
		
		this->media->ReadDir (entry);
		
		if (this->media->dir_temp.name[0] == 0x00) break;			//	empty
		if (this->media->dir_temp.name[0] == 0xE5) continue;		//	erased entry
		if (this->media->dir_temp.name[0] == 0x2E) continue;		//	current directory
		if (this->media->dir_temp.attr & 0x08) continue;			//	label
		if (this->media->dir_temp.attr & 0xC0) continue;			//	strange thing
		if (this->media->dir_temp.attr & 0x10) {					//	directory
			this->media->UnixName (name, this->media->dir_temp.name, this->media->dir_temp.ext);
		} else {
			this->media->UnixName (name, this->media->dir_temp.name, this->media->dir_temp.ext);
		}
		
		if (name[0]) {
			
			//	Trouv une description de fichier/dossier. Il faut maintenant gnrer
			//	une description adquate...
			
			FatElemOne* one = list->Insert ();
			
			if (one == 0) {
				err = 0x8341;
				num = 0;
				delete list;
				return;
			}
			
			strcpy (one->name, name);
			
			if ( (name[0] == '.') && ((name[1] == '.') || (name[1] == 0)) ) {
				one->flags = 0xFF;
				one->size  = 0;
				continue;
			}
			
			this->media->ConvDate (this->media->dir_temp.date[1], 
								   this->media->dir_temp.date[0], time);
			
			this->media->ConvTime (this->media->dir_temp.time[1], 
								   this->media->dir_temp.time[0], time);
			
			
			one->size    = this->media->dir_temp.size[0]
						 + this->media->dir_temp.size[1] * 256
						 + this->media->dir_temp.size[2] * 256 * 256
						 + this->media->dir_temp.size[3] * 256 * 256 * 256;
			
			one->date[0] = make_bcd (time.day);
			one->date[1] = make_bcd (time.month);
			one->date[2] = make_bcd (time.year);
			one->date[3] = make_bcd (time.hour);
			one->date[4] = make_bcd (time.minute);
			one->date[5] = make_bcd (time.second);
			one->flags   = this->media->dir_temp.attr;
			num++;
		}
	}
	
	ch  = (Card32)(list);
	this->open++;
}


void
FatEFS::ListBuf (Card32 ch, Card32& len, Card8* buf, Int16& err)
{
	SmakyNameArgs* args = (SmakyNameArgs*)(buf);
	Card32         num  = len / 64;
	Card32         size = 0;
	
	if (ch) {
		FatList* list = (FatList*)(ch);
		
		while (num--) {
			
			for (;;) {
				
				FatElemOne* one = list->GetNext ();
				if (one == 0) {
					len = size;
					return;
				}
				
				if (one->flags == 0xFF) continue;
				
				strcpy (args->name, one->name);
				
				char* name = args->name;
				
				while (*name && (name[0] != ';')) {
					name[0] = toupper (name[0]);
					name++;
				}
				
				name[0] = 0;
								
				args->d_create   = ((SmakyDate*)(one->date))[0];
				args->d_modif    = args->d_create;
				args->d_use      = args->d_create;
				args->type       = 0;
				args->open       = 0;
				args->attr       = 0x000007FF;
				args->length     = one->size;
				args->block_num  = (args->length+255) / 256;
				args->block_last = args->length - args->block_num * 256;
				args->code       = 0;
				args->rank       = num ? (Card32)(args + 1) : 0;
				
				if ((one->flags & 0x02) == 0) args->attr |= 0x00002000;
				if (one->flags & 0x10)        args->attr |= 0x00001000;
				
				if (one->flags & 0x10) {
					if (list->b_dossier) {
						name[0] = ':';
						name[1] = 0;
					} else {
						name[0] = '.';
						name[1] = 'D';
						name[2] = 'I';
						name[3] = 'R';
						name[4] = 0;
					}
				}
				break;
			}
			
			size += 64;
			args += 1;
		}
		
		err = 0x0000;
		return;
	}
	
	len = 0;
}


void
FatEFS::ListClose (Card32 ch, Int16& err)
{
	if (ch) {
		FatList* list = (FatList*)(ch);
		delete list;
		this->open--;
	}
}


void
FatEFS::Mount ()
{
}


void
FatEFS::Unmount ()
{
}


void
FatEFS::Request (const char* arg, Int16& err)
{
	err = 0x0001;
}




FatEFS::FatElemOne*
FatEFS::FatList::Insert ()
{
	FatElemTable** tpp  = & this->table;
	
	while ( (tpp[0]) && (tpp[0]->used == FATMAXELEM) ) {
		tpp = & (tpp[0]->next);
	}
	
	if (tpp[0] == 0) {
		tpp[0] = new FatElemTable;
		if (tpp[0] == 0) return 0;
	}
	
	this->num++;
	return & tpp[0]->elem[tpp[0]->used++];
}


FatEFS::FatElemOne*
FatEFS::FatList::GetNext ()
{
	FatElemTable** tpp   = & this->table;
	Card32         index = this->index++;
	
	while ( (tpp[0]) && (tpp[0]->used < index) ) {
		index -= FATMAXELEM;
		tpp    = & (tpp[0]->next);
	}
	
	if (tpp[0] == 0) {
		return 0;
	}
	
	return & tpp[0]->elem[index];
}

