/*
 *	fatfatread.cc
 *
 *	(C) Copyright 1995, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#include "fatdisk.h"



/*
 *	Get and decode a FAT (file allocation table) entry.  Returns the
 *	cluster number on success or 1 on failure.
 */

Card32
FATDisk::DecodeFAT (Card32 num)
{
	Card32 fat     = 0;
	Card32 fat_hi  = 0;
	Card32 fat_low = 0;
	Card32 byte_1  = 0;
	Card32 byte_2  = 0;
	Card32 start   = 0;

	if (this->fat_bits == 12) {
		
		/*
		 *	|    byte n     |   byte n+1    |   byte n+2    |
		 *	|7|6|5|4|3|2|1|0|7|6|5|4|3|2|1|0|7|6|5|4|3|2|1|0|
		 *	| | | | | | | | | | | | | | | | | | | | | | | | |
		 *	| n+0.0 | n+0.5 | n+1.0 | n+1.5 | n+2.0 | n+2.5 |
		 *	    \_____  \____   \______/________/_____   /
		 *	      ____\______\________/   _____/  ____\_/
		 *	     /     \      \          /       /     \
		 *	| n+1.5 | n+0.0 | n+0.5 | n+2.0 | n+2.5 | n+1.0 |
		 *	|      FAT entry k      |    FAT entry k+1      |
		 */
		
		
		//	Determine which bytes contain the entry :
		
		start = num * 3 / 2;
		
		if ( (start <= 2)
		  || ((start + 1) > (this->fat_len * MSECTOR_SIZE)) ) {
			return 1;
		}
		
		byte_1 = *(this->fat_buf + start);
		byte_2 = *(this->fat_buf + start + 1);
		
		if (num % 2) {
			fat_hi  = (byte_2 & 0xff) << 4;
			fat_low = (byte_1 & 0xf0) >> 4;
		} else {
			fat_hi  = (byte_2 & 0xf) << 8;
			fat_low = byte_1 & 0xff;
		}
		
		fat = (fat_hi + fat_low) & 0x0fff;
		
	} else {
		
		/*
		 *	|    byte n     |   byte n+1    |
		 *	|7|6|5|4|3|2|1|0|7|6|5|4|3|2|1|0|
		 *	| | | | | | | | | | | | | | | | |
		 *	|         FAT entry k           |
		 */
		
		
		//	Determine which bytes contain the entry :
		
		start = num * 2;
		if ( (start <= 3)
		  || ((start + 1) > (this->fat_len * MSECTOR_SIZE)) ) {
			return 1;
		}
		
		fat = ((*(this->fat_buf + start + 1)) << 8) + *(this->fat_buf + start);
	}
	
	return fat;
}



/*
 *	Read the entire FAT table into memory.  Crude error detection
 *	on wrong FAT encoding scheme.
 */

void
FATDisk::ReadFAT ()
{
	Card32 start = this->fat_start;
	Card32 fat_size;
	Card32 buflen;
	
	//	Let's see if the length of the FAT table is appropriate for
	//	the number of clusters and the encoding scheme.
	
	fat_size = (this->fat_bits == 12) ? (this->num_clus +2) * 3 / 2 : (this->num_clus +2) * 2;
	fat_size = (fat_size / 512) + ((fat_size % 512) ? 1 : 0);
	
	if (fat_size != this->fat_len) {
		error ("fat_read: Wrong FAT encoding?\n");
		exit (1);
	}
	
	//	Only the first copy of the FAT
	
	buflen  = this->fat_len * MSECTOR_SIZE;
	
	this->fat_buf = new Card8[buflen];
	
	if (this->fat_buf == 0) {
		error ("fat_read: no more memory\n");
		exit (1);
	}
	
	//	Read the FAT sectors
	
	for (int i = start; i < start+fat_len; i++) {
		this->ReadDisk (i, this->fat_buf + (i-start)*MSECTOR_SIZE, MSECTOR_SIZE);
	}
	
	//	The encoding scheme
	
	if (this->fat_bits == 12) {
		this->fat_end  = 0xfff;
		this->fat_last = 0xff8;
	} else {
		this->fat_end  = 0xffff;
		this->fat_last = 0xfff8;
	}
}


