/*
 *	fatinit.cc
 *
 *	(C) Copyright 1995, Pierre ARNAUD, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#include "fatdisk.h"

#include <string.h>

#define WORD(boot,x)	((Card32)((boot.x)[0] + ((boot.x)[1] << 8)))
#define DWORD(boot,x)	((Card32)((boot.x)[0] + ((boot.x)[1] << 8) + ((boot.x)[2] << 16) + ((boot.x)[3] << 24)))


/*
 *	Cleanup current buffers, if any.
 */

void
FATDisk::FreeBuffers ()
{
	if (this->fat_buf) delete this->fat_buf;
	if (this->disk_buf) delete this->disk_buf;
	if (this->dir_buf) delete this->dir_buf;
	
	this->fat_buf  = 0;
	this->disk_buf = 0;
	this->dir_buf  = 0;
}


/*
 *	Initialize data about the current disk. Determine its format and
 *	read the FAT.
 */

Int16
FATDisk::Init (char drive)
{
	Card32 tracks  = 0;
	Card32 heads   = 0;
	Card32 sectors = 0;
	Int16  err     = 0;
	
	const FATDevice* dev  = devices;
	
	if (this->channel) {
		this->CloseDevice ();
		this->FreeBuffers ();
	}
	
	//	First look for the first occurrence of the specified drive
	//	letter...
	
	while ((dev->drive) && (dev->drive != drive)) {
		dev++;
	}
	
	//	Now check any matching drive letter...
	
	while (dev->drive == drive) {
		
		this->disk_offset = dev->offset;
		
		err = this->OpenDevice (dev->name, dev->command);
		if (err) return err;
		
		err = this->ReadBootSector ();
		
		if (err == 0) {
			
			heads   = WORD (this->boot_sector, nheads);
			sectors = WORD (this->boot_sector, nsect);
			
			if (heads && sectors) {
				tracks = WORD (this->boot_sector, psect);
				tracks = tracks / (heads * sectors);
			}
			
			if ( (heads == 0)
			  || (heads > 100)
			  || (sectors == 0)
			  || (sectors > 500)
			  || (tracks > 5000)
			  || (this->boot_sector.clsiz == 0) ) {
				
				//	The sanity check didn't succeed. This might either be due to some
				//	very old DOS format (forget about it) or some non-DOS disk...
				
				err = 0x8519;			//	"type d'unit incorrect"
				
				this->CloseDevice ();
				return err;
			}
			
			if ( (dev->tracks == 0)
			  || ( (dev->tracks == tracks)
			    && (dev->heads == heads)
			    && (dev->sectors == sectors) ) ) {
				break;
			}
		}
		
		this->CloseDevice ();
		dev++;
	}
	
	if (dev->drive == 0) {
		err = 0x8519;					//	"type d'unit incorrect"
		return err;
	}
	
	//	All numbers are in sectors, except num_clus (which is in clusters)
	
	this->clus_size = this->boot_sector.clsiz;
	
	this->fat_start = WORD (this->boot_sector, nrsvsect);
	this->fat_len   = WORD (this->boot_sector, fatlen);
	this->fat_error = 0;
	this->fat_bits  = dev->fat_bits;
	
	this->dir_start = this->fat_start + this->boot_sector.nfat * this->fat_len;
	this->dir_len   = WORD (this->boot_sector, dirents) * MDIR_SIZE/MSECTOR_SIZE;
	this->dir_dirty = FALSE;
	
	this->num_fat	= this->boot_sector.nfat;
	this->num_clus  = (WORD (this->boot_sector, psect) == 0)
					? (DWORD (this->boot_sector, bigsect) - this->dir_start - this->dir_len) / this->clus_size
					: (WORD (this->boot_sector, psect) - this->dir_start - this->dir_len) / this->clus_size;
	
	//	Some more sanity checking...
	
	if ( ((this->clus_size * MSECTOR_SIZE) > MAX_CLUSTER)
	  || (WORD (this->boot_sector, secsiz) != MSECTOR_SIZE) ) {
		
		err = 0x8519;					//	"type d'unit incorrect"
		this->CloseDevice ();
		return err;
	}
	
	this->disk_size = (dev->tracks) ? (sectors * heads) : 1;
	this->disk_buf  = new Card8[this->disk_size * MSECTOR_SIZE];
	
	if (this->disk_buf == 0) {
		
		err = 0x8341;					//	"pas assez de mmoire"
		this->CloseDevice ();
		return err;
	}
	
	this->disk_current = -1000;
	this->disk_dirty   = FALSE;
	
	this->ReadFAT ();
	this->ResetChain ();
	
	for (Card32 entry = 0; entry < this->dir_entries; entry++) {
		
		this->ReadDir (entry);
		
		if (this->dir_temp.name[0] == 0x00) break;			//	empty
		if (this->dir_temp.name[0] == 0xE5) continue;		//	erased entry
		
		if (this->dir_temp.attr & 0x08) {
			strncpy (this->volname, this->dir_temp.name, 8);
			this->volname[8] = 0;
			int len = strlen (this->volname);
			strncat (this->volname, this->dir_temp.ext, 3);
			this->volname[len+3] = 0;
			break;
		}
	}
	
	return 0;
}


/*
 *	Read the boot sector.  We glean the disk parameters
 *	from this sector.
 */

Int16
FATDisk::ReadBootSector ()
{
	Int16 err = this->SeekDevice (this->disk_offset);
	return err ? err : this->ReadDevice (& this->boot_sector, MSECTOR_SIZE);
}


