/*
 * Copyright (c) 1992, 1993 William F. Jolitz, TeleMuse
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *	  must display the following acknowledgement:
 *		This software is a component of "386BSD" developed by 
 *		William F. Jolitz, TeleMuse.
 * 4. Neither the name of the developer nor the name "386BSD"
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS A COMPONENT OF 386BSD DEVELOPED BY WILLIAM F. JOLITZ 
 * AND IS INTENDED FOR RESEARCH AND EDUCATIONAL PURPOSES ONLY. THIS 
 * SOFTWARE SHOULD NOT BE CONSIDERED TO BE A COMMERCIAL PRODUCT. 
 * THE DEVELOPER URGES THAT USERS WHO REQUIRE A COMMERCIAL PRODUCT 
 * NOT MAKE USE OF THIS WORK.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.	IN NO EVENT SHALL THE DEVELOPER BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * cdromfs.c:
 * filesystem primatives to interpret ISO-9660, and it's predicessor
 * High Sierra, with optional Rock Ridge extensions. 
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "cdromfs.h"
#include "susp.h"
#include "rrip.h"

const char* cdrom_fmt_names[] = {
	"unknown format",
	"High Sierra",
	"ISO - 9660"
};



/*
 *	Check for the presence of a CDROM filesystem.
 *	If present, pass back parameters for initialization,
 *	otherwise, pass back error.
 *
 *	Effectively, this is a filesystem "mount" operation.
 */

int
is_cdrom_fs (DirEntry* dp, FS* fs)
{
	VolDesc*	vdp;
	DirEntry*	ddp;
	
	//	Logical sector size is at least this big
	
	fs->lss = CDROM_MINLSS;
	fs->lbs = 0;								//	don't know lbs yet
	vdp     = 0;

retry:
	//	Get a buffer of the appropriate logical sector size
	
	if (vdp) {
		delete vdp;
		vdp = 0;
	}
	
	if (fs->lss > CDROM_MAXLSS) return 0;
	
	vdp = (VolDesc*) new char[fs->lss];
	
	//	Locate at the beginning of the descriptor table
	
	cd_lseek (fs->fd, VD_LSN*fs->lss);
	
	//	Walk descriptor table
	
	for(;;) {
		
		unsigned char type;
		
		cd_read (fs->fd, vdp, fs->lss);			//	obtain a descriptor
		
		//	Determine ISO or HSF format of CDROM
		
		if (fs->type == 0) {
			
			if (strncmp (vdp->vol.iso_desc.id, ISO_STANDARD_ID,
						 sizeof (vdp->vol.iso_desc.id)) == 0) {
				fs->type = ISO;
			}
			
			if (strncmp (vdp->vol.hsf_desc.id, HSF_STANDARD_ID,
						 sizeof (vdp->vol.hsf_desc.id)) == 0) {
				fs->type = HSF;
			}
			
			if (strncmp (vdp->vol.hsf_desc.id, ISO_STANDARD_ID,
						 sizeof(vdp->vol.hsf_desc.id)) == 0) {
				fs->type = ISO;
			}
		}
		
		//	If determined, obtain root directory entry.
		
		if (fs->type != 0) {
			
			type = FPDV (vdp, type, fs->type);
			
			if (type == VD_PRIMARY) {
				if (fs->type == HSF) {
					bcopy (&vdp->vol.hsf_desc.root, (caddr_t)dp, sizeof (FSdir));
				} else {
					bcopy (&vdp->vol.iso_desc.root, (caddr_t)dp, sizeof (FSdir));
				}
				
				fs->lbs        = ISO_HWD (FPDVA (vdp, logicalblocksize, fs->type));
				fs->total_size = ISO_WD (FPDVA (vdp, total_volume, fs->type));
				break;
			}
			
		} else {
			
			//	Must be larger logical sector size, try again...
			
			fs->lss *= 2;
			goto retry;
		}
		
		//	Terminating volume
		
		if (type == VD_END) break;
	}
	
	delete vdp;
	vdp = 0;
	
	//	Did not obtain lbs and root
	
	if (fs->lbs == 0) return 0;
	
	fs->name = cdrom_fmt_names[fs->type];
	
	if ((ddp = open_cdf ("/.", fs))) {
		
		SuspCont cont;
		SuspSP*	 sp;
		SuspER*	 er;
		
		int skplen = 0;
		int extlen = EXTLEN (ddp, fs);
		
		//	Enough extensions to find a SUSP record ?
		
		if (extlen < sizeof (SuspSP)) return fs->type;
		
		//	No continuation to begin with...
		
		cont.curr     = 0;
		cont.cont_buf = 0;
		
		//	Attempt to probe for a SUSP identification record :
		//	N.B. SUSP actually should be in first option, however,
		//	CDROM-XA may have usurped that place. Live dangerously
		//	and scan full extension for it.
		
		for (fs->susp_begin = 0;
			 (int)(fs->susp_begin) < (int)(extlen - sizeof (SuspSP));
			 fs->susp_begin++) {
			
			if ((sp = (SuspSP*) handle_susp (ddp, &cont, "SP", fs, 0)) != 0) {
				
				//	"Where's the 0xBEEF ?"
				
				if ((sp->ck1 != SUSP_SPCK1) || (sp->ck2 != SUSP_SPCK2)) goto fail;
				
				//	We do SUSP
				
				fs->options |= USES_SUSP;
				skplen = sp->skplen;
				break;
			}
		}
		
fail:	if (cont.cont_buf) {
			delete cont.cont_buf;
			cont.cont_buf = 0;
		}
		
		if ((fs->options & USES_SUSP) == 0) {
			return fs->type;
		}
		
		cont.curr = 0;

		//	Attempt to find a SUSP Extension record, perhaps we have a Rock Ridge CDROM.
		
		while ((er = (SuspER*) handle_susp (ddp, &cont, "ER", fs, 0)) != 0) {
			
			//	Are we a Rock Ridge Extension ?
			
			if ( (er->len_id != RRIPIDLEN)
			  && (strncmp(er->ids, RRIPID, RRIPIDLEN) != 0) )
				continue;
			
			//	"I'm workin fer Mel Brooks"
			
			fs->options |= USES_RRIP;
			break;
		}
		
		fs->susp_begin = skplen;
		if (cont.cont_buf) {
			delete cont.cont_buf;
			cont.cont_buf = 0;
		}
	}
	
	return fs->type;
}




/*
 *	Obtain a "logical", i.e. relative to the directory entries beginning
 *	(or extent), block from the CDROM.
 *
 *	Effectively, this is a file "block" read operation on a filesystem's
 *	file.
 */

int
get_blk_dirent (DirEntry* dp, char* contents, long lbn, FS* fs)
{
	long	 datalen  = ISO_WD (FDVA (dp, datalen, fs->type));
	long	 location = ISO_WD (FDVA (dp, location, fs->type));
	unsigned iblksiz  = FDV (dp, int_blksz, fs->type);
	unsigned iblkskip = FDV (dp, int_blkskp, fs->type);
	
	//	Out of logical blocks perhaps ?
	
	if (lbntob (fs, lbn) > roundup (datalen, fs->lbs)) return 0;
	
	//	Perform logical block interleave.
	
	if (iblksiz && iblkskip) lbn = (lbn / iblksiz) * iblkskip + lbn;
	
	//	Perform logical to physical translation.
	
	cd_lseek (fs->fd, lbntob (fs, location + lbn));
	return (cd_read (fs->fd, contents, fs->lbs) == fs->lbs);
}


/*
 *	Search the contents of this directory entry, known to be a
 *	directory itself, looking for a component. If found, return
 *	the directory entry associated with the component.
 *
 *	This is the filesystem specific directory scanning mechanism for
 *	the CDROM filesystem.
 */

int
search_dirent (DirEntry* dp, DirEntry* fdp,
			   const char* compname, int comp_namelen,
			   FS* fs)
{
	DirEntry*	ldp     = 0;
	long		datalen = ISO_WD (FDVA (dp, datalen, fs->type));
	long		lbn     = 0;
	long		cnt     = 0;
	long		reclen  = 0;
	char*		buffer  = new char[fs->lbs];
	
	if (comp_namelen == 0) {
		compname     = ".";
		comp_namelen = 1;
	}
	
	while (get_blk_dirent (dp, buffer, lbn, fs)) {
		
		cnt      = datalen > fs->lbs ? fs->lbs : datalen;
		datalen -= cnt;
		ldp      = (DirEntry*)(buffer);
		
		//	Have we a record to match ?
		
		do {
		
			//	Match against component's name and name length.
			
			if (name_match (dp, fdp, ldp, compname, comp_namelen, fs)) {
				delete buffer;
				return 1;
			} else {
				reclen = FDV (ldp, reclen, fs->type);
				cnt   -= reclen;
				ldp    = (DirEntry*)(((char*)(ldp)) + reclen);
			}
		
		} while ((cnt > sizeof (FSdir)) && (reclen));
		
		if (datalen == 0) break;
		lbn++;
	}
	
	delete buffer;
	return 0;
}


/*
 *	Match against component's name and name length. If we find it, return
 *	the contents of the entire record, sans continuations, to caller of
 *	file lookup request that sent us charging off!
 * 
 *	This function evaluates a directory entry to determine if there is
 *	a match between an abstract name and the information present within the
 *	directory entry (name and extenstions). Knowledge of name extensions
 *	is interpeted here only.
 */

int
name_match (DirEntry* dp, DirEntry* fdp,
		    DirEntry* ldp, const char* compname,
		    int comp_namelen, FS* fs)
{
	long		  reclen = FDV (ldp, reclen, fs->type);
	long		  namlen = FDV (ldp, namlen, fs->type);
	char*         name   = FDV (ldp, name, fs->type);
	char*         copy   = name;
	unsigned char fbname = name[0];
	
	char buffer[32];
	strcpy (buffer, name);
	name = buffer;
	copy = buffer;
	
	//	First, check for match of current and parent directory names.
	//	Should not use POSIX ".", ".." names as abstract names.
	
	if ( (namlen == 1)
	  && (fbname <= 1) ) {
		if ( (comp_namelen == ((fbname == 0) ? 1 : 2))
		  && (strncmp((fbname == 0) ? "." : "..", compname, comp_namelen) == 0) )
			goto match;
	}
	
	//	Next, if we have RRIP...
	
	if (fs->options & USES_RRIP) {
		
		SuspCont cont;
		RripNM*  nm;
		
		cont.curr     = 0;
		cont.cont_buf = 0;
		
		//	Find any name records.
		
		while ((nm = (RripNM*) handle_susp (ldp, &cont, "NM", fs, dp == root)) != 0) {
			
			int lenname = nm->sz - sizeof (RripNM);
			
			if ( (comp_namelen > lenname)
			  || (strncmp (compname, nm->name, lenname) != 0) ) {
				if (cont.cont_buf) {
					delete cont.cont_buf;
					cont.cont_buf = 0;
				}
				return 0;
			}
			
			comp_namelen -= lenname;
			compname     += lenname;
		}
		
		//	Need to extract other RRIP too ?
		
		if (comp_namelen == 0) goto match;
		return 0;
	}
	
	//	Otherwise, use native ISO-9660/HSF
	
	copy[namlen] = 0;
	while (*copy && (copy[0] != ';')) copy++;
	copy[0] = 0;
	namlen  = (long)(copy - name);
	
	if ( (reclen >= comp_namelen + sizeof (FSdir) - 1)
	  && (namlen == comp_namelen)
	  && (strnicmp (name, compname, namlen) == 0)) {
		
match:	bcopy ((caddr_t)ldp, (caddr_t)fdp, reclen);
		return 1;
	}
	
	//	Seems like we fail
	
	return 0;
}


/*
 *	Finds a CDROM file for I/O. Returns a directory entry on open
 *	or 0, indicating file was not found.
 *
 *	This is a wrapper function to take the place of the file descriptor
 *	layer of an operating system.
 */

DirEntry*
open_cdf (const char* pathname, FS* fs)
{
	DirEntry* openfile = new DirEntry;
	
	if (pathname[0] == ':') pathname++;
	if (pathname[0] == 0) pathname = ".";
	
	if (lookup (root, openfile, pathname, fs) == 0) {
		delete openfile;
		openfile = 0;
	}
	
	return openfile;
}


/*
 *	Lookup the pathname by interpreting the directory structure of the
 *	CDROM element by element, returning a directory entry if found.
 *	Name translation occurs here, out of the null terminated path name
 *	string. This routine works by recursion.
 *
 *	This is the abstract file name translation facility, where file
 *	pathnames are decomposed into abstract names that are passed to
 *	filesystem specific entry match functions, who are responsible for
 *	sifting through directories and matching abstract records against
 *	file system data.
 */

int
lookup (DirEntry* dp, DirEntry* fdp, const char* pathname, FS* fs)
{
	const char*	nextcomp;
	unsigned	len;
	
	//	Break off the next component of the pathname
	
	if ((nextcomp = strchr (pathname, ':')) == NULL) {
		nextcomp = strchr (pathname, '\0');
	}
	
	len = nextcomp - pathname;
	
	//	Attempt a match, returning component if found
	
	if (search_dirent (dp, fdp, pathname, len, fs)) {
		if (*nextcomp++ == '\0') {
			
			//	If no more components, return found value...
			
			return 1;
			
		} else if (FDV(dp, flags, fs->type) & CD_DIRECTORY) {
			
			//	...otherwise, if this component is a directory,
			//	recursively satisfy lookup; should use abstract
			//	directory type match instead.
			
			return (lookup (fdp, fdp, nextcomp, fs));
		}
	}
	
	return 0;
}



/*
 *	Are we a valid extension?
 */

GenericExtension*
valid_ext (unsigned char* cp, unsigned len)
{
	GenericExtension *gep = (GenericExtension*)(cp);
	int i;
	
	//	Does size at least cover extension format and fit within feild length ?
	
	if ( (gep->sz < sizeof (GenericExtension))
	  || (gep->sz > len) ) {
		return 0;
	}
	
	//	Does each character of the signature correspond to a printable character ?
	
	for (i = 0; i < sizeof (gep->sig); i++) {
		if ( (gep->sig[i] < ' ')
		  || (gep->sig[i] > '~') ) {
			return 0;
		}
	}
	
	if ( (gep->vers < 1)
	  || (gep->vers > 10) ) {
		return 0;
	}
	
	return gep;
}


