
/*
 * Copyright (c) 1992, 1993 William F. Jolitz, TeleMuse
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This software is a component of "386BSD" developed by 
 *	William F. Jolitz, TeleMuse.
 * 4. Neither the name of the developer nor the name "386BSD"
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS A COMPONENT OF 386BSD DEVELOPED BY WILLIAM F. JOLITZ 
 * AND IS INTENDED FOR RESEARCH AND EDUCATIONAL PURPOSES ONLY. THIS 
 * SOFTWARE SHOULD NOT BE CONSIDERED TO BE A COMMERCIAL PRODUCT. 
 * THE DEVELOPER URGES THAT USERS WHO REQUIRE A COMMERCIAL PRODUCT 
 * NOT MAKE USE OF THIS WORK.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE DEVELOPER BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * CDROM Filesystem datastructures and definitions. -wfj
 */

/* Volume Definitions */
#define	VD_LSN		16	/* first logical sector of volume descriptor
				   table */
#define	CDROM_MINLSS	2048	/* minimum and maximum logical sector size */
#define	CDROM_MAXLSS	65536

#define HSF 1
#define ISO 2

/*
 * Rude translation routines for interpreting words, halfwords
 */
#ifndef BIGENDIAN
#define	ISO_WD(s)	(*(unsigned *)(s))
#define	ISO_HWD(s)	(*(unsigned short *)(s))
#else
#define	ISO_WD(s)	(*((unsigned *)(s)+1))
#define	ISO_HWD(s)	(*((unsigned short *)(s)+1))
/*
Or, if you really must (68K, RISC):

#define ISO_WD(s) { \
	unsigned x; \
	bcopy((char *)(s)+sizeof(x), &x, sizeof(x)); \
	x; \
}
#define ISO_HWD(s) { \
	unsigned short x; \
	bcopy((char *)(s)+sizeof(x), &x, sizeof(x)); \
	x; \
}*/
#endif

/* Filesystem types */

typedef unsigned char iso711_t;         /* single octet */
typedef char dchars;         		/* directory character [0-9A-Z_] */
typedef char achars;         		/* application character */
					/* two octets, both byte orders */
typedef struct _i723 { unsigned char i[4]; } iso723_t;
					/* four octets, both byte orders */
typedef struct _i733 { unsigned char i[8]; } iso733_t;

/* ISO 7 byte times -- binary */

typedef struct itime7 {
	unsigned char years;	/* number of years since 1900 */
	unsigned char month;	/* month of the year */
	unsigned char day;	/* day of month */
	unsigned char hour;	/* hour of day */
	unsigned char min;	/* minute of the hour */
	unsigned char sec;	/* second of the minute */
	char tz;		/* timezones, in quarter hour increments
				   (ISO only!) */
				/* or, longitude in units of 3.75 of a degree!*/
				/* negative is west of greenwich */
} iso7time_t;

/* ISO 17 byte time -- ascii */
#define ISO_17_TIME(n)	\
	unsigned char n##_years[4];	/* number of years */	\
	unsigned char n##_month[2];	/* month of the year */	\
	unsigned char n##_day[2];	/* day of month */	\
	unsigned char n##_hour[2];	/* hour of day */	\
	unsigned char n##_min[2];	/* minute of the hour */	\
	unsigned char n##_sec[2];	/* second of the minute */	\
	unsigned char n##_hundreths[2];	/* hundreths of the second */	\
	char n##_tz		/* timezones, in binary, like 7 byte times */

/* HSF 6 byte times -- binary */
typedef struct htime6 {
	unsigned char years;	/* number of years since 1900 */
	unsigned char month;	/* month of the year */
	unsigned char day;	/* day of month */
	unsigned char hour;	/* hour of day */
	unsigned char min;	/* minute of the hour */
	unsigned char sec;	/* second of the minute */
} hsf6time_t;

/* HSF 16 byte time -- ascii */
typedef struct htime16 {
	unsigned char years[4];	/* number of years */
	unsigned char month[2];	/* month of the year */
	unsigned char day[2];	/* day of month */
	unsigned char hour[2];	/* hour of day */
	unsigned char min[2];	/* minute of the hour */
	unsigned char sec[2];	/* second of the minute */
	unsigned char hundreths[2];
				/* hundreths of the second */
} hsf16time_t;

/*
 * CDROM file system directory entries
 */

/* file flags: */
#define	CD_VISABLE	0x01	/* file name is hidden or visable to user */
#define	CD_DIRECTORY	0x02	/* file is a directory and contains entries */
#define	CD_ASSOCIATED	0x04	/* file is opaque to filesystem,
					visable to system implementation  */
#define	CD_EAHSFRECORD	0x04	/* file has HSF extended attribute record fmt */
#define	CD_PROTECTION	0x04	/* used extended attibutes for protection */
#define	CD_ANOTHEREXTNT	0x80	/* file has at least one more extent */

typedef struct iso_dirent {
	iso711_t	reclen;		/* length of directory record */
	iso711_t	earlen;		/* extended attribute record length */
	iso733_t	location;	/* location of extent */
	iso733_t	datalen;	/* data length */
	
	unsigned char years;	/* number of years since 1900 */
	unsigned char month;	/* month of the year */
	unsigned char day;	/* day of month */
	unsigned char hour;	/* hour of day */
	unsigned char min;	/* minute of the hour */
	unsigned char sec;	/* second of the minute */
	char tz;		/* timezones, in quarter hour increments
				   (ISO only!) */
				/* or, longitude in units of 3.75 of a degree!*/
				/* negative is west of greenwich */
	iso711_t	flags;
	iso711_t	int_blksz;	/* interleave size, blocks */
	iso711_t	int_blkskp;	/* interleave skip, blocks */
	iso723_t	vssn;		/* volume set sequence number */
	iso711_t	namlen;		/* length of file identifier */
	dchars		name[0];	/* file identifier */
} isodirent_t;

typedef struct hsf_dirent {
	iso711_t	reclen;		/* length of directory record */
	iso711_t	earlen;		/* extended attribute record length */
	iso733_t	location;	/* location of extent */
	iso733_t	datalen;	/* data length */
	hsf6time_t	rec_date;	/* recording date and time */
	iso711_t	flags;		/* file flags */
	char		reserved;
	iso711_t	int_blksz;	/* interleave size, blocks */
	iso711_t	int_blkskp;	/* interleave skip, blocks */
	iso723_t	vssn;		/* volume set sequence number */
	iso711_t	namlen;		/* length of file identifier */
	dchars		name[0];	/* file identifier */
} hsfdirent_t;

#define	NVOLTYPENAMES	(sizeof(voltypenames)/sizeof(char *))

/* volume descriptor types -- type field of each descriptor */
#define VD_PRIMARY 1
#define VD_END 255

/* ISO 9660 primary volume descriptor */

#define ISO_STANDARD_ID "CD001"

typedef struct iso_primary {
	char type;			/* kind of volume descriptor - 1 */
	char id[5];			/* "CD001" */
	char version;			/* version of this data structure - 1 */
	char reserved1;
	achars system[32];		/* kind of bootstrap area present */
	dchars volume_name[32];		/* name of this volume */
	char reserved2[8];
	iso733_t total_volume;		/* aggregate size of the volume */
	char reserved3[32];
	iso723_t nvolumes;		/* how many volumes in this set */
	iso723_t volume_number;		/* which of the volumes this is */
	iso723_t logicalblocksize;	/* logical block size of CDROM */
	iso733_t pathtablesize;		/* size of path tables */
	long pathtable_lsb[2];		/* path table LITTLEENDIAN */
	long pathtable_msb[2];		/* path table BIGENDIAN */
	isodirent_t root;		/* root directory entry */
	dchars rootname;		/* 0 */
	dchars volumeset[128];		/* group name of the volumes */
	achars publisher[128];		/* who mastered this CDROM */
	achars preparer[128];		/* who prepared data for this CDROM */
	achars application[128];	/* application(s) used for this CDROM */
	dchars copyrightfile[37];	/* contents copyright filename */
	dchars abstractfile[37];	/* contents description filename */
	dchars bibliographicfile[37];	/* contents bibliography filename */
	ISO_17_TIME (creation);		/* volume creation date */
	ISO_17_TIME (modification);	/* volume modification date */
	ISO_17_TIME (expiration);	/* volume expiration date */
	ISO_17_TIME (effective);	/* volume effective date */
	iso711_t direnttype;		/* directory entry type */
	char reserved4;
	char reserved_application[512];	/* reserved for application */
} isodesc_t;

/* High Sierra format standard volume descriptor */

#define HSF_STANDARD_ID "CDROM"

struct hsf_standard {
	iso733_t volume_start;		/* first logical block of the volume */
	char type;			/* kind of volume descriptor - 1 */
	char id[5];			/* "CDROM" */
	char version;			/* version of this data structure - 1 */
	char reserved1;
	achars system[32];		/* kind of bootstrap area present */
	dchars volume_name[32];		/* name of this volume */
	char reserved2[8];
	iso733_t total_volume;		/* aggregate size of the volume */
	char reserved3[32];
	iso723_t nvolumes;		/* how many volumes in this set */
	iso723_t volume_number;		/* which of the volumes this is */
	iso723_t logicalblocksize;	/* logical block size of CDROM */
	iso733_t pathtablesize;		/* size of path tables */
	long pathtable_lsb[4];		/* path table LITTLEENDIAN */
	long pathtable_msb[4];		/* path table BIGENDIAN */
	hsfdirent_t root;		/* root directory entry */
	dchars rootname;		/* 0 */
	dchars volumeset[128];		/* group name of the volumes */
	achars publisher[128];		/* who mastered this CDROM */
	achars preparer[128];		/* who prepared data for this CDROM */
	achars application[128];	/* application(s) used for this CDROM */
	dchars copyrightfile[32];	/* contents copyright filename */
	dchars abstractfile[32];	/* contents description filename */
	hsf16time_t creation;		/* volume creation date */
	hsf16time_t modification;	/* volume modification date */
	hsf16time_t expiration;		/* volume expiration date */
	hsf16time_t effective;		/* volume effective date */
	iso711_t direnttype;		/* directory entry type */
	char reserved4;
	char reserved_application[512];	/* reserved for application */
} hsfdesc_t;



/* ISO 7 byte (HSF 6 byte) times -- binary */
struct CdromTime {
	unsigned char years;	/* number of years since 1900 */
	unsigned char month;	/* month of the year */
	unsigned char day;	/* day of month */
	unsigned char hour;	/* hour of day */
	unsigned char min;	/* minute of the hour */
	unsigned char sec;	/* second of the minute */
	char tz;		/* timezones, in quarter hour increments
				   (ISO only!) */
				/* or, longitude in units of 3.75 of a degree!*/
				/* negative is west of greenwich */
};


#define	CD_FLAGBITS	"vdaEp  m"	/* file flag bits */

/*
 * Handy macro's for converting between absolute block numbers and file
 * releative block numbers.
 */
#define	lbntob(fs, n)	((fs)->lbs * (n))
#define	btolbn(fs, n)	((fs)->lbs * (n))
#define	trunc_lbn(fs, n)	((n) - ((n) % (fs)->lbs)
#undef  roundup
#define	roundup(n, d)	((((n) + (d)) / (d)) * (d))

/* filesystem volume descriptors */
struct VolDesc {
	union cdvol {
		struct iso_primary  iso_desc;
		struct hsf_standard  hsf_desc;
	} vol;
	char pad[2028 - sizeof(union cdvol)];
};
	
/*
 *	Per filesystem information
 */

struct FS {
	const char*	name;			//	kind of cdrom filesystem
	int			fd;				//	open file descriptor
	int 		lss;			//	logical sector size
	int 		lbs;			//	logical block size
	int 		type;			//	which flavor
	int			total_size;		//	total size on the CD
	int 		options;		//	which extensions
								//	one of :
#define	USES_SUSP  0x00000001	//	- uses System Use Sharing Protocol
#define	USES_RRIP  0x00000002	//	- uses Rock Ridge Interchange Protocol
#define	USES_TRANS 0x00000004	//	- uses 386BSD 0.1 name translation
#define	USES_NO_TZ 0x00000008	//	- forces timezones off on timestamps
								//	SUSP specific :
	int			 susp_begin;	//	start of SUSP in System Use Area
								//	Rock Ridge specific
};

/*
 * CDROM filesystem information per file node (native format)
 */

union FSdir {
	isodirent_t iso_dir;
	hsfdirent_t hsf_dir;
};


/*
 *	Filesystem directory entry - maxed out.
 */

struct DirEntry {
	
	FSdir fsd;					//	directory entry
	
	//	Actually, name contains name, reserved field, and extensions area :
	
	char name[255 - sizeof (FSdir)];
};

extern struct DirEntry rootent, currentent;

/* generic extension */

struct GenericExtension
{
	char sig[2];		/* signiture */
	unsigned char sz;	/* size of entry (including possible pad) */
	char vers;		/* version of entry */
};

/* "fetch directory value" */
#define	FDV(b, f, t)	(((t) == ISO) ? (b)->fsd.iso_dir.##f \
				: (b)->fsd.hsf_dir.##f)
#define	FDVA(b, f, t)	(((t) == ISO) ? &((b)->fsd.iso_dir.##f) \
				: &((b)->fsd.hsf_dir.##f))

/* "fetch primary descriptor value" */
#define	FPDV(b, f, t)	(((t) == ISO) ? (b)->vol.iso_desc.##f \
				: (b)->vol.hsf_desc.##f)
#define	FPDVA(b, f, t)	(((t) == ISO) ? &((b)->vol.iso_desc.##f) \
				: &((b)->vol.hsf_desc.##f))

/*
 * Obtain length of System Use Field,
 * compensating for reserved field used to word align
 * the directory entry -- such a feature!
 */
#define EXTLEN(dp, fs) \
	FDV((dp), reclen, (fs)->type) - \
		((sizeof (FSdir)-1 + \
			FDV((dp), namlen, (fs)->type) +1) & ~1)

extern int search_dirent (DirEntry*, DirEntry*, char *, int, FS*);
extern void extractdirent (DirEntry*, FS*);
extern int lookup (DirEntry*, DirEntry*, const char *, FS*);
extern DirEntry* opencdf (char *, FS*);
extern GenericExtension* valid_ext (unsigned char *, unsigned);
extern char* iso_astring (char *, int);
extern char* cdrom_time (CdromTime*, int);
extern int get_blk_dirent (DirEntry* dp, char* contents, long lbn, FS* fs);

extern int cd_open (const char* name);
extern void cd_eject (int ch);
extern void cd_close (int ch);
extern void cd_lseek (int ch, long pos);
extern int cd_read (int ch, void* buf, long siz);
extern DirEntry* open_cdf (const char*, FS*);
extern int name_match (DirEntry*, DirEntry*, DirEntry*, const char*, int, FS*);
extern int is_cdrom_fs (DirEntry*, FS* fs);

extern DirEntry*   root;
extern FS*         cdrom;
extern const char* cdrom_name;

