/*
 *	isoefs.cc
 *
 *	La classe "Iso9660" correspond  l'implmentation d'un EFS permettant la
 *	lecture de volumes CD-ROM ISO ou High Sierra, avec extension Rockridge
 *	ventuelle.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */


#pragma implementation "isoefs.h"

#include <string.h>
#include <ctype.h>
#include "portable.h"
#include "efs.h"
#include "isoefs.h"
#include "cdromfs.h"
#include "ntrel.h"
#include "mon.h"

struct SmakyDate
{
	Card8	day,  month,  year;
	Card8	hour, minute, second;
};

struct SmakyArgs
{
	Card8		type;				//	type de fichier (0)
	Card8		open;				//	compteur d'ouvertures
	Card32		attr;				//	attributs
	Card32		block_num;			//	nombre de blocs
	Card16		block_last;			//	taille dans le dernier bloc
	SmakyDate	d_create;
	SmakyDate	d_modif;
	SmakyDate	d_use;
	Card32		res_1;
	Card32		bd;
	Card32		length;
	Card16		code;
	Card8		res_2[20];
};

struct SmakyNameArgs
{
	char		name[16];			//	nom du fichier
	Card8		type;				//	type de fichier (0)
	Card8		open;				//	compteur d'ouvertures
	Card32		attr;				//	attributs
	Card32		block_num;			//	nombre de blocs
	Card16		block_last;			//	taille dans le dernier bloc
	SmakyDate	d_create;
	SmakyDate	d_modif;
	SmakyDate	d_use;
	Card32		res_1;
	Card32		bd;
	Card32		length;
	Card16		code;
	Card32		rank;
};

static Card8
make_bcd (Card8 bin)
{
	Card8 bcd = (bin / 10) * 16 + (bin % 10);
	return bcd;
}


Iso9660::Iso9660 ()
{
	int i;
	
	for (i = 0; i < ISOMAXCHAN; i++) {
		this->chan[i].used = FALSE;
	}
	
	this->open     = 0;
	this->enter_ok = FALSE;
}


Iso9660::~Iso9660 ()
{
	int i;
	
	for (i = 0; i < ISOMAXCHAN; i++) {
		if (this->chan[i].used) {
			delete this->chan[i].dir_entry;
		}
	}
}


void
Iso9660::Open (const char* name, Card32 mode, Card32 len, Card16& ch, Int16& err)
{
	if (this->JustEnter (err) == FALSE) return;
	
	if (mode & 0x814) {
		err = 0x8530;			//	"fichier protg"
		return;
	}
	
	ch = 0;
	
	for (ch = 0; ch < ISOMAXCHAN; ch++) {
		if (this->chan[ch].used == FALSE) {
			break;
		}
	}
	
	if (ch == ISOMAXCHAN) {
		err = 0x852C;			//	"mmoire du serveur pleine"
		return;
	}
	
	DirEntry*  dp   = open_cdf (name, cdrom);
	
	if (dp == 0) {
		err = 0x8512;			//	"fichier n'existe pas"
		return;
	}
	
	this->chan[ch].used      = TRUE;
	this->chan[ch].dir_entry = dp;
	this->chan[ch].max_pos   = ISO_WD (FDVA (dp, datalen, cdrom->type));
	this->chan[ch].cur_pos   = 0;
	
	strncpy (this->chan[ch].name, name, sizeof (this->chan[ch].name) - 1);
	this->open++;
}


void
Iso9660::Create (const char* name, Card32 attr, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::Delete (const char* name, Card32 mode, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::Rename (const char* name_src, const char* name_dst, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::Move (const char* name_src, const char* name_dst, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::Chatr (const char* name, Card32 attr, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::IfOpen (const char* name, Int16& err)
{
	err = 0x0000;
}


void
Iso9660::Args (const char* name, Card16 mode, Card8* buf, Card16& count, Int16& err)
{
	if (this->JustEnter (err) == FALSE) return;
	
	DirEntry*  dp   = open_cdf (name, cdrom);
	SmakyArgs* args = (SmakyArgs*)(buf);
	
	if (dp == 0) {
		err = 0x8512;		//	"fichier n'existe pas"
		return;
	}
	
	CdromTime* time  = (CdromTime*)(& dp->fsd.iso_dir.years);
	Card8      flags = (FDV (dp, flags, cdrom->type)) & (CD_VISABLE | CD_DIRECTORY);
	
	count = 0;
	
	args->d_create.day    = make_bcd (time->day);
	args->d_create.month  = make_bcd (time->month);
	args->d_create.year   = make_bcd (time->years);
	args->d_create.hour   = make_bcd (time->hour);
	args->d_create.minute = make_bcd (time->min);
	args->d_create.second = make_bcd (time->sec);
	
	args->d_modif    = args->d_create;
	args->d_use      = args->d_create;
	args->type       = 0;
	args->open       = 0;
	args->attr       = 0x000007FF;
	args->length     = ISO_WD (FDVA (dp, datalen, cdrom->type));
	args->block_num  = (args->length+255) / 256;
	args->block_last = args->length - args->block_num * 256;
	args->code       = 0;
	
	if ((flags & CD_VISABLE) == 0) {
		args->attr |= 0x00002000;
	}
	
	if (flags & CD_DIRECTORY) {
		args->attr |= 0x00001000;
	}
	
	delete dp;
}


void
Iso9660::Clear (const char* name, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::Disk (const char* name, Card32& free, Card32& used, Int16& err)
{
	if (this->JustEnter (err) == FALSE) return;
	
	free = 0;
	used = cdrom->total_size * (cdrom->lbs/256);
	err  = 0;
}


void
Iso9660::CDir (const char* name, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::RdByte (Card16 ch, Card32& len, Card8* data, Int16& err)
{
	if ( (ch >= ISOMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	Card32 pos = this->chan[ch].cur_pos + len;
	
	if (pos > this->chan[ch].max_pos) {
		len = this->chan[ch].max_pos - this->chan[ch].cur_pos;
		pos = this->chan[ch].max_pos;
		err = 0x8511;		//	"hors du fichier"
	}
	
	if (len) {
		
		Card32 datalen = this->chan[ch].max_pos;
		Card32 realpos = this->chan[ch].cur_pos;
		Card32 block   = 0;
		Card32 offset  = 0;
		Card32 bl_size = cdrom->lbs;
		Card8  buffer[8192];

		block   = realpos / bl_size;
		offset  = realpos - block * bl_size;
		datalen = len;
		
		for (;;) {
			Card32 mylen = datalen + offset;
			Card32 count = (mylen > bl_size) ? bl_size : mylen;
			Card8* buf   = buffer;
			
			if (get_blk_dirent (this->chan[ch].dir_entry, buffer, block, cdrom) == 0) {
				break;
			}
			
			block   += 1;
			count   -= offset;
			buf     += offset;
			offset   = 0;
			datalen -= count;
			while (count--) *data++ = *buf++;
			if (datalen == 0) break;
		}
		
		this->chan[ch].cur_pos += len;
	}
}


void
Iso9660::WrByte (Card16 ch, Card32& len, const Card8* data, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::SetPos (Card16 ch, Card32& poslow, Card32& poshigh, Card16 mode, Int16& err)
{
	mode = 0x00;			//	******FAUX******IGNORE LES FLAGS******
	
	if ( (ch >= ISOMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	if ( (mode)
	  && ( (poshigh)
	    || (poslow > this->chan[ch].max_pos) ) ) {
		err = 0x8534;			//	"criture impossible"
		return;
	}
	
	if ( (poslow > this->chan[ch].max_pos)
	  || (poshigh) ) {
		poslow  = this->chan[ch].max_pos;
		poshigh = 0;
	}
	
	this->chan[ch].cur_pos = poslow;
}


void
Iso9660::GetPos (Card16 ch, Card32& poslow, Card32& poshigh, Int16& err)
{
	if ( (ch >= ISOMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	poslow  = this->chan[ch].cur_pos;
	poshigh = 0;
}


void
Iso9660::Trunc (Card16 ch, Int16& err)
{
	err = 0x8534;			//	"criture impossible"
}


void
Iso9660::Close (Card16 ch, Card32 h, Card32 d, Int16& err)
{
	if ( (ch >= ISOMAXCHAN)
	  || (this->chan[ch].used == FALSE) ) {
		err = 0x8514;		//	"canal incorrect"
		return;
	}
	
	delete this->chan[ch].dir_entry;
	
	this->chan[ch].used      = FALSE;
	this->chan[ch].dir_entry = 0;
	this->open--;
}


Bool
Iso9660::JustEnter (Int16& err)
{
	if (cdrom->fd == 0) {
		
		//	Ouvre le lecteur de CDROM. Ceci verrouille l'jection du
		//	disque.
		
		cdrom->fd = cd_open (cdrom_name);
		
		if (cdrom->fd & 0xFFFF0000) {
			err = (Int16)(cdrom->fd);	//	erreur d'ouverture
			cdrom->fd = 0;
			return FALSE;
		}
		
		//	On a russi  ouvrir le lecteur CDROM, c'est assez bon signe.
		//	Essayons encore d'y lire quelque chose !
		
		char* buffer = new char[4096];
		
		if (cd_read (cdrom->fd, buffer, 4096) == 0) {
			cd_close (cdrom->fd);
			cdrom->fd = 0;
			err       = 0x8813;			//	"unit non prte"
			return FALSE;
		}
		
		delete buffer;
	}

	if (is_cdrom_fs (root, cdrom) == 0) {
		err = 0x8551;					//	"unit incorrecte"
		return FALSE;
	}
	
	this->enter_ok = TRUE;
	return TRUE;
}

void
Iso9660::Enter (const char* name, Card16 mode, Int16& err)
{
	this->JustEnter (err);
}


void
Iso9660::Release (const char* name, Int16& err)
{
}




void
Iso9660::ListOpen (const char* dir, const char* crit, Card32 mode, Card32& ch, Card32& num, Card32& id, Int16& err)
{
	if ((mode & 0x00020008) == 0x00020008) {
		//	Ne supporte pas les arbres !
		err = 0x0001;
		return;
	}
	
	if (this->JustEnter (err) == FALSE) return;
	
	static    idx = 0;
	DirEntry* dp  = open_cdf (dir, cdrom);
	
	num = 0;
	id  = idx++;
	err = 0;
	ch  = 0;
	
	if (dp == 0) {
		err = 0x851A;		//	"dossier pas trouv"
		return;
	}
	
	IsoList* list = new IsoList;
	
	if (list == 0) {
		err = 0x8341;		//	"plus assez de mmoire"
		delete dp;
		return;
	}
	
	list->b_dossier = ((mode) & (1 << 23)) ? TRUE : FALSE;
	
	Card32 datalen = ISO_WD (FDVA (dp, datalen, cdrom->type));
	Card32 lbn     = 0;
	Card32 count   = 0;
	
	char*  buffer   = new char[cdrom->lbs];
	
	while (get_blk_dirent (dp, buffer, lbn, cdrom)) {
		
		DirEntry* ldp     = (DirEntry*)(buffer);
		Card32    reclen  = FDV (ldp, reclen, cdrom->type);
		Card32    namelen = FDV (ldp, namlen, cdrom->type);
		
		count    = (datalen > cdrom->lbs) ? cdrom->lbs : datalen;
		datalen -= count;
		
		while (count > sizeof (FSdir) && reclen && namelen) {
			char* n = FDV (ldp, name, cdrom->type);
			
			//	Trouv une description de fichier/dossier. Il faut maintenant gnrer
			//	une description adquate...
			
			IsoElemOne* one   = list->Insert ();
			CdromTime*  time  = (CdromTime*)(& ldp->fsd.iso_dir.years);
			Card8       flags = FDV (ldp, flags, cdrom->type);
			
			if (one == 0) {
				err = 0x8341;
				num = 0;
				delete dp;
				delete list;
				delete buffer;
				return;
			}
			
			if ( (namelen == 1) && (n[0] <= 1) ) {
				strcpy (one->name, (n[0] == 0) ? "." : "..");
				one->flags = 0xFF;
				one->size  = 0;
			} else {
				one->size          = ISO_WD (FDVA (ldp, datalen, cdrom->type));
				one->name[namelen] = 0; strncpy (one->name, n, namelen);
				one->date[0]       = make_bcd (time->day);
				one->date[1]       = make_bcd (time->month);
				one->date[2]       = make_bcd (time->years);
				one->date[3]       = make_bcd (time->hour);
				one->date[4]       = make_bcd (time->min);
				one->date[5]       = make_bcd (time->sec);
				one->flags         = flags & (CD_VISABLE | CD_DIRECTORY);
				num++;
			}
			
			count  -= reclen;
			ldp     = (DirEntry*)(((char *) ldp) + reclen);
			reclen  = FDV (ldp, reclen, cdrom->type);
			namelen = FDV (ldp, namlen, cdrom->type);
		}
		
		if (datalen == 0) break;
		lbn++;
	}
	
	delete buffer;
	delete dp;
	
	ch  = (Card32)(list);
	this->open++;
}


void
Iso9660::ListBuf (Card32 ch, Card32& len, Card8* buf, Int16& err)
{
	SmakyNameArgs* args = (SmakyNameArgs*)(buf);
	Card32         num  = len / 64;
	Card32         size = 0;
	
	if (ch) {
		IsoList* list = (IsoList*)(ch);
		
		while (num--) {
			
			for (;;) {
				
				IsoElemOne* one = list->GetNext ();
				if (one == 0) {
					len = size;
					return;
				}
				
				if (one->flags == 0xFF) continue;
				
				strcpy (args->name, one->name);
				
				char* name = args->name;
				
				while (*name && (name[0] != ';')) {
					name[0] = toupper (name[0]);
					name++;
				}
				
				name[0] = 0;
								
				args->d_create   = ((SmakyDate*)(one->date))[0];
				args->d_modif    = args->d_create;
				args->d_use      = args->d_create;
				args->type       = 0;
				args->open       = 0;
				args->attr       = 0x000007FF;
				args->length     = one->size;
				args->block_num  = (args->length+255) / 256;
				args->block_last = args->length - args->block_num * 256;
				args->code       = 0;
				args->rank       = num ? (Card32)(args + 1) : 0;
				
				if ((one->flags & CD_VISABLE) == 0) {
					args->attr |= 0x00002000;
				}
				
				if (one->flags & CD_DIRECTORY) {
					
					args->attr |= 0x00001000;
					
					if (list->b_dossier) {
						name[0] = ':';
						name[1] = 0;
					} else {
						name[0] = '.';
						name[1] = 'D';
						name[2] = 'I';
						name[3] = 'R';
						name[4] = 0;
					}
				}
				
				break;
			}
			
			size += 64;
			args += 1;
		}
		
		err = 0x0000;
		return;
	}
	
	len = 0;
}


void
Iso9660::ListClose (Card32 ch, Int16& err)
{
	if (ch) {
		IsoList* list = (IsoList*)(ch);
		delete list;
		this->open--;
	}
}


void
Iso9660::Mount ()
{
}


void
Iso9660::Unmount ()
{
}


void
Iso9660::Request (const char* arg, Int16& err)
{
	err = 0x0001;
	
	if (strcmp (arg, "EJECT") == 0) {
		
		if (this->open) {
			err = 0x8515;				//	"fichier utilis"
			return;
		}
		
		if (cdrom->fd == 0) {
			
			cdrom->fd = cd_open (cdrom_name);
			
			if (cdrom->fd & 0xFFFF0000) {
				err = (Int16)(cdrom->fd);	//	erreur d'ouverture
				cdrom->fd = 0;
				return;
			}
		}
		
		if (cdrom->fd) {
			
			cd_eject (cdrom->fd);
			cd_close (cdrom->fd);
			
			cdrom->fd      = 0;
			this->enter_ok = FALSE;
		}
		
		err = 0x0000;
		return;
	}
}




Iso9660::IsoElemOne*
Iso9660::IsoList::Insert ()
{
	IsoElemTable** tpp  = & this->table;
	
	while ( (tpp[0]) && (tpp[0]->used == ISOMAXELEM) ) {
		tpp = & (tpp[0]->next);
	}
	
	if (tpp[0] == 0) {
		tpp[0] = new IsoElemTable;
		if (tpp[0] == 0) return 0;
	}
	
	this->num++;
	return & tpp[0]->elem[tpp[0]->used++];
}


Iso9660::IsoElemOne*
Iso9660::IsoList::GetNext ()
{
	IsoElemTable** tpp   = & this->table;
	Card32         index = this->index++;
	
	while ( (tpp[0]) && (tpp[0]->used < index) ) {
		index -= ISOMAXELEM;
		tpp    = & (tpp[0]->next);
	}
	
	if (tpp[0] == 0) {
		return 0;
	}
	
	return & tpp[0]->elem[index];
}

