#include <stddef.h>
#include <malloc.h>
#include <smaky.h>
#ifdef	DEBUG_MALLOC
#include <stdio.h>
#endif

#define	MAXBLK		256
#define	FREE		0x00
#define	USED		0x80
#define	NULLBLK		0x80000000L

static char *_mblk[MAXBLK];	/* system memory heaps, cleared by system */
static long _msiz[MAXBLK];	/* allocated heap sizes, cleared by system */

/*
 *	Up to MAXBLK heaps are allocated from the operating system as they are
 *	needed to fill requests for dynamic memory.  These heaps are then
 *	divided into blocks for parcelling out by the user-callable memory
 *	allocation routines.  If all the storage in a heap is freed, the heap
 *	will be freed to the OS.  Each heap beings with a pointer to the first
 *	free block, or NULL if there are no free blocks in this heap.  Each
 *	block begins with a 4-byte header which defines the number of bytes
 *	in the block, including the header.  Since blocks in a heap are known
 *	to be contiguous, this value also defines the beginning of the next
 *	block in the heap.  The high bit of the header is set if the block
 *	is used and clear if it is free.  The heaps ends with a block header
 *	which indicates a used block containing 0 bytes.  This is the constant
 *	value NULLBLK.  Free blocks contain an additional pointer field,
 *	immediatly following the header, which is a pointer to the header of
 *	the next free block, or NULL.
 */

static char *
makeblk (size)
     long size;
     /*
 *	Get a new major block from the system that will hold at least
 *	<size> bytes.
 */
{
  register int i, e;
  register char *p;
  register long n, minsiz, bsiz, *q;
  long nn, mm;

#define	SEGMENT	8192L

  for (i = 0; i < MAXBLK; ++i)
    {
      if (_mblk[i] != NULL)
	continue;		/* skip used heaps */
      minsiz = (size + 0x200L) & ~0x1FFL;	/* round to nearest 512 */
      if ((minsiz < SEGMENT) && ((i & 1) == 0) && (i < (MAXBLK >> 1)))
	bsiz = (SEGMENT * 4);	/* 32 K */
      else
	{
	  if (minsiz < (SEGMENT * 4))
	    bsiz = (SEGMENT * 16);	/* 128 K */
	  else
	    bsiz = minsiz;
	}
      e = G_argmem (&nn, &mm);
      if (e != 0)
	return (NULL);		/* biggest hole */
      n = ~0x1FFL & (mm - 512L);/* system memory available */
      if (n < bsiz)
	{
	  if (n < minsiz)
	    return (NULL);
	  else
	    bsiz = minsiz;
	}
      nn = bsiz;
      e = G_getmem (&nn, GSM_MTYPCP, &mm);
      if ((e != 0) || (bsiz != nn))
	return (NULL);
      _mblk[i] = p = ((char *) mm);
      _msiz[i] = bsiz;
      q = ((long *) (p + bsiz));/* thread starting blocks */
      q[-1] = NULLBLK;
      q = ((long *) (p + 4L));
      q[-1] = (long) q;
      q[0] = (bsiz - 8L);
      q[1] = (long) NULL;
      p[4] = FREE;
      return (p);
    }
  return (NULL);
}

static char *
splitblk (addr, size)
     register long **addr;
     register long size;
     /*
 *	Split block at *<addr> into a used block containing <size> bytes
 *	and a free block containing the remainder.
 */
{
  register long n, *p, *q;

  n = *(p = *addr);		/* get actual block size */
  if (n > (size + 8L))
    {				/* is it worth splitting? */
      n -= size;
      /* calculate "break" point */
      q = ((long *) (((char *) p) + size));
      p[0] = size;
      q[0] = n;
      q[1] = p[1];
      *addr = q;
    }
  else				/* not worth splitting */
    *addr = ((long *) p[1]);	/* remove from free list */
  *((char *) p) = USED;		/* mark block "used" */
  return ((char *) p);
}

static char *
findblk (size)
     register long size;
     /*
 *	Find the smallest unused block containing at least <size> bytes.
 */
{
  register int i;
  register long n, tsiz = 0x7FFFFFFFL, **p, *q, *tptr = NULL;

  for (i = 0; i < MAXBLK; ++i)
    {
      if ((p = ((long **) _mblk[i])) == NULL)
	continue;		/* skip unavailable heaps */
      while (q = *p)
	{
	  n = *q;
	  if ((n >= size) && (n < tsiz))
	    {			/* it fits */
	      tsiz = n;
	      tptr = ((long *) p);
	    }
	  p = ((long **) (q + 1));
	}
    }
  return ((char *) tptr);
}

static void 
mergeblk (i)
     int i;
     /*
 *	Merge adjacent "free" blocks in heap <i>.  Links in the free chain
 *	are guarenteed to be in forward order.
 */
{
  register long n, *p, *q;
  char *mem;
  char *desc;

  p = (long *) _mblk[i];
  if ((p = ((long *) *p)) == NULL)	/* empty chain */
    return;
  while (q = ((long *) p[1]))
    {
      n = *p;
      if (((char *) p) + n == ((char *) q))
	{			/* adjacent free block */
	  p[1] = q[1];		/* re-link free chain */
	  *p += *q;		/* adjust block size */
	}
      else
	p = q;
    }
  /* check to see if the entire heap can be returned to the OS */
  q = ((long *) (((char *) p) + (*p)));
  if ((((long *) _mblk[i]) == (p - 1)) && ((*q) == NULLBLK))
    {
      mem = _mblk[i];
      desc = mem;
      G_givmem (mem, GSM_MTYPCP, desc);
      _mblk[i] = NULL;
      _msiz[i] = 0L;
    }
}

static long *
unlinkblk (addr)
     register long *addr;
     /*
 *	Unlink memory block at <addr> from free memory chain.
 */
{
  register int i;
  register long *p, *q;

  for (i = 0; i < MAXBLK; ++i)
    {
      if ((q = p = ((long *) _mblk[i])) == NULL)
	continue;		/* skip unavailable blocks */
      if ((addr < p) || (addr > ((long *) (((char *) p) + _msiz[i]))))
	continue;		/* block range check */
      while (p = ((long *) *q))
	{
	  if (p == addr)
	    {			/* found the right block */
	      q[0] = p[1];	/* unlink it */
	      return (p);
	    }
	  q = p + 1;
	}
    }
  return (NULL);
}

/*--------------------- Documented Functions ---------------------------*/

void *
malloc (size)
     register size_t size;
{
  register char *p;

#ifdef DEBUG_MALLOC
  printf ("\nmalloc: size = %d", size);
#endif

  /*if (size == 0)
		return(NULL); only Turbo C does this check */
  if (size <= 4L)
    size = 8L;			/* minimum allocation */
  else
    size = (size + 5L) & ~1L;	/* header & alignment */
  if ((p = findblk (size)) == NULL)
    if ((p = makeblk (size)) == NULL)
      {
#ifdef DEBUG_MALLOC
	printf (" ret = NULL");
#endif
	return (NULL);
      }
  p = splitblk (p, size);
#ifdef DEBUG_MALLOC
  printf (" ret = %X", p + 4);
#endif
  return (p + 4);		/* skip over header */
}

void *
calloc (n, size)
     size_t n;
     size_t size;
{
  register size_t total;
  register char *p, *q;

#ifdef DEBUG_MALLOC
  printf ("\ncalloc: n = %d size = %d", n, size);
#endif
  total = (n * size);
  if (p = malloc (total))
    for (q = p; total--; *q++ = 0)
      ;
#ifdef DEBUG_MALLOC
  printf (" ret = %X", p);
#endif
  return (p);
}

long 
memavail ()
{
  register int i;
  register long n, tsiz = 0L, **p, *q;
  long total, biggest;

  for (i = 0; i < MAXBLK; ++i)
    {
      if ((p = ((long **) _mblk[i])) == NULL)
	continue;		/* skip unavailable heaps */
      while (q = *p)
	{
	  n = *q;
	  if (n > tsiz)		/* largest block so far */
	    tsiz = n;
	  p = ((long **) (q + 1));
	}
    }
  i = G_argmem (&total, &biggest);
  if (i != 0)
    biggest = 0;
  n = biggest;
  if (n < 1024)
    return (tsiz);
  n -= (1024L + 16L);
  n &= ~0x1FFL;			/* system memory available */
  return ((n > tsiz) ? n : tsiz);
}

void 
free (ad)
     void *ad;
{
  register int i;
  register long *p, *q, *addr = (long *) ad;

#ifdef DEBUG_MALLOC
  printf ("\nfree: addr = %X", addr);
#endif
  if (addr == NULL)
    return;
  --addr;			/* point to block header */
  for (i = 0; i < MAXBLK; ++i)
    {
      if ((p = ((long *) _mblk[i])) == NULL)
	continue;		/* skip unavailable blocks */
      if ((addr < p) || (addr > ((long *) (((char *) p) + _msiz[i]))))
	continue;		/* block range check */
      while (q = ((long *) *p))
	{
	  ++q;
	  if ((addr < q) && (addr > p))
	    break;
	  p = q;
	}
      *((char *) addr) = FREE;	/* link into free chain */
      addr[1] = *p;
      *p = ((long) addr);
      mergeblk (i);
      return;
    }
  return;
}

long 
msize (ad)
     void *ad;
{
  register long *addr = (long *) ad;

  return ((addr[-1] & 0x00FFFFFF) - 4L);
}

void *
realloc (ad, size)
     void *ad;
     size_t size;
{
  register long n, m, d, *p, *q, *addr = (long *) ad;

#ifdef DEBUG_MALLOC
  printf ("\nrealloc: addr = %X size = %d", addr, size);
#endif
  if (addr == NULL)
    return (malloc (size));
  if (size == 0)
    {
      free (addr);
      return (addr);
    }
  p = addr - 1;
  n = p[0] & 0x00FFFFFFL;	/* get actual block size */
  m = (size + 5L) & ~1L;	/* calculate new block size */
  if (m <= n)
    {				/* shrink... */
      if ((n -= m) >= 8L)
	{			/* big enough to shrink */
	  /* calculate new break */
	  q = ((long *) (((char *) p) + m));
	  p[0] = m;
	  *((char *) p) = USED;
	  q[0] = n;
	  *((char *) q) = USED;
	  free (q + 1);		/* free remainder */
	}
    }
  else
    {				/* grow... */
      d = *(q = ((long *) (((char *) p) + n)));
      if ((d > 0) &&		/* next block is free */
	  (m <= (n + d)) &&	/* combination is big enough */
	  (unlinkblk (q)))
	{			/* block can be unlinked */
	  p[0] = (n + d);
	  *((char *) p) = USED;	/* extend in place */
	  return (realloc (addr, size));
	}
      if (p = ((long *) malloc (m - 4L)))
	{			/* allocate bigger block */
	  memcpy (p, addr, n - 4L);	/* copy old data */
	  free (addr);		/* free old block */
	  addr = p;
	}
      else
	addr = NULL;
    }
#ifdef DEBUG_MALLOC
  printf (" ret = %X", addr);
#endif
  return (addr);
}
