/*
 * Copyright (c) 1992, 1993 William F. Jolitz, TeleMuse
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This software is a component of "386BSD" developed by 
 *	William F. Jolitz, TeleMuse.
 * 4. Neither the name of the developer nor the name "386BSD"
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS A COMPONENT OF 386BSD DEVELOPED BY WILLIAM F. JOLITZ 
 * AND IS INTENDED FOR RESEARCH AND EDUCATIONAL PURPOSES ONLY. THIS 
 * SOFTWARE SHOULD NOT BE CONSIDERED TO BE A COMMERCIAL PRODUCT. 
 * THE DEVELOPER URGES THAT USERS WHO REQUIRE A COMMERCIAL PRODUCT 
 * NOT MAKE USE OF THIS WORK.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE DEVELOPER BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Rock Ridge Interchange protocol implementation
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "cdromfs.h"
#include "susp.h"
#include "rrip.h"

/*
 *	Use RRIP to flesh out a POSIX attribute buffer.
 */

void
rrip_statb (DirEntry* dp, stat* sb, FS* fs, int r)
{
	SuspCont cont;
	
	RripPX*	 px;
	RripTF*	 tf;
	RripPN*	 pn;
	
	//	No continuation to begin with
	
	cont.curr     = 0;
	cont.cont_buf = 0;
	
	//	Do we have POSIX file attributes for this file ?
	
	if ((px = (RripPX*) handle_susp (dp, &cont, "PX", fs, r)) != 0) {
		sb->st_mode  = ISO_WD (&px->mode);
		sb->st_nlink = ISO_WD (&px->links);
		sb->st_uid   = ISO_WD (&px->uid);
		sb->st_gid   = ISO_WD (&px->gid);
	}

	//	Rescan
	
	cont.curr = 0;
	
	//	Do we have a time field for this file ?
	
	if ((tf = (RripTF*) handle_susp (dp, &cont, "TF", fs, r)) != 0) {
		
		char* tfp = tf->stamps;
		
		if (tf->flags & TF_CREATE) sb->st_ctime = rrip_time (&tfp, tf->flags);
		if (tf->flags & TF_MODIFY) sb->st_mtime = rrip_time (&tfp, tf->flags);
		if (tf->flags & TF_ACCESS) sb->st_atime = rrip_time (&tfp, tf->flags);
	}
	
	//	Rescan
	
	cont.curr = 0;
	
	//	Do we have a POSIX device field for this file ?
	
	if ( ((sb->st_mode & S_IFMT) == S_IFCHR)
	  || ((sb->st_mode & S_IFMT) == S_IFBLK) ) {
		if ((pn = (RripPN*) handle_susp (dp, &cont, "PN", fs, r)) != 0) {
			sb->st_rdev = ISO_WD(&pn->dev_low);
		}
	}
	
	if (cont.cont_buf) delete cont.cont_buf;
	
	//	XXX: what do we set sb->st_dev to ?
	
	sb->st_ino  = ISO_WD (FDVA (dp, location, fs->type));
	sb->st_size = ISO_WD (FDVA (dp, datalen, fs->type));
}

/*
 *	Julian date at end of each month.
 */

static const int jdm[] = { 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };

/*
 *	Convert between ISO time formats and POSIX absolute time
 */

time_t
rrip_time (char** tp, char flg)
{
	CdromTime* crt = (CdromTime*)(tp[0]);
	time_t     val;
	
	//	17 byte BCD times ?
	//	XXX: currently don't even do 17 byte times !
	
	tp[0] = (flg & TF_LONG) ? tp[0] + 17 : tp[0] + 7;

	//	Need to calculate in units of seconds from midnight,
	//	January 1970 (like BC or AD, it's AU == After Unix !)
	
	//	Tally the days.
	
	val = (crt->years - 70) * 365;
	
	//	Compensate for leap years...
	
	if (crt->years > 72) val += (crt->years - 72 - 1) / 4 + 1;
	if (crt->month > 1)  val += jdm[crt->month - 1 - 1];
	if ((crt->month > 2) && ((crt->years % 4) == 0)) val++;
	
	//	Days this month
	
	val += crt->day - 1;
	
	//	Tally seconds in the day
	
	val = ((val * 24 + crt->hour) * 60 + crt->min) * 60 + crt->sec;
	
	//	Compensate for timezone, carefully noticing sign bit
	
	val -= crt->tz*15*60;
	return val;
}


