/*
 *	channel.cc
 *
 *	Gestion de canaux.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#pragma implementation "channel.h"

#include "node.h"
#include "channel.h"
#include "errors.h"

#define MAX_CH	50

static FileChannel* ch_array[MAX_CH] = { 0, 0 /* ... */ };


/*
 *	Initialise un canal.
 */

FileChannel::FileChannel ()
{
	this->node_active = 0;
	this->node_index  = 0;
	this->ch_count    = 0;
	this->position    = 0;
}



/*
 *	Supprime un canal.
 */

FileChannel::~FileChannel ()
{
	this->Close ();
}


/*
 *	Ouvre un canal correspondant  un noeud. L'utilisateur appelle
 *	en principe FileNode::Open qui se charge d'appeler cette mthode.
 */

Bool
FileChannel::Open (FileNode* node, int node_index)
{
	//	Trouve un canal de libre dans la table globale des canaux.
	
	int index = 0;
	
	while ( (++index < MAX_CH) && (ch_array[index]) ) ;
	
	if (index < MAX_CH) {
		
		ch_array[index]   = this;
		
		this->node_active = node;
		this->node_index  = node_index;
		this->ch_count    = index;
		this->position    = 0;
		
		this->node_active->RegisterChannel (this->node_index, this);
		
		return TRUE;
	}
	
	return FALSE;
}


/*
 *	Ferme le canal. Libre le canal global et local.
 */

Bool
FileChannel::Close ()
{
	if (this->node_active) {
		this->node_active->RegisterChannel (this->node_index, 0);
		this->node_active = 0;
		this->node_index  = 0;
	}
	
	if (this->ch_count) {
		ch_array[this->ch_count] = 0;
	}
	
	return TRUE;
}


/*
 *	Retourne le pointeur au nime canal.
 */

FileChannel*
FileChannel::GetChannel (int index)
{
	if ( (index > 0)
	  && (index < MAX_CH) ) {
		return ch_array[index];
	}
	
	return 0;
}


/*
 *	Lit depuis le canal, c'est en grande partie le noeud qui doit
 *	faire le travail.
 */

Bool
FileChannel::Read (Card8* data, Card32& len, Int16& err)
{
	err = ERR_ILL_CHANNEL;
	
	if (this->node_active) {
		this->node_active->ReadData (this->position, len, data, err);
	}
	
	return (err == ERR_OK) ? TRUE : FALSE;
}



/*
 *	Ecrit sur le canal, c'est en grande partie le noeud qui doit
 *	faire le travail.
 */

Bool
FileChannel::Write (const Card8* data, Card32& len, Int16& err)
{
	Int16 err = ERR_ILL_CHANNEL;
	
	if (this->node_active) {
		this->node_active->WriteData (this->position, len, data, err);
	}
	
	return (err == ERR_OK) ? TRUE : FALSE;
}


/*
 *	Raccourcit un fichier  la longueur actuelle.
 */

Bool
FileChannel::Trunc (Int16& err)
{
	err = ERR_ILL_CHANNEL;
	
	if (this->node_active) {
		this->node_active->Resize (this->position, err);
	}
	
	return (err == ERR_OK) ? TRUE : FALSE;
}


/*
 *	Fixe la nouvelle position pour lire...
 */

Bool
FileChannel::SetPos (Card32& len, Int16& err)
{
	err = ERR_ILL_CHANNEL;
	
	if (this->node_active) {
		Card32 max;
		
		max = this->node_active->FileLength ();
		err = ERR_OK;
		
		if (len > max) {
			len = max;
			err = ERR_OUT_OF_FILE;
		}
		
		this->position = len;
	}
	
	return (err == ERR_OK) ? TRUE : FALSE;
}

