/*
 *	efsopen.cc
 *
 *	Gre l'ouverture et la fermeture d'un fichier quelconque.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#include "efs.h"
#include "node.h"
#include "errors.h"
#include "channel.h"


/*
 *	Ouvre un noeud en lecture et/ou en criture. En fonction du noeud, diverses
 *	actions peuvent avoir lieu.
 */

void
EFS::Open (const char* name, Card32 mode, Card32 len, Card16& ch, Int16& err)
{
	FileNode*    node    = FileNode::Find (this->node, name);
	FileChannel* channel = 0;
	
	if (node == 0) {
		err = ERR_FILE_NOT_FOUND;
		return;
	}
	
	channel = node->Open ();
	
	if ( (channel)
	  && (channel->IsOpen ()) ) {
		
		if (mode & (1 << 13)) {
			Card32 pos = node->FileLength ();
			channel->SetPos (pos, err);
		}
		
		ch  = channel->Channel ();
		err = ERR_OK;
		return;
	}
	
	if (channel) {
		delete channel;
	}
	
	ch  = 0;
	err = ERR_ILL_CHANNEL;
}


/*
 *	Ferme un canal. Libre le noeud associ.
 */

void
EFS::Close (Card16 ch, Card32 h, Card32 d, Int16& err)
{
	FileChannel* channel = FileChannel::GetChannel (ch);
	
	if (channel) {
		delete channel;
		err = ERR_OK;
		return;
	}
	
	err = ERR_ILL_CHANNEL;
}


/*
 *	Lit des donnes depuis le noeud. On passe simplement la main au
 *	canal...
 */

void
EFS::RdByte (Card16 ch, Card32& len, Card8* data, Int16& err)
{
	FileChannel* channel = FileChannel::GetChannel (ch);
	
	if (channel) {
		channel->Read (data, len, err);
		return;
	}
	
	err = ERR_ILL_CHANNEL;
}


/*
 *	Ecrit des donnes vers le noeud. On passe simplement la main au
 *	canal...
 */

void
EFS::WrByte (Card16 ch, Card32& len, const Card8* data, Int16& err)
{
	FileChannel* channel = FileChannel::GetChannel (ch);
	
	if (channel) {
		channel->Write (data, len, err);
		return;
	}
	
	err = ERR_ILL_CHANNEL;
}


/*
 *	Dfinit la position depuis laquelle lire/crire prochainement.
 *	On gre le cas "bte" de la position aprs la fin puis on laisse
 *	faire le canal.
 */

void
EFS::SetPos (Card16 ch, Card32& poslow, Card32& poshigh, Card16 mode, Int16& err)
{
	FileChannel* channel = FileChannel::GetChannel (ch);
	
	if (channel) {
		if (poshigh) {
			poslow  = 0xFFFFFFFF;
			poshigh = 0x00000000;
		}
		
		channel->SetPos (poslow, err);
		return;
	}
	
	err = ERR_ILL_CHANNEL;
}


/*
 *	Retourne la position actuelle dans le canal.
 */

void
EFS::GetPos (Card16 ch, Card32& poslow, Card32& poshigh, Int16& err)
{
	FileChannel* channel = FileChannel::GetChannel (ch);
	
	if (channel) {
		
		poslow  = 0x00000000;
		poshigh = 0x00000000;
		
		channel->GetPos (poslow, err);
		return;
	}
	
	err = ERR_ILL_CHANNEL;
}


/*
 *	Tronque le fichier. Laisse faire le canal.
 */

void
EFS::Trunc (Card16 ch, Int16& err)
{
	FileChannel* channel = FileChannel::GetChannel (ch);
	
	if (channel) {
		channel->Trunc (err);
		return;
	}
	
	err = ERR_ILL_CHANNEL;
}


