/*
 *	node.h
 *
 *	Gestion de noeuds approchant le systme des i-nodes de UNIX. Ceci
 *	permet de grer un arbre de fichiers.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */


#ifndef _NODE_H_
#define _NODE_H_

#include <c++/portable.h>

#define	MAX_CHANNEL	16

class  FileNode;
class  FileChannel;
struct FileData;
class  NtrBAL;
class  NtrSync;

/*
 *	Un noeud reprsente les donnes au moyen de morceaux de
 *	donnes.
 */

struct FileData
{
	FileData*		next;						//	bloc de donnes suivant
	Card32			size;						//	taille des donnes (au total)
	Card32			used;						//	taille rellement utilise
	char			data[4];					//	donnes (x bytes)
};


/*
 *	La classe FileNode gre un noeud dans la hirarchie des fichiers,
 *	que ce soit un fichier ou un dossier ne joue aucun rle.
 */

class FileNode
{
	
protected:
	FileNode*		next;						//	fichier suivant
	FileNode*		prev;						//	fichier prcdent
	
	char			name[16];					//	nom du fichier
	int				name_len;					//	longueur du nom
	
	FileNode*		parent;						//	pointeur au pre ou 0
	FileNode*		children;					//	pointeur aux fichiers ou 0
	
	FileData*		data;						//	pointeur aux donnes
	
	Card32			file_length;				//	taille du fichier
	
	Card32			channel_count;				//	nombre de canaux ouverts
	FileChannel*	channel[MAX_CHANNEL];		//	canaux ouverts
	
	Bool			is_dir;						//	TRUE => c'est un dossier
	Bool			is_del_ok;					//	TRUE => c'est un fichier destructible
	Bool			is_hidden;					//	TRUE => c'est un fichier cach
	Bool			is_alive;					//	TRUE => c'est un fichier rel
	Bool			is_mirror;					//	TRUE => c'est un fichier mirroir
	Bool			is_sync;					//	TRUE => c'est un fichier synchrone
	Bool			is_cycle;					//	TRUE => c'est un fichier cyclique
	Bool			is_dummy_cre_del;			//	TRUE => ignore "cre" et "dtruit"
	
	Card32			cycle_len;					//	taille du cycle
	Card8*			cycle_buffer;				//	buffer cyclique
	
protected:
	Card32			spy_id;						//	identificateur pour l'espionnage
	char			spy_name[32];				//	nom du noeud pour l'espionnage
	NtrBAL*			spy_bal;					//	BAL rserve pour l'espion
	NtrSync*		spy_sync;					//	smaphore  activer aprs l'envoi sur la BAL
	
	Bool			do_spy_open_close;			//	TRUE => signale ouvertures/fermetures
	Bool			do_spy_read;				//	TRUE => signale les lectures
	Bool			do_spy_write;				//	TRUE => signale les critures
	Bool			do_spy_cre_del;				//	TRUE => signale create/delete
	
protected:
	virtual void SpyReport (Card32 event, Card32 arg_1 = 0, Card32 arg_2 = 0);
	
public:
	FileNode (const char* name = "", FileNode* parent = 0);
	virtual ~ FileNode ();
	
	void SetupSpy (Card32 id, const char* name, void* bal, void* sync = 0);
	
	void DoSpyOpenClose (Bool t)    { this->do_spy_open_close = t; }
	void DoSpyRead (Bool t)         { this->do_spy_read       = t; }
	void DoSpyWrite (Bool t)        { this->do_spy_write      = t; }
	void DoSpyCreateDelete (Bool t) { this->do_spy_cre_del    = t; }
	
	Bool IsDir () const				{ return this->is_dir; }
	Bool IsHidden () const			{ return this->is_hidden; }
	Bool IsDelOk () const			{ return this->is_del_ok; }
	Bool IsAlive () const			{ return this->is_alive; }
	Bool IsMirror () const			{ return this->is_mirror; }
	Bool IsSync () const			{ return this->is_sync; }
	Bool IsCycle () const			{ return this->is_cycle; }
	Bool IsDummyCreDel () const		{ return this->is_dummy_cre_del; }
	
	Card32 ChannelCount () const 	{ return this->channel_count; }
	Card32 FileLength () const   	{ return this->file_length; }
	
	FileNode* Children () const  	{ return this->children; }
	FileNode* Next () const      	{ return this->next; }
	const char* Name () const    	{ return this->name; }
	
	void SetHidden (Bool s)			{ this->is_hidden = s; }
	void SetDelOk (Bool s)			{ this->is_del_ok = s; }
	void SetAlive (Bool s)			{ this->is_alive  = s; }
	void SetMirror (Bool s)			{ this->is_mirror = s; }
	void SetSync (Bool s)			{ this->is_sync   = s; }
	void SetCycle (Bool s)			{ this->is_cycle  = s; }
	void SetDummyCreDel (Bool s)	{ this->is_dummy_cre_del = s; }
	void SetCycleLen (Card32 len);
	
	virtual void SetName (const char* name);
	
	virtual Int16 ShadowCreate ();
	virtual Int16 ShadowDelete ();
	
	virtual FileChannel* Open ();
	
	virtual void ReadData (Card32& pos, Card32& len, Card8* data, Int16& err);
	virtual void WriteData (Card32& pos, Card32& len, const Card8* data, Int16& err);
	virtual void Resize (Card32& pos, Int16& err);
	
	virtual void RegisterChannel (int index, FileChannel* channel);
	
	//	Fonctions globales de gestion de noeuds :
	
	static FileNode* Init (const char* name);
	static void      Kill (FileNode* root);
	static FileNode* Find (FileNode* root, const char* name, Bool is_dir = FALSE);
};

#endif

