/*
 *	nodebase.cc
 *
 *	Oprations de base avec des noeuds,  savoir ouverture et fermeture,
 *	lecture et criture.
 *
 *	(C) Copyright 1995, Pierre Arnaud, OPaC bright ideas, CH-1437 SUSCEVAZ
 */

#include <string.h>
#include <stdio.h>
#include <c++/ntrel.h>

#include "node.h"
#include "channel.h"
#include "nif.h"
#include "errors.h"

#define	STOCK_SIZE	1024


/*
 *	Ouvre un noeud et retourne un canal permettant d'y accder. Les mthodes
 *	pour lire et crire des donnes sont propres au noeud.
 */

FileChannel*
FileNode::Open ()
{
	FileChannel* channel = new FileChannel;
	
	if (channel == 0) {
		return 0;
	}
	
	if (this->channel_count < MAX_CHANNEL) {
		for (int index = 0; index < MAX_CHANNEL; index++) {
			if (this->channel[index] == 0) {
				if (channel->Open (this, index)) {
					return channel;
				}
				
				break;
			}
		}
	}
	
	delete channel;
	return 0;
}


/*
 *	Un nouveau canal va tre insr dans le noeud (ou ventuellement
 *	retir si le pointeur est nul).
 */

void
FileNode::RegisterChannel (int index, FileChannel* channel)
{
	int sub = this->channel[index] ? 1 : 0;
	int add = channel ? 1 : 0;
	int mod = add - sub;
	
	this->channel_count += mod;
	this->channel[index] = channel;
	
	if (this->do_spy_open_close) {
		if (add) this->SpyReport (SPY_OPEN);
		if (sub) this->SpyReport (SPY_CLOSE);
	}
}


/*
 *	Mthode de base  appeler par toutes les sous-classes afin de
 *	permettre de tenir des statistiques sur les lectures.
 */

void
FileNode::ReadData (Card32& pos, Card32& len, Card8* data, Int16& err)
{
	err = ERR_OK;
	
	if (pos > this->file_length) {
		err = ERR_OUT_OF_FILE;
		return;
	}
	
	Card32 end = pos + len;
	
	if (end > this->file_length) {
		end = this->file_length;
		len = end - pos;
		err = ERR_OUT_OF_FILE;
	}
	
	if ( (len == 0)
	  && (this->is_sync) ) {
		err = ERR_COME_AGAIN;
		return;
	}
	
	if (this->do_spy_read) {
		this->SpyReport (SPY_READ, pos, len);
	}
	
	if (this->is_cycle) {
		if ( (this->cycle_len == 0)
		  || (this->cycle_buffer == 0) ) {
			err = ERR_OUT_OF_FILE;
			len = 0;
			return;
		}
		
		Card8* buffer = this->cycle_buffer;
		Card32 offset = pos % this->cycle_len;
		Card32 count  = len;
		
		while (count--) {
			
			*data++ = buffer[offset++];
			
			if (offset == this->cycle_len) {
				offset = 0;
			}
		}
		
		pos += len;
		return;
	}
	
	FileData* dptr  = this->data;
	Card32    count = len;
	Card32    skip  = pos;
	Card32    rest  = 0;
	Card32    copy  = 0;
	Card8*    src   = 0;
	
	while (dptr && count) {
		if (skip > dptr->used) {
			skip -= dptr->used;
			dptr  = dptr->next;
			continue;
		}
		
		src    = dptr->data + skip;
		rest   = dptr->used - skip;
		skip   = 0;
		copy   = (count > rest) ? rest : count;
		count -= copy;
		
		while (copy--) {
			*data++ = *src++;
		}
		
		dptr = dptr->next;
	}
	
	pos += len;
}

	static inline void AfX8 (Card32 x) {
		asm (	"movel	%0,d4\n\t"
				".word	0xa032\n\t"
			: : "g" (x) : "d4", "d7" );
	}
	static inline void AfText (const char *text) {
		asm (	"movel	%0,a3\n\t"
			".word	0xa040\n\t"
			: 
			: "g" (text) : "a3", "d7" );
	}

/*
 *	Mthode de base  appeler par toutes les sous-classes afin de
 *	permettre de tenir des statistiques sur les critures.
 */

void
FileNode::WriteData (Card32& pos, Card32& len, const Card8* data, Int16& err)
{
	if (this->do_spy_write) {
		this->SpyReport (SPY_WRITE, pos, len);
	}
	
	if (this->is_cycle) {
		if ( (this->cycle_len == 0)
		  || (this->cycle_buffer == 0) ) {
			err = ERR_NO_MEM;
			len = 0;
			return;
		}
		
		Card8* buffer = this->cycle_buffer;
		Card32 offset = pos % this->cycle_len;
		Card32 count  = len;
		
		while (count--) {
			
			buffer[offset++] = *data++;
			
			if (offset == this->cycle_len) {
				offset = 0;
			}
		}
		
		pos += len;
		err  = ERR_OK;
		
		if (pos > this->file_length) {
			this->file_length = pos;
		}
		
		return;
	}
	
	FileData** dpptr = & this->data;
	FileData*  dptr  = dpptr[0];
	Card32     skip  = pos;
	Card32     room  = 0;
	Card32     count = len;
	Card32     more  = 0;
	Card32     copy  = 0;
	Card8*     dst   = 0;
	
	while (count) {
		
		dptr = dpptr[0];
		
		if (dptr == 0) {
			
			dptr = (FileData*) new char[sizeof (FileData)+STOCK_SIZE];
			
			if (dptr == 0) {
				err = ERR_NO_MEM;
				return;
			}
			
			dptr->next = 0;
			dptr->size = STOCK_SIZE;
			dptr->used = 0;
			
			dpptr[0] = dptr;
		}
		
		if (skip > dptr->used) {
			
			skip -= dptr->used;
			dpptr = & dptr->next;
			
			if (dptr->next == 0) {
				AfText ("Strange write: used="); AfX8 (dptr->used);
				AfText (" size="); AfX8 (dptr->size);
				AfText ("\r");
			}
			
			continue;
		}
		
		room   = dptr->size - skip;
		dst    = dptr->data + skip;
		copy   = (count > room) ? room : count;
		count -= copy;
		dpptr  = & dptr->next;
		more   = ((copy+skip) > dptr->used) ? (copy+skip - dptr->used) : 0;
		skip   = 0;
		
		//	Allonge le bloc si cela est ncessaire. Si le bloc de donnes est
		//	allong, c'est forcment le dernier, de plus il faut allonger le
		//	fichier.
		
		dptr->used        += more;
		this->file_length += more;
		
		while (copy--) {
			*dst++ = *data++;
		}
	}
	
	pos += len;
}


/*
 *	Modifie la taille d'un fichier, que ce soit pour le raccourcir
 *	ou le rallonger (pas implment).
 */

void
FileNode::Resize (Card32& pos, Int16& err)
{
	FileData** dpptr = & this->data;
	FileData*  dptr  = dpptr[0];
	Card32     count = pos;
	
	while (dptr) {
		if (count > dptr->used) {		//	??? used or size ???
			count -= dptr->used;
			dptr   = dptr->next;
			continue;
		}
		
		//	Raccourcit le bloc (ventuellement en le supprimant s'il
		//	devient vide...) et le fichier par la mme occasion.
		
		this->file_length -= dptr->used - count;
		dptr->used         = count;
		
		if (count == 0) {
			dpptr[0] = dptr->next;
			delete dptr;
		} else {
			dpptr = & dptr->next;
			count = 0;
		}
		
		dptr = dpptr[0];
	}
	
	err = ERR_OK;
}


/*
 *	Transmet un vnement  l'observateur.
 */

void
FileNode::SpyReport (Card32 event, Card32 arg_1, Card32 arg_2)
{
	if (this->spy_bal) {
		SpyRecord spy = { event, 0, this->spy_id, this->spy_name, arg_1, arg_2 };
		this->spy_bal->GiveMessage (&spy, sizeof (spy), BAL_COPY_MESSAGE);
		if (this->spy_sync) {
			this->spy_sync->Signal ();
		}
	}
}




/*
 *	Cre un noeud pour le fichier "ombre".
 */

Int16
FileNode::ShadowCreate ()
{
	return ERR_OK;
}


/*
 *	Supprime un noeud pour le fichier "ombre".
 */

Int16
FileNode::ShadowDelete ()
{
	return ERR_OK;
}


