/*
 *	nodemirror.cc
 *
 *	Noeud particulier permettant de grer des fichiers mirroirs.
 *
 *	(C) Copyright 1995-1996, Pierre Arnaud, OPaC bright ideas
 *	    CH-1437 SUSCEVAZ
 */

#include <string.h>
#include <c++/fos.h>
#include <c++/mon.h>

#include "smakyargs.h"
#include "nodemirror.h"
#include "nif.h"
#include "errors.h"


FileNodeMirror::FileNodeMirror (const char* name, FileNode* parent)
	: FileNode (name, parent)
{
	this->mirror[0] = 0;
	this->channel   = 0;
	this->error     = 0;
	
	this->is_read_ok    = TRUE;
	this->is_write_ok   = TRUE;
	this->is_unit       = FALSE;
	this->is_large_unit = FALSE;
	
	this->do_auto_create   = FALSE;
	this->do_silent_create = FALSE;
	this->do_real_delete   = TRUE;
	
	FileNode::SetMirror (TRUE);
	FileNode::SetDelOk (TRUE);
}


FileNodeMirror::~FileNodeMirror ()
{
	this->CloseIfNeeded ();
}


void
FileNodeMirror::OpenIfNeeded ()
{
	if (this->channel == 0) {
		
		Card32 mode = FOS_OPEN_READ | FOS_OPEN_WRITE;
		
		this->is_read_ok  = TRUE;
		this->is_write_ok = TRUE;
		
		while (mode) {
			
			//	Tente d'ouvrir en lecture et criture. Abandonne un des modes
			//	si cela s'avre ncessaire.
			
			this->is_unit = FALSE;
			this->error   = FOS::Open (this->mirror, mode, this->channel);
			
			if (this->error == 0x8553) {
				
				//	Il n'y a pas de nom de fichier, c'est donc un nom d'unit...
				
				mode &= ~ FOS_OPEN_WRITE;
				mode |= FOS_OPEN_EXTEND | FOS_OPEN_MMUNIT;
				
				this->is_write_ok = FALSE;
				this->is_unit     = TRUE;
				
				this->error = FOS::Open (this->mirror, mode, this->channel);
				
//				AfText ("Opened MM-unit\r");
			}
			
			if (this->error == 0x8533) {
				mode &= ~ FOS_OPEN_READ;
				this->is_read_ok = FALSE;
				continue;
			}
			
			if (this->error == 0x8534) {
				mode &= ~ FOS_OPEN_WRITE;
				this->is_write_ok = FALSE;
				continue;
			}
			
			break;
		}
		
		if ( (this->error == ERR_FILE_NOT_FOUND)
		  && (this->do_auto_create) ) {
			this->error = FOS::Open (this->mirror, mode | FOS_OPEN_CREATE, this->channel);
		}
		
		if (this->error) {
			this->channel = 0;
		}
	}
}


void
FileNodeMirror::CloseIfNeeded ()
{
	if (this->channel) {
		
		FOS::Close (this->channel);
		
		this->channel = 0;
		this->error   = 0;
		this->is_unit = FALSE;
	}
}


/*
 *	Dfinit le fichier mirroir associ. On cherche sa taille, car elle
 *	sera utilise ensuite automatiquement...
 */

void
FileNodeMirror::SetMirror (const char* name)
{
	this->CloseIfNeeded ();
	
	strcpy (this->mirror, name);
	
	SmakyArgs arguments = { 0 /* ... */ };
	Card16    channel   = 0;
	int       error     = 0;
	
	error = FOS::ArgsOpen (this->mirror, FOS_OPEN_READ, arguments, channel);
	
	if (error == 0x8553) {
				
		//	Il n'y a pas de nom de fichier, c'est donc un nom d'unit...
		
		Card32 free = 0;
		Card32 used = 0;
		
		error = FOS::Disk (this->mirror, free, used);
		
//		AfText ("Disk returned "); AfX8 (free); AfSpace (); AfX8 (used); AfCR ();
		
		if (error == 0) {
			this->is_large_unit = (free+used > 20000) ? TRUE : FALSE;
			this->file_length   = (free+used) * 256;
			this->file_length  += (this->is_large_unit) ? (32*256) : (16*256);
			return;
		}
	}
	
	if (error) {
		this->file_length = 0;
	} else {
		this->file_length = arguments.length;
		FOS::Close (channel);
	}
}


/*
 *	Cre le fichier mirroir.
 */

Int16
FileNodeMirror::ShadowCreate ()
{
	this->CloseIfNeeded ();
	
	if (this->do_spy_cre_del) {
		this->SpyReport (SPY_CREATE, (Card32)(this->name));
	}
	
	Card32 attr  = ATTR_DEFAULT | (this->IsHidden () ? 0 : ATTR_LIST)
							    | (this->IsDelOk () ? ATTR_DELETE : 0);
	
	int    error = FOS::Create (this->mirror, attr);
	
	//	Si la cration est silencieuse, on laisse tomber les erreurs qui
	//	indiquent que le fichier existait dj.
	
	if ( (this->do_silent_create)
	  && (error == ERR_FILE_EXISTS) ) {
		error = 0;
	}
	
	return error;
}


/*
 *	Supprime le fichier mirroir.
 */

Int16
FileNodeMirror::ShadowDelete ()
{
	this->CloseIfNeeded ();
	
	if (this->do_spy_cre_del) {
		this->SpyReport (SPY_DELETE, (Card32)(this->name));
	}
	
	if (this->do_real_delete) {
		
		Int16 error = FOS::Delete (this->mirror);
		
		if (error == ERR_OK) {
			this->file_length = 0;
		}
		
		return error;
	}
	
	return ERR_OK;
}


void
FileNodeMirror::ReadData (Card32& pos, Card32& len, Card8* data, Int16& err)
{
	this->OpenIfNeeded ();

	if (this->channel == 0) {
		err = this->error;
		return;
	}
	
	if (this->is_read_ok == FALSE) {	
		err = ERR_OUT_OF_FILE;
		len = 0;
		return;
	}
	
	if (this->do_spy_read) {
		this->SpyReport (SPY_READ, pos, len);
	}
	
	if (this->is_unit) {
		
		static char* buffer = new char[64*1024+256];
		Card32       total  = len;
		Card32       offset = (this->is_large_unit) ? (32*256) : (16*256);
		
		if (pos > this->file_length) {
			err = 0x8511;
			len = 0;
			return;
		}
		
		err = 0;
		
		if (pos+len > this->file_length) {
			len = this->file_length - pos;
			err = 0x8511;
		}
		
		AfText ("Read "); AfX8 (pos); AfSpace (); AfX8 (len); AfSpace (); AfX4 (err); AfSpace ();
		
		while (total) {
			
			Card32 skip   = pos & 0x000000FF;
			Card32 copy   = (total > (64*1024-skip)) ? (64*1024-skip) : (total);
			
			Card8  ph = (pos < offset) ? 0xFF : 0x00;
			Card32 pl = (pos & 0xFFFFFF00);
			Card32 l  = (copy+256) & 0xFFFFFF00;
			
			if (this->is_large_unit) {
				ph  = 0x40;					//	seek absolu support par $SCSI
			} else {
				pl -= offset;				//	seek normal pour les petits drives, tient compte de l'offset
			}
			
			AfX2 (ph); AfX8 (pl); AfSpace (); AfX8 (l); AfSpace ();
			
			FOS::SetPos (this->channel, pl, ph);
			FOS::Read (this->channel, l, buffer);
			
			memcpy (data, buffer+skip, copy);
			
			total -= copy;
			pos   += copy;
			data  += copy;
		}
		
		AfCR ();
		
		return;
	}
	
	err  = FOS::SetPos (this->channel, pos);
	err  = FOS::Read (this->channel, len, data);
	pos += len;
}


void
FileNodeMirror::WriteData (Card32& pos, Card32& len, const Card8* data, Int16& err)
{
	this->OpenIfNeeded ();
	
	if (this->channel == 0) {
		err = this->error;
		return;
	}
	
	if (this->is_write_ok == FALSE) {	
		err = ERR_OUT_OF_FILE;
		len = 0;
		return;
	}
	
	if (this->do_spy_write) {
		this->SpyReport (SPY_WRITE, pos, len);
	}
	
	err  = FOS::SetPos (this->channel, pos);
	err  = FOS::Write (this->channel, len, data);
	pos += len;
	
	if (this->file_length < pos) {
		this->file_length = pos;
	}
}


void
FileNodeMirror::Resize (Card32& pos, Int16& err)
{
	this->OpenIfNeeded ();
	
	if (this->channel == 0) {
		err = ERR_ILL_CHANNEL;
		return;
	}
	
	if (this->is_unit) {
		err = 0;
		return;
	}
	
	err = FOS::SetPos (this->channel, pos);
	err = FOS::Trunc (this->channel);
	
	if (this->file_length > pos) {
		this->file_length = pos;
	}
}


void
FileNodeMirror::RegisterChannel (int index, FileChannel* channel)
{
	if (channel == 0) {
		this->CloseIfNeeded ();
	}
	
	FileNode::RegisterChannel (index, channel);
}


