/*
 *	atom.cxx
 *
 *	Smaky Registry / INTERNAL.
 *
 *	Atoms are unique strings.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#include "atom.h"
#include "portable.h"
#include "registry.h"

/*****************************************************************************/

struct AtomRec
{
	AtomRec*		next;
	AtomRec*		prev;
	int				length;
	char*			name;
	char			store[1];
};

/*****************************************************************************/

#define	PADDING			(0x3L)
#define	MIN_MEMORY		1000

/*****************************************************************************/

static AtomRec* g_head = 0;
static AtomRec* g_tail = 0;

/*****************************************************************************/

//	In order to avoid memory leaks when leaving the program, we
//	clean up the atom list.

static class AtomMemory
{
	struct Memory
	{
		Memory*		next;
		Memory*		prev;
		char*		ptr;
		long		len;
	};

	Memory*			head;
	Memory*			tail;

public:
	AtomMemory ()
	{
		this->head = 0;
		this->tail = 0;
	}

	~ AtomMemory ()
	{
		Memory* iter = this->head;
		Memory* next = 0;
		
		while (iter) {
			next = iter->next;
			delete iter;
			iter = next;
		}
		this->head = 0;
		this->tail = 0;
	}

	char* Alloc (long size)
	{
		size +=  PADDING;
		size &= ~PADDING;
		
		Memory* mem = this->head;
		long    len = (size < MIN_MEMORY) ? MIN_MEMORY : size;
		
again:	while (mem) {
			if (mem->len >= size) {
				char* ptr = mem->ptr;
				mem->len -= size;
				mem->ptr += size;
				return ptr;
			}
			
			mem = mem->next;
		}
		
		len += sizeof (Memory);
		mem  = (Memory*) new char[len];
		
		if (mem) {
			mem->next = this->head;
			mem->prev = 0;
			(mem->next ? mem->next->prev : this->tail) = mem;
			(mem->prev ? mem->prev->next : this->head) = mem;
			mem->len  = len;
			mem->ptr  = (char*)(mem+1);
			goto again;
		}

		return 0;
	}
} memory;

/*****************************************************************************/

static const char*
AtomFind (const char* n)
{
	AtomRec* find = g_head;
	int      len  = strlen (n);

	while (find) {
		if ( (find->length == len)
		  && (find->store[0] == n[0])
		  && (strcmp (find->name, n) == 0) ) {
			return find->name;
		}
		
		find = find->next;
	}

	find = (AtomRec*) memory.Alloc (sizeof (AtomRec) + len);
	
	find->next   = 0;
	find->prev   = g_tail;
	find->length = len;
	find->name   = find->store;
	strcpy (find->store, n);
	
	(find->next ? find->next->prev : g_tail) = find;
	(find->prev ? find->prev->next : g_head) = find;
	
	return find->name;
}

/*****************************************************************************/

Atom::Atom (const char* n)
{
	if (n == 0) n = "";
	this->name = AtomFind (n);
}

Atom&
Atom::operator = (const char* n)
{
	if (n == 0) n = "";
	this->name = AtomFind (n);
	return *this;
}


