/*
 *	regdata.cxx
 *
 *	Smaky Registry / INTERNAL.
 *
 *	The Registry Data class makes dynamic data publishing
 *	possible in any registry sub-tree.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#include "regdata.h"
#include "registry.h"

#include <c++/mon.h>

RegData::IDList*	RegData::id_list = 0;

/*
 *	Base class does not allocate any special information.
 *
 *	The constructor should initialise the "class ID" field
 *	with the proper values.
 */

RegData::RegData ()
{
	this->class_id       = 0;
	this->startup_len    = 0;
	this->startup_data   = 0;
	this->startup_reg    = 0;
	this->startup_father = 0;
}

RegData::~RegData ()
{
	if (this->startup_data) {
		delete this->startup_data;
	}
}


/*
 *	The startup method should be overridden in order to add
 *	the desired initialisation (e.g. read information from the
 *	specified file).
 *
 *	Call the `super' method first.
 */

Bool
RegData::Startup (const char* data, Card32 len, Registry* reg, RegKey* father)
{
	if (this->startup_data) {
		delete this->startup_data;
	}
	
	if ( (len == 0)
	  || (data == 0) ) {
		this->startup_data = 0;
		this->startup_len  = 0;
		return TRUE;
	}
	
	char* copy = new char[len];
	
	if (copy == 0) return FALSE;
	
	memcpy (copy, data, len);
	
	this->startup_data   = copy;
	this->startup_len    = len;
	this->startup_reg    = reg;
	this->startup_father = father;
	
	return TRUE;
}


/*
 *	Clone this data object: this allocates storage for the
 *	data object and the clones it (using the overridden
 *	methods).
 */

RegData*
RegData::Clone ()
{
	RegData* data = this->AllocFromClassID (this->class_id);
	
	if (data) {
		data->Startup (this->startup_data, this->startup_len,
					   this->startup_reg, this->startup_father);
	}
	
	return data;
}


/*
 *	Dynamically find the proper class allocation function and
 *	call it in order to create the instance of the specified
 *	type.
 */

RegData*
RegData::AllocFromClassID (Card32 id)
{
	IDList* elem = RegData::id_list;
	
	while (elem) {
		
		if (elem->id == id) {
			return (elem->func) ();
		}

		elem = elem->next;
	}
	
	return 0;
}


/*
 *	Each data publishing class should call this member function
 *	in order to identify itself and provide a proper allocator.
 */

void
RegData::RegisterClassID (Card32 id, RegData* (*func)())
{
	AfText ("Register Class ID : id=");
	AfX8 (id);
	AfCR ();
	
	IDList* elem = new IDList;
	
	if (elem) {
		elem->next = RegData::id_list;
		elem->id   = id;
		elem->func = func;
		RegData::id_list = elem;
	}
}

