/*
 *	reghive.cxx
 *
 *	Smaky Registry / INTERNAL.
 *
 *	The registry hive class provides methods to handle leaves
 *	of the tree; these are collected in chunks named hives.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#include "smreg.h"
#include "registry.h"
#include "regkey.h"
#include "reghive.h"
#include "regdata.h"

/*****************************************************************************/

#define	MAKE_VALUE(x)	((RegValue*)((Card32)(x)+base))
#define	MAKE_STRING(x)	((char*)((Card32)(x)+base))

#define	PADDING			(0x0fL)
#define	PADDING_MASK	((Card32)(~(PADDING)))
#define	MIN_MEMORY		1000

/*****************************************************************************/

/*
 *	When a hive is created, the registry will be notified of its
 *	existence.
 *
 *	A hive is a collection of linked memory chunks containing the
 *	data held by the registry.
 */

RegHive::RegHive (Registry* reg)
{
	this->registry = reg;
	this->free     = 0;
	this->head     = 0;
	this->tail     = 0;
	this->changed  = FALSE;

	this->registry->NotifyNewHive (this);
}

RegHive::~RegHive ()
{
	this->registry->NotifyDeleteHive (this);
	
	Memory* mem  = this->head;
	Memory* next = 0;

	while (mem) {
		next = mem->next;
		delete mem;
		mem  = next;
	}
}

/*****************************************************************************/

/*
 *	Internal method used to allocate memory chunks from a
 *	pre-allocated pool of chunks.
 */

void*
RegHive::AllocMemory (Card32 size)
{
	size += sizeof (Memory*) + PADDING;
	size &= ~PADDING;
	
	Memory* mem = this->head;
	Card32  len = (size < MIN_MEMORY) ? MIN_MEMORY : size;

	while (mem) {
		if (mem->len >= size) {
			Memory** ptr = (Memory**)(mem->ptr);
			*ptr++ = mem;
			mem->len -= size;
			mem->ptr += size;
			return ptr;
		}
		
		mem = mem->next;
	}
	
	len += sizeof (Memory);
	mem  = (Memory*) new char[len];

	if (mem) {
		mem->next = this->head;
		mem->prev = 0;
		(mem->next ? mem->next->prev : this->tail) = mem;
		(mem->prev ? mem->prev->next : this->head) = mem;
		mem->len  = len - sizeof (Memory) - size;
		mem->ptr  = (char*)(mem+1) + size;
		return mem+1;
	}

	return 0;
}

void
RegHive::FreeMemory (void* ptr, Card32 size)
{
	size += sizeof (Memory*) + PADDING;
	size &= ~PADDING;
	
	Memory** mptr = (Memory**)(ptr);
	Memory*  mem  = *(--mptr);
	
	if ((char*)(mptr)+size == mem->ptr) {
		mem->ptr -= size;
		mem->len += size;
	}
}

void
RegHive::CopyMemory (const void* src, void* dst, Card32 size)
{
	memcpy (dst, src, size);
}

/*****************************************************************************/

/*
 *	Parse the data (coming from an image of a hive file) and build an
 *	internal representation :
 *
 *	1. Translate offsets into pointers (relocation).
 *	2. Translate strings into atoms.
 *	
 *	The data from this hive is not mounted into the registry yet.
 */

int
RegHive::RelocateData (void* data)
{
	Card32    base   = (Card32)(data);
	RegValue* value  = (RegValue*)(data);
	int       count  = 0;
	RegValue* next   = 0;
	RegValue* parent = 0;
	RegValue* child  = 0;
	
	//	Check to see if the first value in the hive is matching the very
	//	special one we expect to find there.
	
	if ( (value->type != REG_MAGIC_TYPE)
	  || (value->data.cross.prev != 0) ) {
		return REG_ERR_CORRUPT;
	}
	
	
	//	This is an iterative, depth first, exploration of the full hive. We
	//	reverse pointers as we move further down the tree : this way we can
	//	come up easily, without having to use any stack.
	
	while (value) {
		
		switch (X_TYPE (value->type)) {
			
			//	Handle a link to another level. Follow the link and maintain
			//	information so we can backtrack later on.
			
			case REG_TYP_X_LINK:
				
				child = 0;
				
				if (value->data.link.value) {
					child = MAKE_VALUE (value->data.link.value);
				}
				
				if (X_TYPE (child->type) != REG_TYP_X_CROSS) {
					return REG_ERR_CORRUPT;
				}
				
				value->data.link.value = child;
				this->RelocateData (child);
				break;

			//	Cross reference : this will lead us to the next chunk of
			//	values (when this one is finished)
			
			case REG_TYP_X_CROSS:
				
				if (value->data.cross.next) {
					value->data.cross.next = MAKE_VALUE (value->data.cross.next);
				}
				
				if (value->data.cross.prev) {
					value->data.cross.prev = MAKE_VALUE (value->data.cross.prev);
				}
				
				if (value == child) {
					
					//	Make the child point to the grand parent so that we
					//	can restore the parent's parent later on.
					
					value->data.cross.prev  = parent->data.link.value;
					parent->data.link.value = child;
				}
				
				count = value->name.count;
				next  = value->data.cross.next;
				break;
			
			//	Dynamic data : this will have to relocate the "initialisation"
			//	part of the record.
			
			case REG_TYP_X_DDATA:
				value->data.dynamic.data = (RegData*) MAKE_STRING (value->data.dynamic.data);
				break;

			//	Arrays, string definitions and token definitions will need
			//	to be transformed into pointers.
			
			default:
				if ((value->type & REG_TYP_ARRAY) == 0) {
					break;
				}
				
				//	fall through...
			
			case REG_TYP_X_STRING:
				value->data.array.data = MAKE_STRING (value->data.array.data);
				break;
		}
		
		if (count == 0) {
			return REG_ERR_CORRUPT;
		}
		
again:	//	If we just rewinded from the child to the parent, continue
		//	here with the parent...
		
		//	The value's name might need to be atomified, if it is defined by
		//	a conventional string.
		
		switch (X_NAME (value->type)) {
			
			case REG_NAM_ATOM:
			case REG_NAM_STRING:
				
				//	Relocate the string (points into string table if the hive has
				//	been properly saved), then make an atom of it.
				
				{
					const char* string = MAKE_STRING (value->name.string);
					value->name.atom   = this->registry->MakeAtom (string);
					break;
				}
			
			default:
				break;
		}
		
		count--;
		value++;

		if (count == 0) {
			value = next;
			next  = 0;
		}
		
		if ( (value == 0)
		  && (parent != 0) ) {
			
			value  = parent;					//	restore chunk
			count  = value->data.link.temp;		//	restore remaining in chunk
			child  = parent->data.link.value;	//	head of chunk we are leaving
			parent = child->data.cross.prev;	//	parent of chunk we are restoring
			
			child->data.cross.prev = 0;			//	clean up start of chunk we left
			value->data.link.temp  = 0;			//	clean up temporary

			goto again;
		}
	}
	
	//	First pass finished. We now have completely relocated the tree.
	
	return REG_ERR_OK;
}

/*****************************************************************************/

/*
 *	Load a hive into the registry. This will simply add each value
 *	into one of the keys (eventually creating new keys as needed).
 */

int
RegHive::LoadData (void* data, RegKey* root)
{
	RegValue* value = (RegValue*)(data);
	RegValue* next  = 0;
	Card32      count = 0;
	
	if (root == 0) {
		root = registry->Root ();
	}
	
	while (value) {
		
		switch (X_TYPE (value->type)) {
			
			//	Remember which will be the next chunk with data in it.
			//	Use it as soon as this one gets empty.
			
			case REG_TYP_X_CROSS:
				next  = value->data.cross.next;
				count = value->name.count;
				break;
			
			//	Ignore following cases :
			
			case REG_TYP_X_LEVEL:
			case REG_TYP_X_STRING:
				break;
			
			//	If we have found a link, we insert it like any other
			//	value. This will be handled in a special way (and we
			//	will be called recursively).
			
			case REG_TYP_X_LINK:
			case REG_TYP_X_DDATA:
				
				//	Fall through...
				
			default:
				root->Insert (*value, this);
				break;
		}

		count--;
		value++;
		
		if (count == 0) {
			value = next;
			next  = 0;
		}
	}

	return REG_ERR_OK;
}

/*****************************************************************************/

/*
 *	Import data from a binary image. The data will be modified by this
 *	method and should be disposed of afterwards.
 */

int
RegHive::ImportData (void* data, RegKey* root)
{
	int err = this->RelocateData (data);
	if (err) return err;
	
	err = this->LoadData (data, root);
	if (err) return err;
	
	this->ResetChanges ();
	
	return 0;
}

/*****************************************************************************/

/*
 *	Create a new value in the hive. This will allocate memory to
 *	store the value; if it is an array, even more memory has to be
 *	allocated.
 */

int
RegHive::CreateValue (RegValue*& value, const RegValue& model, RegKey* parent)
{
	int    err  = 0;
	void*  data = 0;
	Card32 num  = model.data.array.num;
	
	this->Changed ();
	
	if (this->free == 0) {
		err = this->CreateFreeValues ();
		if (err) return err;
	}
	
	if ((model.type & REG_TYP_ARRAY) && (num)) {
		err = this->CloneData (data, model);
		if (err) return err;
	}
	
	value       = this->free;
	this->free  = value->data.cross.next;
	
	value->type = model.type;
	value->name = model.name;
	value->data = model.data;
	
	if (model.type & REG_TYP_ARRAY) {
		value->data.array.data = data;
		value->data.array.num  = num;
	}

	//	If a dynamic data instance has to be allocated, this will be
	//	done here: use the class ID and the startup parameter to
	//	allocate and initialise the registry data instance.
	
	if ( (X_TYPE (model.type) == REG_TYP_X_DDATA)
	  && (model.data.dynamic.temp != 0) ) {
		
		Card32 len  = ((Card32)(model.data.dynamic.temp));
		Card32 id   = ((Card32*)(model.data.dynamic.data))[0];
		char*  data = ((char*)(model.data.dynamic.data)) + 4;
		
		RegData* dynamic = RegData::AllocFromClassID (id);
		
		if (dynamic) {
			dynamic->Startup (data, len, this->registry, parent);
		}
		
		value->data.dynamic.data = dynamic;
		value->data.dynamic.temp = 0x00000000;
		
		return (dynamic == 0) ? REG_ERR_FULL : 0;
	}
	
	//	If a link is being created, replace it by a new key and then
	//	build recursively the values which belong to that key !
	
	if (X_TYPE (model.type) == REG_TYP_X_LINK) {
		
		value->type &= 0xFFFFFF00;
		value->type |= REG_TYP_X_LEVEL;
		
		value->data.level.key  = new DataRegKey (this->registry, parent);
		value->data.level.temp = 0;
		
		err = (value->data.level.key)
			? this->LoadData (model.data.link.value, value->data.level.key)
			: REG_ERR_FULL;
		
		if (err) {
			value->data.cross.next = this->free;
			this->free = value;
			return err;
		}
	}
	
	return REG_ERR_OK;
}


/*
 *	The value will have to be replaced by the model.
 */

int
RegHive::RecycleValue (RegValue*& value, const RegValue& model, RegKey* parent)
{
	this->Changed ();
	
	int err = 0;
	
	if (value->type & REG_TYP_ARRAY) {
		
		Card32 len = this->GetDataBytes (value);
		
		if (value->data.array.data) {
			this->FreeMemory (value->data.array.data, len);
		}
		
		value->data.array.data = 0;
		value->data.array.num  = 0;
	}
	
	
	//	Dynamic data has to be freed and will be rebuilt from scratch when
	//	needed.
	
	if (X_TYPE (value->type) == REG_TYP_X_DDATA) {
		delete value->data.dynamic.data;
	}
	
	if (X_TYPE (model.type) == REG_TYP_X_DDATA) {
		
		value->type = model.type;
		value->name = model.name;
		
		value->data.dynamic.data = model.data.dynamic.data->Clone ();

		return (value->data.dynamic.data) ? 0 : REG_ERR_FULL;
	}
	
	
	//	Recycling a RegKey is not that simple. If one existed before recycling,
	//	it might have to be emptied (if it will cease to live). If one did not,
	//	it will have to be created.
	
	if ( (X_TYPE (value->type) == REG_TYP_X_LEVEL)
	  && (X_TYPE (model.type) != REG_TYP_X_LEVEL) ) {
		delete value->data.level.key;
	}
	
	if ( (X_TYPE (value->type) != REG_TYP_X_LEVEL)
	  && (X_TYPE (model.type) == REG_TYP_X_LEVEL) ) {
		value->data.level.key  = 0;
		value->data.level.temp = 0;
	}
	
	if (X_TYPE (model.type) == REG_TYP_X_LEVEL) {
		
		value->type = model.type;
		value->name = model.name;
		
		if (value->data.level.key == 0) {
			value->data.level.key = new DataRegKey (this->registry, parent);
			if (value->data.level.key == 0) {
				return REG_ERR_FULL;
			}
		}
		
		Card32   index   = 0;
		RegKey*  key     = model.data.level.key;
		RegKey*  root    = value->data.level.key;
		void*    dispose = 0;
		RegValue data;
		
		while (key->Find (index, data, 0, dispose)) {
			err = root->Insert (data, this);
			if (dispose) key->Dispose (dispose);
			if (err) return err;
		}

		return 0;
	}

	value[0] = model;
	
	if ( (model.type & REG_TYP_ARRAY)
	  && (model.data.array.num) ) {
		err = this->CloneData (value[0].data.array.data, model);
		if (err) return err;
	}
	
	return REG_ERR_OK;
}

int
RegHive::DisposeValue (RegValue*& value)
{
	this->Changed ();
	
	if (X_TYPE (value->type) == REG_TYP_X_LEVEL) {
		delete value->data.level.key;
	}
	
	if (X_TYPE (value->type) == REG_TYP_X_DDATA) {
		delete value->data.dynamic.data;
	}
	
	return REG_ERR_OK;
}

/*****************************************************************************/

/*
 *	Make new free values which can be used to allocate values
 *	simply by picking in a list.
 */

int
RegHive::CreateFreeValues ()
{
	Card32 num = 20;
	Card32 len = sizeof (RegValue) * num;
	this->free = (RegValue*) this->AllocMemory (len);
	
	if (this->free == 0) {
		return REG_ERR_FULL;
	}
	
	RegValue* head = this->free;
	
	while (--num) {
		head->data.cross.next = head+1;
		head->data.cross.prev = 0;
		head++;
	}

	head->data.cross.next = 0;
	head->data.cross.prev = 0;

	return REG_ERR_OK;
}


/*
 *	Clone data and allocate memory for it. This has to be reworked,
 *	since we don't ever free the memory !
 *
 *	Call this only for an array.
 */

int
RegHive::CloneData (void*& data, const RegValue& model)
{
	Card32 len;
	
	len  = this->GetDataBytes (&model);
	data = this->AllocMemory (len);
	
	if (data == 0) {
		return REG_ERR_FULL;
	}
	
	this->CopyMemory (model.data.array.data, data, len);

	return REG_ERR_OK;
}


/*
 *	Return the size of the array data in bytes. Return 0 if it
 *	is not an array.
 */

Card32
RegHive::GetDataBytes (const RegValue* model)
{
	if ((model->type & REG_TYP_ARRAY) == 0) {
		return 0;
	}
	
	Card32 len = model->data.array.num;
	
	switch (X_TYPE (model->type)) {
		
		case REG_TYP_BOOLEAN:
			len = (len + 7) / 8;
			break;
		
		case REG_TYP_UNICODE:
			len *= 2;
			break;
		
		case REG_TYP_INTEGER:
		case REG_TYP_CARDINAL:
			len *= 4;
			break;
		
		case REG_TYP_REAL:
			len *= 8;
			break;
	}
	
	return len;
}

/*****************************************************************************/

/*
 *	The stream is allocated by the hive. You will have to call "SetWriter"
 *	on it.
 */

int
RegHive::SaveBegin (HiveStream*& stream)
{
	stream = (HiveStream*)(this->AllocMemory (sizeof (HiveStream)));
	
	if (stream == 0) {
		return REG_ERR_FULL;
	}
	
	stream->pass   = 1;
	stream->datnum = 0;
	stream->datlen = 0;
	stream->outlen = 0;
	stream->strlen = 0;
	stream->datmax = 0;
	stream->strmax = 0;
	stream->eofpos = 0;
	stream->writer = 0;
	stream->arg    = 0;
	return 0;
}

/*
 *	You should call this method as long as "SaveEnd" says to keep on
 *	providing data. There are several passes :
 *
 *	Pass 1:	Count elements and find out how much string memory
 *			there will be, and how much outline data we will have
 *			to store.
 *
 *	Pass 2:	Ignore.
 *	
 *	Pass 3:	Store the records with the proper offsets to the string
 *			and outline data.
 *
 *	Pass 4:	Store the outline data.
 *
 *	Pass 5:	Store the strings.
 *
 *	Pass 6: Save the outline data.
 */

int
RegHive::SaveData (HiveStream* stream, RegValue* value)
{
	if (stream == 0) {
		return REG_ERR_ILLOP;
	}
	
	Card32 n_type = (value->type) & 0xFF000000;
	Card32 a_type = (value->type) & 0x0000FF00;
	Card32 d_type = (value->type) & 0x000000FF;

	RegValue  data;
	HiveStream* deep;
	int         err;
	
	switch (stream->pass) {
		
		case 1:
			if (n_type != REG_NAM_TOKEN) {
				stream->strlen += strlen (value->name.atom)+1;
			}
			if (a_type & REG_TYP_ARRAY) {
				stream->datlen += this->GetDataBytes (value);
			}
			
			if (d_type == REG_TYP_X_DDATA) {
				stream->datlen += value->data.dynamic.data->GetStartupLength ();
				stream->datlen += 4;
			}
			
			stream->datnum++;
			
			if (d_type == REG_TYP_X_LEVEL) {
				
				err = this->SaveBegin (deep);
				if (err) return err;
				
				deep->writer = stream->writer;
				deep->arg    = stream->arg;
				
				//	Remember about the link and compute the whole sub-tree's
				//	size. This is recursive !
				
				value->data.level.temp  = (Card32)(deep);
				err = value->data.level.key->SaveHiveTree (this, deep);
				if (err) return err;
				
				//	The output stream will hold the whole sub-tree too. It will
				//	be handled in a very special way (stored after the inlined
				//	data and strings).
				
				stream->outlen += deep->eofpos;
			}
			
			return 0;
		
		case 2:
			return 0;

		case 3:
			data = value[0];
			
			if (n_type != REG_NAM_TOKEN) {
				Card32 len = strlen (value->name.atom)+1;
				data.name.count = stream->strlen + stream->offset + stream->datmax;
				stream->strlen += len;
			}
			if (a_type & REG_TYP_ARRAY) {
				Card32 len = this->GetDataBytes (value);
				data.data.array.data = (void*)(stream->datlen + stream->offset);
				stream->datlen += len;
			}
			
			if (d_type == REG_TYP_X_DDATA) {
				Card32 len = value->data.dynamic.data->GetStartupLength ();
				data.data.dynamic.data = (RegData*)(stream->datlen + stream->offset);
				data.data.dynamic.temp = (Card32)(len+4);
				stream->datlen += len;
				stream->datlen += 4;
			}
			
			if (d_type == REG_TYP_X_LEVEL) {
				deep = (HiveStream*)(value->data.link.temp);
				data.type &= 0xFFFFFF00;
				data.type |= REG_TYP_X_LINK;
				data.data.link.value = (RegValue*)(stream->eofpos);
				data.data.link.temp  = 0;
				stream->eofpos += deep->eofpos;			//	the end-of-file will move
			}
			
			return stream->Write (&data, sizeof (data));
		
		case 4:
			if (a_type & REG_TYP_ARRAY) {
				Card32 len = this->GetDataBytes (value);
				return stream->Write (value->data.array.data, len);
			}
			
			if (d_type == REG_TYP_X_DDATA) {
				Card32      len  = value->data.dynamic.data->GetStartupLength ();
				Card32      id   = value->data.dynamic.data->GetClassID ();
				const char* init = value->data.dynamic.data->GetStartupData ();
				stream->Write (&id, 4);
				return stream->Write (init, len);
			}
			
			return 0;
		
		case 5:
			if (n_type != REG_NAM_TOKEN) {
				Card32 len = strlen (value->name.atom)+1;
				return stream->Write (value->name.atom, len);
			}
			return 0;

		case 6:
			if (d_type == REG_TYP_X_LEVEL) {
				deep = (HiveStream*)(value->data.link.temp);
				while (deep && deep->pass) {
					err = value->data.level.key->SaveHiveTree (this, deep);
					if (err) return err;
				}
			}
			return 0;
	}
	
	return REG_ERR_ILLOP;
}


/*
 *	Call this method whenever you have finished with the whole
 *	bunch of values for that sub-tree. If "again" is set, you
 *	are required to start all over again (call N times SaveData
 *	and call SaveEnd again).
 *	
 *	Pass 1:	Remember how many elements have been seen.
 *	Pass 2: Just save the starting mark.
 *	Pass 3:	Go to pass 4.
 *	Pass 4:	Go to pass 5.
 *	Pass 5:	Output padding and eof for this sub-tree.
 *	Pass 6: End.
 *
 *	This stream is automatically freed.
 */

int
RegHive::SaveEnd (HiveStream*& stream, Bool& again)
{
	if (stream == 0) {
		again = FALSE;
		return REG_ERR_ILLOP;
	}

	const char zero[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	RegValue value;
	Card32     pos;
	
	switch (stream->pass) {
		
		case 1:
			stream->datmax = stream->datlen;
			stream->strmax = stream->strlen;
			stream->datlen = 0;
			stream->strlen = 0;
			stream->offset = (stream->datnum + 1) * sizeof (RegValue);
			
			pos  = stream->offset;
			pos += stream->datmax;
			pos += stream->strmax;
			pos += stream->outlen;
			pos += sizeof (value);			//	final record (end of file marker)
			pos += PADDING;
			pos &= PADDING_MASK;
			
			stream->eofpos = pos;
			
			again = TRUE;
			stream->pass++;
			return 0;

		case 2:
			pos  = stream->offset;
			pos += stream->datmax;
			pos += stream->strmax;
			pos += PADDING;
			pos &= PADDING_MASK;
			
			stream->eofpos = pos;		//	don't count the linked data
			
			value.type            = REG_MAGIC_TYPE;
			value.data.cross.next = (RegValue*)(pos);
			value.data.cross.prev = 0;
			value.name.count      = stream->datnum + 1;
			
			stream->Write (&value, sizeof (value));

			stream->eofpos += sizeof (value);
			
		case 3:
		case 4:
			again = TRUE;
			
			stream->pass++;
			return 0;
			
		case 5:
			pos  = ~ (stream->offset + stream->datmax + stream->strmax);
			pos += 1;
			pos &= PADDING;

			if (pos) stream->Write (zero, pos);
			
			value.type            = REG_TYP_X_CROSS;
			value.data.cross.next = 0;
			value.data.cross.prev = 0;
			value.name.count      = 1;

			stream->Write (&value, sizeof (value));

			again = TRUE;
			
			stream->pass++;
			return 0;
			
		case 6:
			break;
	}
	
	again = FALSE;
	
	this->FreeMemory (stream, sizeof (HiveStream));
	this->ResetChanges ();
	
	stream = 0;
	return 0;
}
