/*
 *	reghive.h
 *
 *	Smaky Registry / INTERNAL.
 *
 *	The registry hive class provides methods to handle leaves
 *	of the tree; these are collected in chunks named hives.
 *
 *	(C) Copyright 1996, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#ifndef _REGHIVE_H_
#define	_REGHIVE_H_

#include "smreg.h"

class RegHive;
class RegKey;
class Registry;

/*****************************************************************************/

/*
 *	Utility class used to store data from the hive. You should
 *	provide a writer...
 */

class HiveStream
{
	friend RegHive;
	
protected:
	int			pass;
	Card32		datnum;
	Card32		datlen;
	Card32		strlen;
	Card32		outlen;
	Card32		datmax;
	Card32		strmax;
	Card32		offset;
	Card32		eofpos;
	
	int			(*writer)(const void*, Card32, void*);
	void*		arg;
	
public:
	void SetWriter (int (*wfunc)(const void* data, Card32 len, void* arg), void* arg)
	{
		this->writer = wfunc; this->arg = arg;
	}
	
	int Write (const void* data, Card32 len)
	{
		return (this->writer == 0)
			 ? REG_ERR_ILLOP
			 : (this->writer)(data, len, this->arg);
	}
	
	Bool Saving () const { return pass ? TRUE : FALSE; }
};

/*****************************************************************************/

class RegHive
{
	struct Memory
	{
		Memory*	next;
		Memory*	prev;
		Card32	len;
		char*	ptr;
	};

private:
	Registry*	registry;
	RegValue*	free;
	
	Memory*		head;
	Memory*		tail;

	Bool		changed;

protected:
	int CreateFreeValues ();
	int CloneData (void*& data, const RegValue& model);

	void* AllocMemory (Card32 size);
	void  FreeMemory (void* ptr, Card32 size);
	void  CopyMemory (const void* src, void* dst, Card32 size);
	
	int RelocateData (void* data);
	int LoadData (void* data, RegKey* root = 0);

public:
	RegHive (Registry* reg);
	~RegHive ();
	
	int ImportData (void* data, RegKey* root = 0);
	
	int SaveBegin (HiveStream*& stream);
	int SaveData (HiveStream* stream, RegValue* value);
	int SaveEnd (HiveStream*& stream, Bool& again);
	
	int CreateValue (RegValue*& value, const RegValue& model, RegKey* parent);
	int RecycleValue (RegValue*& value, const RegValue& model, RegKey* parent);
	int DisposeValue (RegValue*& value);

	Bool HasChanged () const	{ return this->changed; }
	void ResetChanges ()		{ this->changed = FALSE; }
	void Changed ()				{ this->changed = TRUE; }

	static Card32 GetDataBytes (const RegValue* value);
};

/*****************************************************************************/

#endif /* _REGHIVE_H_ */
