/*
 *	registry.cpp
 *
 *	Smaky Registry / INTERNAL.
 *
 *	The registry class provides useful methods for accessing some
 *	keys and handling the atom data base.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#include <stdlib.h>
#include <stdio.h>

#include "registry.h"
#include "regkey.h"
#include "atom.h"

#define	ARRAY_GROWTH	100

Registry::Registry ()
{
	this->root        = new DataRegKey (this, 0);
	
	this->enter_array = 0;
	this->enter_num   = 0;
	this->enter_max   = 0;
	this->id_of_next  = 0;
	this->id_wrapped  = FALSE;
	
	this->hive_array  = 0;
	this->hive_num    = 0;
	this->hive_max    = 0;
}

Registry::~Registry ()
{
	delete this->root;
	delete this->enter_array;
	delete this->hive_array;
}

Bool
Registry::GrowEnterArray ()
{
	Card32   index = 0;
	Card32   max   = this->enter_max + ARRAY_GROWTH;
	KeyInfo* array = new KeyInfo [max];
	
	if (array == 0) {
		return FALSE;
	}

	for (index = 0; index < max; index++) {
		array[index].key = 0;
		array[index].id  = 0;
	}

	if (this->enter_array) {
		for (index = 0; index < this->enter_num; index++) {
			array[index] = this->enter_array[index];
		}
		delete this->enter_array;
	}

	this->enter_array = array;
	this->enter_max   = max;

	return TRUE;
}

Bool
Registry::GrowHiveArray ()
{
	Card32    index = 0;
	Card32    max   = this->hive_max + ARRAY_GROWTH;
	HiveInfo* array = new HiveInfo [max];
	
	if (array == 0) {
		return FALSE;
	}
	
	for (index = 0; index < max; index++) {
		array[index].hive = 0;
		array[index].id   = 0;
	}

	if (this->hive_array) {
		for (index = 0; index < this->hive_num; index++) {
			array[index] = this->hive_array[index];
		}
		delete this->hive_array;
	}

	this->hive_array = array;
	this->hive_max   = max;

	return TRUE;
}

Card32
Registry::NewID ()
{
	this->id_of_next++;
	
	if (this->id_of_next == 0) {
		this->id_wrapped = TRUE;
	}
	
	if (this->id_wrapped) {
		
		Bool match = TRUE;
		
		while (match) {
			
			do {
				this->id_of_next++;
			} while (this->id_of_next == 0);
			
			match = FALSE;
			
			for (Card32 index = 0; index < this->enter_num; index++) {
				if (this->enter_array[index].id == this->id_of_next) {
					match = TRUE;
					break;
				}
			}
		}
	}
	
	return this->id_of_next;
}


const char*
Registry::MakeAtom (const char* name)
{
	Atom atom (name);
	return atom;
}

RegKey*
Registry::Root () const
{
	return this->root;
}

/*****************************************************************************/

Bool
Registry::NotifyEnter (RegKey* key)
{
	if (this->enter_num == this->enter_max) {
		if (this->GrowEnterArray () == FALSE) {
			return FALSE;
		}
	}

	this->enter_array[this->enter_num].key = key;
	this->enter_array[this->enter_num].id  = this->NewID ();
	
	key->SetID (this->enter_array[this->enter_num].id);
	
	this->enter_num++;
	
	return TRUE;
}

Bool
Registry::NotifyRelease (RegKey* key)
{
	if (this->enter_num) {
		for (Card32 index = 0; index < this->enter_num; index++) {
			if (this->enter_array[index].key == key) {
				this->enter_array[index].key = 0;
				this->enter_array[index].id  = 0;
				this->enter_num--;
				this->enter_array[index] = this->enter_array[this->enter_num];
				return TRUE;
			}
		}
	}

	return FALSE;
}

RegKey*
Registry::FindKeyFromID (Card32 id)
{
	if (this->enter_num) {
		for (Card32 index = 0; index < this->enter_num; index++) {
			if (this->enter_array[index].id == id) {
				return this->enter_array[index].key;
			}
		}
	}

	return 0;
}

Card32
Registry::KeyID (RegKey* key)
{
	return (key) ? key->GetID () : 0;
}


/*****************************************************************************/

Bool
Registry::NotifyNewHive (RegHive* hive)
{
	if (this->hive_num == this->hive_max) {
		if (this->GrowHiveArray () == FALSE) {
			return FALSE;
		}
	}

	this->hive_array[this->hive_num].hive = hive;
	this->hive_array[this->hive_num].id   = 0;
	this->hive_num++;
	
	return TRUE;
}

Bool
Registry::NotifyDeleteHive (RegHive* hive)
{
	if (this->hive_num) {
		for (Card32 index = 0; index < this->hive_num; index++) {
			if (this->hive_array[index].hive == hive) {
				this->hive_array[index].hive = 0;
				this->hive_array[index].id   = 0;
				this->hive_num--;
				this->hive_array[index] = this->hive_array[this->hive_num];
				return TRUE;
			}
		}
	}

	return FALSE;
}

/*****************************************************************************/

void*
operator new (size_t size)
{
	return malloc (size);
}

void
operator delete (void* ptr)
{
	free (ptr);
}

