/*
 *	regkey.cpp
 *
 *	Smaky Registry / INTERNAL.
 *
 *	The registry key class provides methods to handle nodes
 *	of the tree.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */


#include "smreg.h"
#include "registry.h"
#include "regkey.h"
#include "reghive.h"

/*****************************************************************************/
/*																			 */
/*								INLINE FUNCTIONS							 */
/*																			 */
/*****************************************************************************/

/*
 *	Check to see if the name is already in compact form (either
 *	an atom or a token). If it is still a string, make an atom
 *	out of it.
 */

inline void
RegKey::Atomify (RegValue& value)
{
	if (X_NAME (value.type) == REG_NAM_STRING) {
		value.name.atom = this->registry->MakeAtom (value.name.string);
		value.type     += (Card32)(REG_NAM_ATOM) - (Card32)(REG_NAM_STRING);
	}
}

/*
 *	Compute the hash value of a tokenised or an atomified name.
 *	This is fast and straightforward.
 */

inline Card32
DataRegKey::HashName (RegValue& value)
{
	return value.name.token % this->h_num;
}

/*****************************************************************************/

RegKey::RegKey (Registry* reg, RegKey* father)
{
	this->registry = reg;
	this->enter    = 0;
	this->parent   = father;
	this->child    = 0;
	this->pending  = 0;
	this->id       = 0;
	this->hive     = 0;
}

RegKey::~RegKey ()
{
	while (this->enter) {
		
		//	Pretend we have no children. How could we possibly remove
		//	them, anyway ?
		
		this->child = 0;
		this->Release ();
	}
}

/*****************************************************************************/

DataRegKey::DataRegKey (Registry* reg, RegKey* father)
					 : RegKey (reg, father)
{
	this->h_num    = H_NUM;
	this->h_heap   = 0;
	this->count    = 0;
	
	for (Card32 i = 0; i < H_NUM; i++) {
		for (Card32 j = 0; j < HASH_L1; j++) {
			this->h_table[i].data[j].value = 0;
			this->h_table[i].data[j].hive  = 0;
		}
		this->h_table[i].next = 0;
	}
}

DataRegKey::~DataRegKey ()
{
	Hash*  iter  = this->h_table;
	Card32 num   = this->h_num;
	
	while (num--) {
		Hash* hash = iter;
		while (hash) {
			for (int i = 0; i < HASH_L1; i++) {
				if (hash->data[i].value) {
					if (hash->data[i].value) {
						this->KillEntry (hash, i);
						i--;
					}
				}
			}
			hash = hash->next;
		}
		iter++;
	}

	HashHeap* heap = this->h_heap;
	HashHeap* next = 0;
	
	while (heap) {
		next = heap->next;
		delete heap;
		heap = next;
	}
}

DataRegKey::Hash*
DataRegKey::NewHash ()
{
	if ( (this->h_heap == 0)
	  || (this->h_heap->index == H_HEAP) ) {
		
		HashHeap* heap = new HashHeap;
		if (heap == 0) return 0;
		
		heap->next   = this->h_heap;
		heap->index  = 0;
		this->h_heap = heap;
	}

	Hash* hash = & this->h_heap->heap[this->h_heap->index++];
	
	for (int index = 0; index < HASH_L1; index++) {
		hash->data[index].value = 0;
		hash->data[index].hive  = 0;
	}
	
	hash->next = 0;

	return hash;
}

/*****************************************************************************/

/*
 *	Entering a key will delay the parent's release until the
 *	children have all been released.
 */

Bool
RegKey::Enter ()
{
	if ( (this->enter == 0)
	  && (this->parent) ) {
		this->parent->child++;
	}

	if ( (this->enter == 1)
	  && (this->pending) ) {
		
		//	We badly wanted to be released but could not since some
		//	children still were entered. Forget about that release
		//	and stay with a single enter.
		
		this->pending = 0;
		return TRUE;
	}
	
	if (this->enter == 0) {
		if (this->HandleEnter () == FALSE) {
			return FALSE;
		}
	}
	
	this->enter++;
	
	return TRUE;
}


/*
 *	In order to release a key, all its children must have been
 *	released. If this is not the case yet, remember that we have
 *	one more pending release...
 */

Bool
RegKey::Release ()
{
	if (this->enter == 0) {
		return FALSE;
	}
	
	if ( (this->enter == 1)
	  && (this->child) ) {
		
		//	Last enter should be released... Yet we cannot do it,
		//	since children are still entered.
		
		this->pending++;
		return TRUE;
	}
	
	if (this->enter == 1) {
		if (this->HandleRelease () == FALSE) {
			return FALSE;
		}
	}
	
	this->enter--;

	if ( (this->enter == 0)
	  && (this->parent)
	  && (this->parent->enter)
	  && (this->parent->child-- == 1)
	  && (this->parent->pending) ) {
		
		//	The parent has no more entered children and badly
		//	wanted to be released, so release it now !
		
		this->parent->pending = 0;
		return this->parent->Release ();
	}
	
	return TRUE;
}

/*****************************************************************************/

Bool
RegKey::HandleEnter ()
{
	return this->registry->NotifyEnter (this);
}

Bool
RegKey::HandleRelease ()
{
	return this->registry->NotifyRelease (this);
}

/*****************************************************************************/

/*
 *	Insert a value into a key. If the value already exists, simply
 *	replace the previous one. If not, allocate space for the value
 *	(this makes a deep copy) and insert it into the key.
 */

int
DataRegKey::Insert (RegValue& value, RegHive* hive)
{
	this->Atomify (value);
	
	Card32 hash = this->HashName (value);
	Hash*  find = this->h_table + hash;
	int    err  = 0;

	if (X_TYPE (value.type) == REG_TYP_X_LEVEL) {
		RegKey* key = value.data.level.key;
		if (key) {
			key->SetHive (hive);
		}
	}
	
	for (;;) {
		for (int index = 0; index < HASH_L1; index++) {
			
			//	If the value cannot be found, we have to insert a new
			//	one into the hive. Easy !
			
			if (find->data[index].value == 0) {
				err = hive->CreateValue (find->data[index].value, value, this);
				find->data[index].hive = (err) ? 0 : hive;
				this->count += (err) ? 0 : 1;
				return err;
			}
			
			//	If the value already exists, just recycle it.
			
			if (find->data[index].value->name.token == value.name.token) {
				err = (find->data[index].hive == hive)
					? hive->RecycleValue (find->data[index].value, value, this)
					: (find->data[index].hive->DisposeValue (find->data[index].value),
					   hive->CreateValue (find->data[index].value, value, this));
				find->data[index].hive = (err) ? 0 : hive;
				return err;
			}
		}
		
		if (find->next == 0) {
			find->next = this->NewHash ();
			if (find->next == 0) return REG_ERR_END;
		}
		
		find = find->next;
	}

	return REG_ERR_END;
}


/*
 *	Remove a value in a key, either by its value description
 *	(just uses the name) or by its name.
 */

Bool
DataRegKey::Remove (RegValue& value)
{
	this->Atomify (value);
	
	Card32 hash = this->HashName (value);
	Hash*  find = this->h_table + hash;
	
	while (find) {
		for (int index = 0; index < HASH_L1; index++) {
			if ( (find->data[index].value != 0)
			  && (find->data[index].value->name.token == value.name.token) ) {
				
				//	We do not accept to remove "entered" keys. Check before
				//	removing them.
				
				if (X_TYPE (find->data[index].value->type) == REG_TYP_X_LEVEL) {
					RegKey* key = find->data[index].value->data.level.key;
					if ( (key)
					  && (key->GetEnterCount () != 0) ) {
						return FALSE;
					}
				}
				
				this->KillEntry (find, index);
				return TRUE;
			}
		}
		find = find->next;
	}

	return FALSE;
}


/*
 *	Find quickly a value in a key. This uses hashing and atom
 *	matching. It should be as fast as lightning !
 */

Bool
DataRegKey::Find (RegValue& value, RegHive** hive_ptr, void*& dispose)
{
	dispose = 0;
	
	this->Atomify (value);
	
	Card32 hash = this->HashName (value);
	Hash*  find = this->h_table + hash;

	while (find) {
		for (int index = 0; index < HASH_L1; index++) {
			if ( (find->data[index].value != 0)
			  && (find->data[index].value->name.token == value.name.token) ) {
				
				value = * (find->data[index].value);
				
				if (hive_ptr) {
					hive_ptr[0] = find->data[index].hive;
				}
				
				return TRUE;
			}
		}
		find = find->next;
	}

	return FALSE;
}


Bool
DataRegKey::Find (Card32& iter, RegValue& value, RegHive** hive_ptr, void*& dispose)
{
	dispose = 0;
	
	Bool   ok    = FALSE;
	Card32 index = iter & 0xFFFF;
	Card32 hash  = (iter >> 16) & 0xFFFF;
	
	for (;;) {
		if (hash >= this->h_num) return FALSE;
		
		Hash*  h = & this->h_table[hash];
		Card32 i = index;
		
		while (i >= HASH_L1) {
			h  = h->next;
			i -= HASH_L1;
		}
		
		if (h == 0) {
			hash++;
			index = 0;
			continue;
		}

		index++;
		
		if (h->data[i].value) {
			value = h->data[i].value[0];
			
			if (hive_ptr) {
				hive_ptr[0] = h->data[i].hive;
			}
			
			ok    = TRUE;
			break;
		}
	}

	iter = (index) | (hash << 16);
	return ok;
}


/*
 *	When finding data in a key, some memory or other resources might be
 *	temporarily allocated. These will be disposed of as soon as the caller
 *	has copied it to its own data space.
 */

void
DataRegKey::Dispose (void* dispose)
{
}

/*****************************************************************************/

/*
 *	Save the full tree contained in a specified hive. This does just one
 *	single pass.
 */

int
DataRegKey::SaveHiveTree (RegHive* hive, HiveStream*& stream)
{
	Hash*  iter  = this->h_table;
	Card32 num   = this->h_num;
	Bool   again = FALSE;
	int    err   = 0;
	
	while (num--) {
		Hash* hash = iter;
		while (hash) {
			for (int i = 0; i < HASH_L1; i++) {
				if (hash->data[i].value) {
					if (hash->data[i].hive == hive) {
						err = hive->SaveData (stream, hash->data[i].value);
						if (err) return err;
					} else if (X_TYPE (hash->data[i].value->type) == REG_TYP_X_LEVEL) {
						hash->data[i].value->data.level.key->SaveHiveTree (hive, stream);
					}
				}
			}
			hash = hash->next;
		}
		iter++;
	}

	return hive->SaveEnd (stream, again);
}


/*
 *	Remove any values which belong to a specified hive.
 */

int
DataRegKey::KillHiveTree (RegHive* hive)
{
	Hash*  iter  = this->h_table;
	Card32 num   = this->h_num;
	
	while (num--) {
		Hash* hash = iter;
		while (hash) {
			for (int i = 0; i < HASH_L1; i++) {
				if (hash->data[i].value) {
					
					//	If there is a valid sub-tree, walk through and remove any
					//	references to the specified hive.
					
					if ( (X_TYPE (hash->data[i].value->type) == REG_TYP_X_LEVEL)
					  && (hash->data[i].value->data.level.key) ) {
						hash->data[i].value->data.level.key->KillHiveTree (hive);
					}
					
					//	If we have found an element stored in the matching hive, just
					//	remove the data. If that element is a sub-tree, kill it.
					
					if (hash->data[i].hive == hive) {
						this->KillEntry (hash, i);
						i--;
					}
				}
			}
			hash = hash->next;
		}
		iter++;
	}

	return REG_ERR_OK;
}

void
DataRegKey::KillEntry (Hash* h, int index)
{
	Value& v = h->data[index];
	
	v.hive->DisposeValue (v.value);
	
	v.hive  = 0;
	v.value = 0;
	
	this->count--;
	
	//	We don't tolerate holes in the list of elements pointed to by an entry
	//	in the hash table. Move the last element in order to fill the gap.
	
	index++;
	
	if (index == HASH_L1) {
		h     = h->next;
		index = 0;
	}
	
	Value* valid = 0;
	
	while (h) {
		for (int i = index; i < HASH_L1; i++) {
			if (h->data[i].value) {
				valid = & h->data[i];
			}
		}
		h = h->next;
	}
	
	//	There was an element after the one we deleted. Move it to fill the
	//	gap.
	
	if (valid) {
		v = valid[0];
		valid->value = 0;
		valid->hive  = 0;
	}
}

/*****************************************************************************/

/*
 *	The shadow key class just provides an "in between" class which can be
 *	used to create and remove dynamically complete sub-trees.
 *
 *	All the handling is done in the HandleEnter and HandleRelease methods,
 *	which either create or remove the shadowed key.
 */

ShadowRegKey::ShadowRegKey (Registry* reg, RegKey* father)
						 : RegKey (reg, father)
{
	this->shadow      = 0;
	this->helper      = 0;
	this->helper_info = 0;
}

ShadowRegKey::~ShadowRegKey ()
{
	if (this->helper) {
		delete this->helper;
	}
}

/*
 *	On enter, the helper will either create or just enter the
 *	shadow key.
 */

Bool
ShadowRegKey::HandleEnter ()
{
	Bool release = FALSE;
	
	if (this->helper) {
		if (RegKey::HandleEnter ()) {
			
			if (this->helper->Enter (this->shadow)) {
				return TRUE;
			}
			
			RegKey::HandleRelease ();
		}
	}
	
	return FALSE;
}

/*
 *	On release, the helper might want to dispose of the shadow
 *	key or simply release it.
 */

Bool
ShadowRegKey::HandleRelease ()
{
	if (this->helper) {
		if (this->helper->Release (this->shadow)) {
			return RegKey::HandleRelease ();
		}
	}
	
	return FALSE;
}


/*
 *	Following methods just call the shadowed key's...
 */

int
ShadowRegKey::Insert (RegValue& value, RegHive* hive)
{
	return (this->shadow)
		 ? this->shadow->Insert (value, hive)
		 : REG_ERR_END;
}

Bool
ShadowRegKey::Remove (RegValue& value)
{
	return (this->shadow)
		 ? this->shadow->Remove (value)
		 : FALSE;
}

Bool
ShadowRegKey::Find (RegValue& value, RegHive** hive_ptr, void*& dispose)
{
	return (this->shadow)
		 ? this->shadow->Find (value, hive_ptr, dispose)
		 : FALSE;
}

Bool
ShadowRegKey::Find (Card32& iter, RegValue& value, RegHive** hive_ptr, void*& dispose)
{
	return (this->shadow)
		 ? this->shadow->Find (iter, value, hive_ptr, dispose)
		 : FALSE;
}

void
ShadowRegKey::Dispose (void* dispose)
{
	if (this->shadow) {
		this->shadow->Dispose (dispose);
	}
}

int
ShadowRegKey::SaveHiveTree (RegHive* hive, HiveStream*& stream)
{
	return (this->shadow)
		 ? this->shadow->SaveHiveTree (hive, stream)
		 : REG_ERR_OK;
}

int
ShadowRegKey::KillHiveTree (RegHive* hive)
{
	return (this->shadow)
		 ? this->shadow->KillHiveTree (hive)
		 : REG_ERR_OK;
}

