/*
 *	regkey.h
 *
 *	Smaky Registry / INTERNAL.
 *
 *	The registry key class provides methods to handle nodes
 *	of the tree.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#ifndef _REGKEY_H_
#define	_REGKEY_H_

#include "smreg.h"

class RegHive;
class Registry;
class HiveStream;

/*****************************************************************************/

/*
 *	The RegKey class is just used as a base class to implement any kind
 *	of data holders. The default registry offers the DataRegKey to store
 *	information as data records.
 */

class RegKey
{
	
protected:
	Registry*		registry;			//	pointer to registry
	
	Card32			enter;				//	number of "enters"
	RegKey*			parent;				//	parent key
	RegHive*		hive;				//	parent hive
	Card32			child;				//	children count
	Card32			pending;			//	pending release count
	Card32			id;					//	id of entered key
	
protected:
	RegKey (Registry* reg, RegKey* father);
	
	void Atomify (RegValue& value);
	
	virtual Bool HandleEnter ();		//	override if needed
	virtual Bool HandleRelease ();		//	override if needed
	
public:
	virtual ~ RegKey ();
	
	//	Enter and release are provided for the caller only. If you have to
	//	override these behaviours, please override the HandleEnter and the
	//	HandleRelease methods.
	
	Bool Enter ();
	Bool Release ();
	
	//	The following methods are just useful to access private fields.
	//	They need not be overridden.
	
	Card32 GetEnterCount () const	{ return this->enter; }
	Card32 GetID () const			{ return this->id; }
	void SetID (Card32 id)			{ this->id = id; }

	void SetHive (RegHive* hive)	{ this->hive = hive; }
	RegHive* GetHive () const		{ return this->hive; }
	
	//	Insert data into the key. The value specifies everything about the
	//	data (name, type & data). The data should be stored in the specified
	//	hive.
	
	virtual int  Insert (RegValue& value, RegHive* hive) = 0;
	
	//	Remove data from the key. The value specifies just the name of the
	//	data to be removed.
	
	virtual Bool Remove (RegValue& value) = 0;
	
	//	Find data in a key and possibly return the hive in which it is stored.
	//	This might return a special value (dispose) which needs to be disposed
	//	of later on; this feature is provided in order to support access to
	//	dynamic data. The data to look for is specified by its name.
	
	virtual Bool Find (RegValue& io_value, RegHive** p_hive, void*& dispose) = 0;
	
	//	Find data in a key, given an iterator.
	
	virtual Bool Find (Card32& iter, RegValue& out_value, RegHive** p_hive, void*& dispose) = 0;
	
	//	Dispose of any temporary storage which might have been allocated by one
	//	of the Find methods.
	
	virtual void Dispose (void* dispose) = 0;
	
	//	SaveHiveTree and KillHiveTree should walk through each element stored
	//	in the key and -- for each matching the specified hive -- either save
	//	it or delete it.
	
	virtual int  SaveHiveTree (RegHive* hive, HiveStream*&) = 0;
	virtual int  KillHiveTree (RegHive* hive) = 0;
};

/*****************************************************************************/

/*
 *	Real implementation of a Registry Key and associated data. The
 *	data is accessed using hash keys.
 */

#define	HASH_L1		7
#define	H_NUM		37
#define	H_HEAP		16

class DataRegKey : public RegKey
{
	struct Value
	{
		RegValue*	value;			//	pointer to the value stored in the hive
		RegHive*	hive;			//	hive which contains this value
	};
	
	struct Hash
	{
		Value		data[HASH_L1];	//	associated data (value/hive)
		Hash*		next;			//	next values with same hash
	};

	struct HashHeap
	{
		Hash		heap[H_HEAP];	//	heap
		HashHeap*	next;			//	next hash heap
		int			index;			//	index into heap (first free)
	};
	
protected:
	Card32			h_num;			//	number of elements in hash table
	Hash			h_table[H_NUM];	//	pointer to hash table
	HashHeap*		h_heap;			//	list of free hash sub-tables
	Card32			count;			//	number of valid values
	
protected:
	Card32 HashName (RegValue& value);
	Hash* NewHash ();
	void KillEntry (Hash* h, int index);
	
public:
	DataRegKey (Registry* reg, RegKey* father);
	virtual ~DataRegKey ();
	
	virtual int  Insert (RegValue& value, RegHive* hive);
	virtual Bool Remove (RegValue& value);
	
	virtual Bool Find (RegValue& value, RegHive** hive_ptr, void*& dispose);
	virtual Bool Find (Card32& iter, RegValue& value, RegHive** hive_ptr, void*& dispose);
	virtual void Dispose (void* dispose);
	
	virtual int  SaveHiveTree (RegHive* hive, HiveStream*&);
	virtual int  KillHiveTree (RegHive* hive);
};

/*****************************************************************************/

/*
 *	The Shadow Helper class provides a mechanism which can be used
 *	to obtain the shadow RegKey (e.g. by loading it on demand) for
 *	the ShadowRegKey class.
 */

class ShadowHelper
{
public:
	ShadowHelper ()						{ }
	virtual ~ ShadowHelper ()			{ }
	
	virtual Bool Enter (RegKey*& shadow) = 0;
	virtual Bool Release (RegKey*& shadow) = 0;
};

/*****************************************************************************/

/*
 *	The ShadowRegKey class just forwards the calls to another class.
 */

class ShadowRegKey : public RegKey
{
	
protected:
	RegKey*			shadow;				//	shadowed registry key
	ShadowHelper*	helper;				//	helper class used to build shadow
	void*			helper_info;		//	info for the helper class (if needed)

protected:
	virtual Bool HandleEnter ();
	virtual Bool HandleRelease ();
	
public:
	ShadowRegKey (Registry* reg, RegKey* father);
	virtual ~ ShadowRegKey ();
	
	void SetShadow (RegKey* key)		{ this->shadow		= key;	}
	void SetHelper (ShadowHelper* help)	{ this->helper		= help;	}
	void SetHelperInfo (void* info)		{ this->helper_info = info;	}
	
	RegKey* GetShadow () const			{ return this->shadow;		}
	ShadowHelper* GetHelper () const	{ return this->helper;		}
	void* GetHelperInfo () const		{ return this->helper_info;	}
	
	virtual int  Insert (RegValue& value, RegHive* hive);
	virtual Bool Remove (RegValue& value);
	
	virtual Bool Find (RegValue& value, RegHive** hive_ptr, void*& dispose);
	virtual Bool Find (Card32& iter, RegValue& value, RegHive** hive_ptr, void*& dispose);
	virtual void Dispose (void* dispose);
	
	virtual int  SaveHiveTree (RegHive* hive, HiveStream*&);
	virtual int  KillHiveTree (RegHive* hive);
};

/*****************************************************************************/

#endif /* _REGKEY_H_ */

